/*
 * Decompiled with CFR 0.152.
 */
package picard.arrays;

import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.variant.variantcontext.Allele;
import htsjdk.variant.variantcontext.Genotype;
import htsjdk.variant.variantcontext.VariantContext;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.arrays.illumina.IlluminaAdpcFileWriter;
import picard.arrays.illumina.IlluminaGenotype;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.GenotypingArraysProgramGroup;

@CommandLineProgramProperties(summary="VcfToAdpc takes a VCF, as generated by GtcToVcf and generates an Illumina 'adpc.bin' file from it. An adpc.bin file is a binary file containing genotyping array intensity data that can be exported by Illumina's GenomeStudio and Beadstudio analysis tools. The adpc.bin file is used as an input to <a href='https://genome.sph.umich.edu/wiki/VerifyIDintensity'>VerifyIDintensity</a> a tool for detecting and estimating sample contamination of Illumina genotyping array data. If more than one VCF is used, they must all have the same number of loci.<h4>Usage example:</h4><pre>java -jar picard.jar VcfToAdpc \\<br />      VCF=input.vcf \\<br />      OUTPUT=output.adpc.bin \\<br />      SAMPLES_FILE=output.samples.txt \\<br />      NUM_MARKERS_FILE=output.num_markers.txt \\<br /></pre>", oneLineSummary="Program to convert an Arrays VCF to an ADPC file.", programGroup=GenotypingArraysProgramGroup.class)
@DocumentedFeature
public class VcfToAdpc
extends CommandLineProgram {
    static final String USAGE_DETAILS = "VcfToAdpc takes a VCF, as generated by GtcToVcf and generates an Illumina 'adpc.bin' file from it. An adpc.bin file is a binary file containing genotyping array intensity data that can be exported by Illumina's GenomeStudio and Beadstudio analysis tools. The adpc.bin file is used as an input to <a href='https://genome.sph.umich.edu/wiki/VerifyIDintensity'>VerifyIDintensity</a> a tool for detecting and estimating sample contamination of Illumina genotyping array data. If more than one VCF is used, they must all have the same number of loci.<h4>Usage example:</h4><pre>java -jar picard.jar VcfToAdpc \\<br />      VCF=input.vcf \\<br />      OUTPUT=output.adpc.bin \\<br />      SAMPLES_FILE=output.samples.txt \\<br />      NUM_MARKERS_FILE=output.num_markers.txt \\<br /></pre>";
    private final Log log = Log.getInstance(VcfToAdpc.class);
    @Argument(doc="One or more VCF files containing array intensity data.")
    public List<File> VCF;
    @Argument(shortName="O", doc="The output (adpc.bin) file to write.")
    public File OUTPUT;
    @Argument(shortName="SF", doc="A text file into which the names of the samples will be written. These will be in the same order as the data in the adpc.bin file.")
    public File SAMPLES_FILE;
    @Argument(shortName="NMF", doc="A text file into which the number of loci in the VCF will be written. This is useful for calling verifyIDIntensity.")
    public File NUM_MARKERS_FILE;

    @Override
    protected int doWork() {
        List inputs = IOUtil.unrollFiles(this.VCF, (String[])IOUtil.VCF_EXTENSIONS);
        IOUtil.assertFilesAreReadable((List)inputs);
        IOUtil.assertFileIsWritable((File)this.SAMPLES_FILE);
        IOUtil.assertFileIsWritable((File)this.NUM_MARKERS_FILE);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        ArrayList<String> sampleNames = new ArrayList<String>();
        Integer numberOfLoci = null;
        try (IlluminaAdpcFileWriter adpcFileWriter = new IlluminaAdpcFileWriter(this.OUTPUT);){
            for (File inputVcf : inputs) {
                VCFFileReader vcfFileReader = new VCFFileReader(inputVcf, false);
                VCFHeader header = vcfFileReader.getFileHeader();
                for (int sampleNumber = 0; sampleNumber < header.getNGenotypeSamples(); ++sampleNumber) {
                    String sampleName = (String)header.getGenotypeSamples().get(sampleNumber);
                    sampleNames.add(sampleName);
                    this.log.info(new Object[]{"Processing sample: " + sampleName + " from VCF: " + inputVcf.getAbsolutePath()});
                    CloseableIterator variants = vcfFileReader.iterator();
                    int lociCount = 0;
                    while (variants.hasNext()) {
                        VariantContext context = (VariantContext)variants.next();
                        float gcScore = this.getFloatAttribute(context, "GC_SCORE");
                        Genotype genotype = context.getGenotype(sampleNumber);
                        IlluminaGenotype illuminaGenotype = this.getIlluminaGenotype(genotype, context);
                        int rawXIntensity = this.getUnsignedShortAttributeAsInt(genotype, "X");
                        int rawYIntensity = this.getUnsignedShortAttributeAsInt(genotype, "Y");
                        Float normalizedXIntensity = this.getFloatAttribute(genotype, "NORMX");
                        Float normalizedYIntensity = this.getFloatAttribute(genotype, "NORMY");
                        IlluminaAdpcFileWriter.Record record = new IlluminaAdpcFileWriter.Record(rawXIntensity, rawYIntensity, normalizedXIntensity, normalizedYIntensity, gcScore, illuminaGenotype);
                        adpcFileWriter.write(record);
                        ++lociCount;
                    }
                    if (lociCount == 0) {
                        throw new PicardException("Found no records in VCF' " + inputVcf.getAbsolutePath() + "'");
                    }
                    if (numberOfLoci == null) {
                        numberOfLoci = lociCount;
                        continue;
                    }
                    if (lociCount == numberOfLoci) continue;
                    throw new PicardException("VCFs have differing number of loci");
                }
            }
            this.writeTextToFile(this.SAMPLES_FILE, StringUtils.join(sampleNames, (String)"\n"));
            this.writeTextToFile(this.NUM_MARKERS_FILE, "" + numberOfLoci);
        }
        catch (Exception e) {
            this.log.error((Throwable)e, new Object[0]);
            return 1;
        }
        return 0;
    }

    private void writeTextToFile(File output, String text) throws IOException {
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(output), StandardCharsets.UTF_8));){
            writer.write(text);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private IlluminaGenotype getIlluminaGenotype(Genotype genotype, VariantContext context) {
        if (!genotype.isCalled()) return IlluminaGenotype.NN;
        String illuminaAlleleA = StringUtils.stripEnd((String)this.getStringAttribute(context, "ALLELE_A"), (String)"*");
        String illuminaAlleleB = StringUtils.stripEnd((String)this.getStringAttribute(context, "ALLELE_B"), (String)"*");
        if (genotype.getAlleles().size() != 2) {
            throw new PicardException("Unexpected number of called alleles in variant context " + context + " found alleles: " + genotype.getAlleles());
        }
        Allele calledAllele1 = genotype.getAllele(0);
        Allele calledAllele2 = genotype.getAllele(1);
        if (calledAllele1.basesMatch(illuminaAlleleA)) {
            if (calledAllele2.basesMatch(illuminaAlleleA)) {
                return IlluminaGenotype.AA;
            }
            if (!calledAllele2.basesMatch(illuminaAlleleB)) throw new PicardException("Error matching called alleles to Illumina alleles.  Context: " + context);
            return IlluminaGenotype.AB;
        }
        if (!calledAllele1.basesMatch(illuminaAlleleB)) throw new PicardException("Error matching called alleles to Illumina alleles.  Context: " + context);
        if (calledAllele2.basesMatch(illuminaAlleleA)) {
            return IlluminaGenotype.AB;
        }
        if (!calledAllele2.basesMatch(illuminaAlleleB)) throw new PicardException("Error matching called alleles to Illumina alleles.  Context: " + context);
        return IlluminaGenotype.BB;
    }

    private int getUnsignedShortAttributeAsInt(Genotype genotype, String key) {
        int attributeAsInt = Integer.parseInt(this.getRequiredAttribute(genotype, key).toString());
        if (attributeAsInt < 0) {
            throw new PicardException("Value for key " + key + " (" + attributeAsInt + ") is <= 0!  Invalid value for unsigned int");
        }
        if (attributeAsInt > 65535) {
            this.log.warn(new Object[]{"Value for key " + key + " (" + attributeAsInt + ") is > " + 65535 + " (truncating it)"});
            return 65535;
        }
        return attributeAsInt;
    }

    private Float getFloatAttribute(Genotype genotype, String key) {
        Object value = genotype.getAnyAttribute(key);
        if (value != null && !value.equals("?")) {
            return Float.valueOf(Float.parseFloat(value.toString()));
        }
        return null;
    }

    private Object getRequiredAttribute(Genotype genotype, String key) {
        Object value = genotype.getAnyAttribute(key);
        if (value == null) {
            throw new PicardException("Unable to find attribute " + key + " in VCF Genotype field.  Is this an Arrays VCF file?");
        }
        return value;
    }

    private float getFloatAttribute(VariantContext context, String key) {
        return Float.parseFloat(this.getRequiredAttribute(context, key).toString());
    }

    private String getStringAttribute(VariantContext context, String key) {
        return this.getRequiredAttribute(context, key).toString();
    }

    private Object getRequiredAttribute(VariantContext context, String key) {
        Object value = context.getAttribute(key);
        if (value == null) {
            throw new PicardException("Unable to find attribute " + key + " in VCF.  Is this an Arrays VCF file?");
        }
        return value;
    }
}

