/*
 * Decompiled with CFR 0.152.
 */
package picard.fastq;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMRecordIterator;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.filter.AggregateFilter;
import htsjdk.samtools.filter.FailsVendorReadQualityFilter;
import htsjdk.samtools.filter.FilteringSamIterator;
import htsjdk.samtools.filter.SamRecordFilter;
import htsjdk.samtools.filter.TagFilter;
import htsjdk.samtools.filter.WholeReadClippedFilter;
import htsjdk.samtools.util.BinaryCodec;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.PeekableIterator;
import htsjdk.utils.ValidationUtils;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import picard.PicardException;

public class SamToBfqWriter {
    private static final int SEED_REGION_LENGTH = 28;
    private static final int MAX_SEED_REGION_NOCALL_FIXES = 2;
    private final File samFile;
    private final String outputPrefix;
    private final String namePrefix;
    private final int nameTrim;
    private final File referenceSequence;
    private final boolean pairedReads;
    private int wrote = 0;
    private int increment = 1;
    private int chunk = 0;
    private BinaryCodec codec1;
    private BinaryCodec codec2;
    private final Log log = Log.getInstance(SamToBfqWriter.class);
    private final boolean includeNonPfReads;
    private final boolean clipAdapters;
    private final Integer basesToWrite;

    public SamToBfqWriter(File samFile, File referenceSequence, String outputPrefix, Integer total, Integer chunk, boolean pairedReads, String namePrefix, boolean includeNonPfReads, boolean clipAdapters, Integer basesToWrite) {
        IOUtil.assertFileIsReadable((File)samFile);
        this.samFile = samFile;
        this.outputPrefix = outputPrefix;
        this.pairedReads = pairedReads;
        this.referenceSequence = referenceSequence;
        if (total != null) {
            double writeable = this.countWritableRecords();
            this.increment = (int)Math.floor(writeable / total.doubleValue());
            if (this.increment == 0) {
                this.increment = 1;
            }
        }
        if (chunk != null) {
            this.chunk = chunk;
        }
        this.namePrefix = namePrefix;
        this.nameTrim = namePrefix != null ? namePrefix.length() : 0;
        this.includeNonPfReads = includeNonPfReads;
        this.clipAdapters = clipAdapters;
        this.basesToWrite = basesToWrite;
    }

    public SamToBfqWriter(File samFile, String outputPrefix, boolean pairedReads, String namePrefix, boolean includeNonPfReads) {
        this(samFile, null, outputPrefix, null, null, pairedReads, namePrefix, includeNonPfReads, true, null);
    }

    public void writeBfqFiles() {
        SamReader reader = SamReaderFactory.makeDefault().open(this.samFile);
        SAMRecordIterator iterator = reader.iterator();
        TagFilter tagFilter = new TagFilter("XN", (Object)1);
        FailsVendorReadQualityFilter qualityFilter = new FailsVendorReadQualityFilter();
        WholeReadClippedFilter clippedFilter = new WholeReadClippedFilter();
        if (!this.pairedReads) {
            ArrayList<SamRecordFilter> filters = new ArrayList<SamRecordFilter>();
            filters.add((SamRecordFilter)tagFilter);
            filters.add((SamRecordFilter)clippedFilter);
            if (!this.includeNonPfReads) {
                filters.add((SamRecordFilter)qualityFilter);
            }
            this.writeSingleEndBfqs((Iterator<SAMRecord>)iterator, filters);
            this.codec1.close();
        } else {
            this.writePairedEndBfqs((Iterator<SAMRecord>)iterator, tagFilter, qualityFilter, new SamRecordFilter[]{clippedFilter});
            this.codec1.close();
            this.codec2.close();
        }
        this.log.info(new Object[]{"Wrote " + this.wrote + " bfq records."});
        CloserUtil.close((Object)reader);
    }

    private void writePairedEndBfqs(Iterator<SAMRecord> iterator, TagFilter tagFilter, FailsVendorReadQualityFilter qualityFilter, SamRecordFilter ... otherFilters) {
        int fileIndex = 0;
        this.initializeNextBfqFiles(fileIndex++);
        int records = 0;
        block0: while (iterator.hasNext()) {
            SAMRecord first = iterator.next();
            if (!iterator.hasNext()) {
                throw new PicardException("Mismatched number of records in " + this.samFile.getAbsolutePath());
            }
            SAMRecord second = iterator.next();
            if (!second.getReadName().equals(first.getReadName()) || first.getFirstOfPairFlag() == second.getFirstOfPairFlag()) {
                throw new PicardException("Unmatched read pairs in " + this.samFile.getAbsolutePath() + ": " + first.getReadName() + ", " + second.getReadName() + ".");
            }
            if (tagFilter.filterOut(first) && tagFilter.filterOut(second) || !this.includeNonPfReads && (qualityFilter.filterOut(first) || qualityFilter.filterOut(second))) continue;
            for (SamRecordFilter filter : otherFilters) {
                if (filter.filterOut(first) || filter.filterOut(second)) continue block0;
            }
            if (++records % this.increment != 0) continue;
            first.setReadName(first.getReadName() + "/1");
            this.writeFastqRecord(first.getFirstOfPairFlag() ? this.codec1 : this.codec2, first);
            second.setReadName(second.getReadName() + "/2");
            this.writeFastqRecord(second.getFirstOfPairFlag() ? this.codec1 : this.codec2, second);
            ++this.wrote;
            if (this.wrote % 1000000 == 0) {
                this.log.info(new Object[]{this.wrote + " records written."});
            }
            if (this.chunk <= 0 || this.wrote % this.chunk != 0) continue;
            this.initializeNextBfqFiles(fileIndex++);
        }
    }

    private void writeSingleEndBfqs(Iterator<SAMRecord> iterator, List<SamRecordFilter> filters) {
        int fileIndex = 0;
        this.initializeNextBfqFiles(fileIndex++);
        int records = 0;
        FilteringSamIterator it = new FilteringSamIterator(iterator, (SamRecordFilter)new AggregateFilter(filters));
        while (it.hasNext()) {
            SAMRecord record = it.next();
            if (++records % this.increment != 0) continue;
            record.setReadName(record.getReadName() + "/1");
            this.writeFastqRecord(this.codec1, record);
            ++this.wrote;
            if (this.wrote % 1000000 == 0) {
                this.log.info(new Object[]{this.wrote + " records processed."});
            }
            if (this.chunk <= 0 || this.wrote % this.chunk != 0) continue;
            this.initializeNextBfqFiles(fileIndex++);
        }
    }

    private void initializeNextBfqFiles(int fileIndex) {
        if (this.codec1 != null) {
            this.codec1.close();
            if (this.pairedReads) {
                this.codec2.close();
            }
        }
        File bfq1 = this.getOutputFile(this.outputPrefix, 1, fileIndex);
        this.codec1 = new BinaryCodec(IOUtil.openFileForWriting((File)bfq1));
        this.log.info(new Object[]{"Now writing to file " + bfq1.getAbsolutePath()});
        if (this.pairedReads) {
            File bfq2 = this.getOutputFile(this.outputPrefix, 2, fileIndex);
            this.codec2 = new BinaryCodec(IOUtil.openFileForWriting((File)bfq2));
            this.log.info(new Object[]{"Now writing to file " + bfq2.getAbsolutePath()});
        }
    }

    private void writeFastqRecord(BinaryCodec codec, SAMRecord rec) {
        Integer trimPoint;
        String readName = rec.getReadName();
        if (this.namePrefix != null && readName.startsWith(this.namePrefix)) {
            readName = readName.substring(this.nameTrim);
        }
        codec.writeString(readName, true, true);
        char[] seqs = rec.getReadString().toCharArray();
        char[] quals = rec.getBaseQualityString().toCharArray();
        int retainedLength = seqs.length;
        if (this.clipAdapters && (trimPoint = rec.getIntegerAttribute("XT")) != null) {
            ValidationUtils.validateArg((rec.getReadLength() == seqs.length ? 1 : 0) != 0, () -> "length of read and seqs differ. Found " + rec.getReadLength() + " and '" + seqs.length + ".");
            retainedLength = Math.min(seqs.length, Math.max(28, trimPoint - 1));
        }
        codec.writeInt(this.basesToWrite != null ? this.basesToWrite : seqs.length);
        byte[] seqsAndQuals = this.encodeSeqsAndQuals(seqs, quals, retainedLength);
        codec.writeBytes(seqsAndQuals);
    }

    private byte[] encodeSeqsAndQuals(char[] seqs, char[] quals, int retainedLength) {
        int i;
        byte[] seqsAndQuals = new byte[this.basesToWrite == null ? seqs.length : this.basesToWrite];
        int seedRegionNoCallFixes = 0;
        for (i = 0; i < retainedLength && i < seqsAndQuals.length; ++i) {
            int base;
            int quality = Math.min(quals[i] - 33, 63);
            switch (seqs[i]) {
                case 'A': 
                case 'a': {
                    base = 0;
                    break;
                }
                case 'C': 
                case 'c': {
                    base = 1;
                    break;
                }
                case 'G': 
                case 'g': {
                    base = 2;
                    break;
                }
                case 'T': 
                case 't': {
                    base = 3;
                    break;
                }
                case '.': 
                case 'N': 
                case 'n': {
                    base = 0;
                    if (i < 28) {
                        if (seedRegionNoCallFixes < 2) {
                            quality = 1;
                            ++seedRegionNoCallFixes;
                            break;
                        }
                        quality = 0;
                        break;
                    }
                    quality = 1;
                    break;
                }
                default: {
                    throw new PicardException("Unknown base when writing bfq file: " + seqs[i]);
                }
            }
            seqsAndQuals[i] = this.encodeBaseAndQuality(base, quality);
        }
        for (i = retainedLength; i < seqsAndQuals.length; ++i) {
            seqsAndQuals[i] = this.encodeBaseAndQuality(0, 1);
        }
        return seqsAndQuals;
    }

    private byte encodeBaseAndQuality(int base, int quality) {
        return (byte)(base << 6 | quality);
    }

    private int countWritableRecords() {
        int count = 0;
        SamReader reader = SamReaderFactory.makeDefault().referenceSequence(this.referenceSequence).open(this.samFile);
        if (!reader.getFileHeader().getSortOrder().equals((Object)SAMFileHeader.SortOrder.queryname)) {
            throw new PicardException("Input file (" + this.samFile.getAbsolutePath() + ") needs to be sorted by queryname.");
        }
        PeekableIterator it = new PeekableIterator((Iterator)reader.iterator());
        if (!this.pairedReads) {
            ArrayList<Object> filters = new ArrayList<Object>();
            filters.add(new TagFilter("XN", (Object)1));
            if (!this.includeNonPfReads) {
                filters.add(new FailsVendorReadQualityFilter());
            }
            FilteringSamIterator itr = new FilteringSamIterator((Iterator)it, (SamRecordFilter)new AggregateFilter(filters));
            while (itr.hasNext()) {
                itr.next();
                ++count;
            }
        } else {
            while (it.hasNext()) {
                SAMRecord first = (SAMRecord)it.next();
                SAMRecord second = (SAMRecord)it.next();
                if (first.getAttribute("XN") != null && second.getAttribute("XN") != null || !this.includeNonPfReads && (first.getReadFailsVendorQualityCheckFlag() || second.getReadFailsVendorQualityCheckFlag())) continue;
                ++count;
            }
        }
        it.close();
        CloserUtil.close((Object)reader);
        return count;
    }

    private File getOutputFile(String outputPrefix, int read, int index) {
        File result = new File(outputPrefix + index + "." + read + ".bfq");
        IOUtil.assertFileIsWritable((File)result);
        return result;
    }
}

