/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.SAMValidationError;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.fastq.FastqWriter;
import htsjdk.samtools.fastq.FastqWriterFactory;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Lazy;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import htsjdk.samtools.util.TrimmingUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.ReadDataManipulationProgramGroup;

@CommandLineProgramProperties(summary="Converts a SAM/BAM/CRAM file to FASTQ. Extracts read sequences and qualities from the input SAM/BAM/CRAM file and writes them intothe output file in Sanger FASTQ format.See <a href=\"http://maq.sourceforge.net/fastq.shtml\">MAQ FASTQ specification</a> for details.This tool can be used by way of a pipe to run BWA MEM on unmapped BAM (uBAM) files efficiently.</p><p>In the RC mode (default is True), if the read is aligned and the alignment is to the reverse strand on the genome,the read's sequence from input sam file will be reverse-complemented prior to writing it to FASTQ in order restore correctlythe original read sequence as it was generated by the sequencer.</p><br /><h4>Usage example:</h4><pre>java -jar picard.jar SamToFastq <br />     I=input.bam<br />     FASTQ=output.fastq</pre><hr />", oneLineSummary="Converts a SAM/BAM/CRAM file to FASTQ.", programGroup=ReadDataManipulationProgramGroup.class)
@DocumentedFeature
public class SamToFastq
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Converts a SAM/BAM/CRAM file to FASTQ.";
    static final String USAGE_DETAILS = " Extracts read sequences and qualities from the input SAM/BAM/CRAM file and writes them intothe output file in Sanger FASTQ format.See <a href=\"http://maq.sourceforge.net/fastq.shtml\">MAQ FASTQ specification</a> for details.This tool can be used by way of a pipe to run BWA MEM on unmapped BAM (uBAM) files efficiently.</p><p>In the RC mode (default is True), if the read is aligned and the alignment is to the reverse strand on the genome,the read's sequence from input sam file will be reverse-complemented prior to writing it to FASTQ in order restore correctlythe original read sequence as it was generated by the sequencer.</p><br /><h4>Usage example:</h4><pre>java -jar picard.jar SamToFastq <br />     I=input.bam<br />     FASTQ=output.fastq</pre><hr />";
    @Argument(doc="Input SAM/BAM/CRAM file to extract reads from", shortName="I")
    public File INPUT;
    @Argument(shortName="F", doc="Output FASTQ file (single-end fastq or, if paired, first end of the pair FASTQ).", mutex={"OUTPUT_PER_RG", "COMPRESS_OUTPUTS_PER_RG", "OUTPUT_DIR"})
    public File FASTQ;
    @Argument(shortName="F2", doc="Output FASTQ file (if paired, second end of the pair FASTQ).", optional=true, mutex={"OUTPUT_PER_RG", "COMPRESS_OUTPUTS_PER_RG"})
    public File SECOND_END_FASTQ;
    @Argument(shortName="FU", doc="Output FASTQ file for unpaired reads; may only be provided in paired-FASTQ mode", optional=true, mutex={"OUTPUT_PER_RG", "COMPRESS_OUTPUTS_PER_RG"})
    public File UNPAIRED_FASTQ;
    @Argument(shortName="OPRG", doc="Output a FASTQ file per read group (two FASTQ files per read group if the group is paired).", optional=true, mutex={"FASTQ", "SECOND_END_FASTQ", "UNPAIRED_FASTQ"})
    public boolean OUTPUT_PER_RG;
    @Argument(shortName="GZOPRG", doc="Compress output FASTQ files per read group using gzip and append a .gz extension to the file names.", mutex={"FASTQ", "SECOND_END_FASTQ", "UNPAIRED_FASTQ"})
    public Boolean COMPRESS_OUTPUTS_PER_RG = false;
    @Argument(shortName="RGT", doc="The read group tag (PU or ID) to be used to output a FASTQ file per read group.")
    public String RG_TAG = "PU";
    @Argument(shortName="ODIR", doc="Directory in which to output the FASTQ file(s).  Used only when OUTPUT_PER_RG is true.", optional=true)
    public File OUTPUT_DIR;
    @Argument(shortName="RC", doc="Re-reverse bases and qualities of reads with negative strand flag set before writing them to FASTQ", optional=true)
    public boolean RE_REVERSE = true;
    @Argument(shortName="INTER", doc="Will generate an interleaved fastq if paired, each line will have /1 or /2 to describe which end it came from")
    public boolean INTERLEAVE = false;
    @Argument(shortName="NON_PF", doc="Include non-PF reads from the SAM file into the output FASTQ files. PF means 'passes filtering'. Reads whose 'not passing quality controls' flag is set are non-PF reads. See GATK Dictionary for more info.")
    public boolean INCLUDE_NON_PF_READS = false;
    @Argument(shortName="CLIP_ATTR", doc="The attribute that stores the position at which the SAM record should be clipped", optional=true)
    public String CLIPPING_ATTRIBUTE;
    @Argument(shortName="CLIP_ACT", doc="The action that should be taken with clipped reads: 'X' means the reads and qualities should be trimmed at the clipped position; 'N' means the bases should be changed to Ns in the clipped region; and any integer means that the base qualities should be set to that value in the clipped region.", optional=true)
    public String CLIPPING_ACTION;
    @Argument(shortName="CLIP_MIN", doc="When performing clipping with the CLIPPING_ATTRIBUTE and CLIPPING_ACTION parameters, ensure that the resulting reads after clipping are at least CLIPPING_MIN_LENGTH bases long. If the original read is shorter than CLIPPING_MIN_LENGTH then the original read length will be maintained.")
    public int CLIPPING_MIN_LENGTH = 0;
    @Argument(shortName="R1_TRIM", doc="The number of bases to trim from the beginning of read 1.")
    public int READ1_TRIM = 0;
    @Argument(shortName="R1_MAX_BASES", doc="The maximum number of bases to write from read 1 after trimming. If there are fewer than this many bases left after trimming, all will be written.  If this value is null then all bases left after trimming will be written.", optional=true)
    public Integer READ1_MAX_BASES_TO_WRITE;
    @Argument(shortName="R2_TRIM", doc="The number of bases to trim from the beginning of read 2.")
    public int READ2_TRIM = 0;
    @Argument(shortName="R2_MAX_BASES", doc="The maximum number of bases to write from read 2 after trimming. If there are fewer than this many bases left after trimming, all will be written.  If this value is null then all bases left after trimming will be written.", optional=true)
    public Integer READ2_MAX_BASES_TO_WRITE;
    @Argument(shortName="Q", doc="End-trim reads using the phred/bwa quality trimming algorithm and this quality.", optional=true)
    public Integer QUALITY;
    @Argument(doc="If true, include non-primary alignments in the output.  Support of non-primary alignments in SamToFastq is not comprehensive, so there may be exceptions if this is set to true and there are paired reads with non-primary alignments.")
    public boolean INCLUDE_NON_PRIMARY_ALIGNMENTS = false;
    private static final String CLIP_TRIM = "X";
    private static final String CLIP_TO_N = "N";
    private final Log log = Log.getInstance(SamToFastq.class);

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.INPUT);
        SamReader reader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        HashMap<String, SAMRecord> firstSeenMates = new HashMap<String, SAMRecord>();
        FastqWriterFactory factory = new FastqWriterFactory();
        factory.setCreateMd5(this.CREATE_MD5_FILE);
        this.initializeAdditionalWriters();
        Map<SAMReadGroupRecord, FastqWriters> writers = this.generateWriters(reader.getFileHeader().getReadGroups(), factory);
        Map<SAMReadGroupRecord, List<FastqWriter>> additionalWriters = this.generateAdditionalWriters(reader.getFileHeader().getReadGroups(), factory);
        if (writers.isEmpty()) {
            String msgBase = this.INPUT + " does not contain Read Groups";
            String msg = this.OUTPUT_PER_RG ? msgBase + ", consider not using the OUTPUT_PER_RG option" : msgBase;
            throw new PicardException(msg);
        }
        ProgressLogger progress = new ProgressLogger(this.log);
        for (SAMRecord currentRecord : reader) {
            this.handleRecord(currentRecord, writers, additionalWriters, firstSeenMates);
            progress.record(currentRecord);
        }
        CloserUtil.close((Object)reader);
        for (FastqWriters writerMapping : new HashSet<FastqWriters>(writers.values())) {
            writerMapping.closeAll();
        }
        HashSet additionalWriterSet = new HashSet();
        additionalWriters.values().forEach(additionalWriterSet::addAll);
        for (FastqWriter fastqWriter : additionalWriterSet) {
            fastqWriter.close();
        }
        if (!firstSeenMates.isEmpty()) {
            SAMUtils.processValidationError((SAMValidationError)new SAMValidationError(SAMValidationError.Type.MATE_NOT_FOUND, "Found " + firstSeenMates.size() + " unpaired mates", null), (ValidationStringency)this.VALIDATION_STRINGENCY);
        }
        return 0;
    }

    private Map<SAMReadGroupRecord, FastqWriters> generateWriters(List<SAMReadGroupRecord> samReadGroupRecords, FastqWriterFactory factory) {
        HashMap<SAMReadGroupRecord, FastqWriters> writerMap = new HashMap<SAMReadGroupRecord, FastqWriters>();
        if (!this.OUTPUT_PER_RG) {
            FastqWriter secondOfPairWriter;
            IOUtil.assertFileIsWritable((File)this.FASTQ);
            FastqWriter firstOfPairWriter = factory.newWriter(this.FASTQ);
            if (this.INTERLEAVE) {
                secondOfPairWriter = firstOfPairWriter;
            } else if (this.SECOND_END_FASTQ != null) {
                IOUtil.assertFileIsWritable((File)this.SECOND_END_FASTQ);
                secondOfPairWriter = factory.newWriter(this.SECOND_END_FASTQ);
            } else {
                secondOfPairWriter = null;
            }
            FastqWriter unpairedWriter = this.UNPAIRED_FASTQ == null ? firstOfPairWriter : factory.newWriter(this.UNPAIRED_FASTQ);
            FastqWriters fastqWriters = new FastqWriters(firstOfPairWriter, secondOfPairWriter, unpairedWriter);
            writerMap.put(null, fastqWriters);
            for (SAMReadGroupRecord rg : samReadGroupRecords) {
                writerMap.put(rg, fastqWriters);
            }
        } else {
            for (SAMReadGroupRecord rg : samReadGroupRecords) {
                FastqWriter firstOfPairWriter = factory.newWriter(this.makeReadGroupFile(rg, "_1"));
                Lazy lazySecondOfPairWriter = new Lazy(() -> this.INTERLEAVE ? firstOfPairWriter : factory.newWriter(this.makeReadGroupFile(rg, "_2")));
                writerMap.put(rg, new FastqWriters(firstOfPairWriter, lazySecondOfPairWriter, firstOfPairWriter));
            }
        }
        return writerMap;
    }

    protected void initializeAdditionalWriters() {
    }

    protected Map<SAMReadGroupRecord, List<FastqWriter>> generateAdditionalWriters(List<SAMReadGroupRecord> readGroups, FastqWriterFactory factory) {
        return Collections.emptyMap();
    }

    private void handleRecord(SAMRecord currentRecord, Map<SAMReadGroupRecord, FastqWriters> writers, Map<SAMReadGroupRecord, List<FastqWriter>> additionalWriters, Map<String, SAMRecord> firstSeenMates) {
        if (currentRecord.isSecondaryOrSupplementary() && !this.INCLUDE_NON_PRIMARY_ALIGNMENTS) {
            return;
        }
        if (currentRecord.getReadFailsVendorQualityCheckFlag() && !this.INCLUDE_NON_PF_READS) {
            return;
        }
        FastqWriters fq = writers.get(currentRecord.getReadGroup());
        SAMRecord read1 = null;
        SAMRecord read2 = null;
        if (currentRecord.getReadPairedFlag()) {
            String currentReadName = currentRecord.getReadName();
            SAMRecord firstRecord = firstSeenMates.remove(currentReadName);
            if (firstRecord == null) {
                firstSeenMates.put(currentReadName, currentRecord);
            } else {
                SamToFastq.assertPairedMates(firstRecord, currentRecord);
                read1 = currentRecord.getFirstOfPairFlag() ? currentRecord : firstRecord;
                read2 = currentRecord.getFirstOfPairFlag() ? firstRecord : currentRecord;
                this.writeRecord(read1, 1, fq.getFirstOfPair(), this.READ1_TRIM, this.READ1_MAX_BASES_TO_WRITE);
                FastqWriter secondOfPairWriter = fq.getSecondOfPair();
                if (secondOfPairWriter == null) {
                    throw new PicardException("Input contains paired reads but no SECOND_END_FASTQ specified.");
                }
                this.writeRecord(read2, 2, secondOfPairWriter, this.READ2_TRIM, this.READ2_MAX_BASES_TO_WRITE);
            }
        } else {
            this.writeRecord(currentRecord, null, fq.getUnpaired(), this.READ1_TRIM, this.READ1_MAX_BASES_TO_WRITE);
        }
        this.handleAdditionalRecords(currentRecord, additionalWriters, read1, read2);
    }

    protected void handleAdditionalRecords(SAMRecord currentRecord, Map<SAMReadGroupRecord, List<FastqWriter>> additionalWriters, SAMRecord read1, SAMRecord read2) {
    }

    private File makeReadGroupFile(SAMReadGroupRecord readGroup, String preExtSuffix) {
        String fileName = null;
        if (this.RG_TAG.equalsIgnoreCase("PU")) {
            fileName = readGroup.getPlatformUnit();
        } else if (this.RG_TAG.equalsIgnoreCase("ID")) {
            fileName = readGroup.getReadGroupId();
        }
        if (fileName == null) {
            throw new PicardException("The selected RG_TAG: " + this.RG_TAG + " is not present in the header.");
        }
        fileName = IOUtil.makeFileNameSafe((String)fileName);
        if (preExtSuffix != null) {
            fileName = fileName + preExtSuffix;
        }
        fileName = fileName + (this.COMPRESS_OUTPUTS_PER_RG != false ? ".fastq.gz" : ".fastq");
        File result = this.OUTPUT_DIR != null ? new File(this.OUTPUT_DIR, fileName) : new File(fileName);
        IOUtil.assertFileIsWritable((File)result);
        return result;
    }

    private void writeRecord(SAMRecord read, Integer mateNumber, FastqWriter writer, int basesToTrim, Integer maxBasesToWrite) {
        byte[] quals;
        int qualityTrimIndex;
        String seqHeader = mateNumber == null ? read.getReadName() : read.getReadName() + "/" + mateNumber;
        String readString = read.getReadString();
        String baseQualities = read.getBaseQualityString();
        if (this.CLIPPING_ATTRIBUTE != null) {
            Integer clipPoint = (Integer)read.getAttribute(this.CLIPPING_ATTRIBUTE);
            if (clipPoint != null && clipPoint < this.CLIPPING_MIN_LENGTH) {
                clipPoint = Math.min(readString.length(), this.CLIPPING_MIN_LENGTH);
            }
            if (clipPoint != null) {
                if (this.CLIPPING_ACTION.equalsIgnoreCase(CLIP_TRIM)) {
                    readString = SamToFastq.clip(readString, clipPoint, null, !read.getReadNegativeStrandFlag());
                    baseQualities = SamToFastq.clip(baseQualities, clipPoint, null, !read.getReadNegativeStrandFlag());
                } else if (this.CLIPPING_ACTION.equalsIgnoreCase(CLIP_TO_N)) {
                    readString = SamToFastq.clip(readString, clipPoint, Character.valueOf(CLIP_TO_N.charAt(0)), !read.getReadNegativeStrandFlag());
                } else {
                    char newQual = SAMUtils.phredToFastq((byte[])new byte[]{(byte)Integer.parseInt(this.CLIPPING_ACTION)}).charAt(0);
                    baseQualities = SamToFastq.clip(baseQualities, clipPoint, Character.valueOf(newQual), !read.getReadNegativeStrandFlag());
                }
            }
        }
        if (this.RE_REVERSE && read.getReadNegativeStrandFlag()) {
            readString = SequenceUtil.reverseComplement((String)readString);
            baseQualities = StringUtil.reverseString((String)baseQualities);
        }
        if (basesToTrim > 0) {
            readString = readString.substring(basesToTrim);
            baseQualities = baseQualities.substring(basesToTrim);
        }
        if (this.QUALITY != null && (qualityTrimIndex = Math.max(1, TrimmingUtil.findQualityTrimPoint((byte[])(quals = SAMUtils.fastqToPhred((String)baseQualities)), (int)this.QUALITY))) < quals.length) {
            readString = readString.substring(0, qualityTrimIndex);
            baseQualities = baseQualities.substring(0, qualityTrimIndex);
        }
        if (maxBasesToWrite != null && maxBasesToWrite < readString.length()) {
            readString = readString.substring(0, maxBasesToWrite);
            baseQualities = baseQualities.substring(0, maxBasesToWrite);
        }
        writer.write(new FastqRecord(seqHeader, readString, "", baseQualities));
    }

    private static String clip(String src, int point, Character replacement, boolean posStrand) {
        StringBuilder result;
        block4: {
            int len = src.length();
            result = new StringBuilder(posStrand ? src.substring(0, point - 1) : src.substring(len - point + 1));
            if (replacement == null) break block4;
            if (posStrand) {
                for (int i = point; i <= len; ++i) {
                    result.append(replacement);
                }
            } else {
                for (int i = 0; i <= len - point; ++i) {
                    result.insert(0, replacement);
                }
            }
        }
        return result.toString();
    }

    protected static void assertPairedMates(SAMRecord record1, SAMRecord record2) {
        if (!(record1.getFirstOfPairFlag() && record2.getSecondOfPairFlag() || record2.getFirstOfPairFlag() && record1.getSecondOfPairFlag())) {
            throw new PicardException("Illegal mate state: " + record1.getReadName());
        }
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errors = new ArrayList<String>();
        if (this.INTERLEAVE && this.SECOND_END_FASTQ != null) {
            errors.add("Cannot set INTERLEAVE to true and pass in a SECOND_END_FASTQ");
        }
        if (this.UNPAIRED_FASTQ != null && this.SECOND_END_FASTQ == null) {
            errors.add("UNPAIRED_FASTQ may only be set when also emitting read1 and read2 fastqs (so SECOND_END_FASTQ must also be set).");
        }
        if (this.CLIPPING_ATTRIBUTE != null && this.CLIPPING_ACTION == null || this.CLIPPING_ATTRIBUTE == null && this.CLIPPING_ACTION != null) {
            errors.add("Both or neither of CLIPPING_ATTRIBUTE and CLIPPING_ACTION should be set.");
        }
        if (this.CLIPPING_ACTION != null && !this.CLIPPING_ACTION.equals(CLIP_TO_N) && !this.CLIPPING_ACTION.equals(CLIP_TRIM)) {
            try {
                Integer.parseInt(this.CLIPPING_ACTION);
            }
            catch (NumberFormatException nfe) {
                errors.add("CLIPPING ACTION must be one of: N, X, or an integer");
            }
        }
        if (this.OUTPUT_PER_RG && this.OUTPUT_DIR == null || !this.OUTPUT_PER_RG && this.OUTPUT_DIR != null) {
            errors.add("If OUTPUT_PER_RG is true, then OUTPUT_DIR should be set. If ");
        }
        if (this.OUTPUT_PER_RG) {
            if (this.RG_TAG == null) {
                errors.add("If OUTPUT_PER_RG is true, then RG_TAG should be set.");
            } else if (!this.RG_TAG.equalsIgnoreCase("PU") && !this.RG_TAG.equalsIgnoreCase("ID")) {
                errors.add("RG_TAG must be: PU or ID");
            }
        }
        return errors.isEmpty() ? super.customCommandLineValidation() : errors.toArray(new String[errors.size()]);
    }

    private static final class FastqWriters {
        private final FastqWriter firstOfPair;
        private final FastqWriter unpaired;
        private final Lazy<FastqWriter> secondOfPair;

        private FastqWriters(FastqWriter firstOfPair, Lazy<FastqWriter> secondOfPair, FastqWriter unpaired) {
            this.firstOfPair = firstOfPair;
            this.unpaired = unpaired;
            this.secondOfPair = secondOfPair;
        }

        private FastqWriters(FastqWriter firstOfPair, FastqWriter secondOfPair, FastqWriter unpaired) {
            this(firstOfPair, (Lazy<FastqWriter>)new Lazy(() -> secondOfPair), unpaired);
        }

        private FastqWriter getFirstOfPair() {
            return this.firstOfPair;
        }

        private FastqWriter getSecondOfPair() {
            return (FastqWriter)this.secondOfPair.get();
        }

        private FastqWriter getUnpaired() {
            return this.unpaired;
        }

        private void closeAll() {
            HashSet<Object> fastqWriters = new HashSet<Object>();
            fastqWriters.add(this.firstOfPair);
            fastqWriters.add(this.unpaired);
            if (this.secondOfPair.isInitialized()) {
                fastqWriters.add(this.secondOfPair.get());
            }
            for (FastqWriter fastqWriter : fastqWriters) {
                fastqWriter.close();
            }
        }
    }
}

