/*
 * Decompiled with CFR 0.152.
 */
package picard.fingerprint;

import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.util.FormatUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import java.io.File;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.concurrent.TimeUnit;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.Alpha;
import picard.fingerprint.Fingerprint;
import picard.fingerprint.FingerprintChecker;
import picard.fingerprint.HaplotypeMap;
import picard.fingerprint.MatchResults;

@CommandLineProgramProperties(usage="Checks if all read groups within a set of BAM files appear to come from the same individual", usageShort="Checks if all read groups appear to come from the same individual", programGroup=Alpha.class)
public class CrosscheckReadGroupFingerprints
extends CommandLineProgram {
    @Option(shortName="I", doc="One or more input BAM files (or lists of BAM files) to compare fingerprints for.")
    public List<File> INPUT;
    @Option(shortName="O", optional=true, doc="Optional output file to write metrics to. Default is to write to stdout.")
    public File OUTPUT;
    @Option(shortName="H", doc="The file of haplotype data to use to pick SNPs to fingerprint")
    public File HAPLOTYPE_MAP;
    @Option(shortName="LOD", doc="If any two read groups match with a LOD score lower than the threshold the program will exit with a non-zero code to indicate error. 0 means equal probability the read groups match vs. come from different individuals, negative numbers mean N logs more likely that the read groups are from different individuals and positive numbers mean N logs more likely that the read groups are from the sample individual.")
    public double LOD_THRESHOLD = 0.0;
    @Option(doc="Instead of producing the normal comparison of read-groups, roll fingerprints up to the sample level and print out a sample x sample matrix with LOD scores.")
    public boolean CROSSCHECK_SAMPLES = false;
    @Option(doc="Instead of producing the normal comparison of read-groups, roll fingerprints up to the library level and print out a library x library matrix with LOD scores.")
    public boolean CROSSCHECK_LIBRARIES = false;
    @Option(doc="The number of threads to use to process BAM files and generate Fingerprints.")
    public int NUM_THREADS = 1;
    @Option(doc="Allow the use of duplicate reads in performing the comparison. Can be useful when duplicate marking has been overly aggressive and coverage is low.")
    public boolean ALLOW_DUPLICATE_READS = false;
    @Option(doc="Assumed genotyping error rate that provides a floor on the probability that a genotype comes from the expected sample.")
    public double GENOTYPING_ERROR_RATE = 0.01;
    @Option(doc="If true then only read groups that do not relate to each other as expected will have their LODs reported.")
    public boolean OUTPUT_ERRORS_ONLY = false;
    @Option(doc="The rate at which a het in a normal sample turns into a hom in the tumor.", optional=true)
    public double LOSS_OF_HET_RATE = 0.5;
    @Option(doc="Expect all read groups' fingerprints to match, irrespective of their sample names.  By default (with this value set to false), read groups with different sample names are expected to mismatch, and those with the same sample name are expected to match.")
    public boolean EXPECT_ALL_READ_GROUPS_TO_MATCH = false;
    @Option(doc="When one or more mismatches between read groups are detected, exit with this value instead of 0.")
    public int EXIT_CODE_WHEN_MISMATCH = 1;
    private final Log log = Log.getInstance(CrosscheckReadGroupFingerprints.class);
    private final FormatUtil formatUtil = new FormatUtil();
    public static final String EXPECTED_MATCH = "EXPECTED MATCH";
    public static final String EXPECTED_MISMATCH = "EXPECTED MISMATCH";
    public static final String UNEXPECTED_MATCH = "UNEXPECTED MATCH";
    public static final String UNEXPECTED_MISMATCH = "UNEXPECTED MISMATCH";

    public static void main(String[] stringArray) {
        new CrosscheckReadGroupFingerprints().instanceMainWithExit(stringArray);
    }

    @Override
    protected int doWork() {
        for (File object2 : this.INPUT) {
            IOUtil.assertFileIsReadable((File)object2);
        }
        IOUtil.assertFileIsReadable((File)this.HAPLOTYPE_MAP);
        if (this.OUTPUT != null) {
            IOUtil.assertFileIsWritable((File)this.OUTPUT);
        }
        HaplotypeMap haplotypeMap = new HaplotypeMap(this.HAPLOTYPE_MAP);
        FingerprintChecker fingerprintChecker = new FingerprintChecker(haplotypeMap);
        fingerprintChecker.setAllowDuplicateReads(this.ALLOW_DUPLICATE_READS);
        this.log.info(new Object[]{"Done checking input files, moving onto fingerprinting files."});
        List list = IOUtil.unrollFiles(this.INPUT, (String[])new String[]{".bam", ".sam"});
        Map<SAMReadGroupRecord, Fingerprint> map = fingerprintChecker.fingerprintSamFiles(list, this.NUM_THREADS, 1, TimeUnit.DAYS);
        ArrayList<Fingerprint> arrayList = new ArrayList<Fingerprint>(map.values());
        this.log.info(new Object[]{"Finished generating fingerprints from BAM files, moving on to cross-checking."});
        PrintStream printStream = this.OUTPUT != null ? new PrintStream(IOUtil.openFileForWriting((File)this.OUTPUT), true) : System.out;
        if (this.CROSSCHECK_SAMPLES) {
            this.crossCheckSamples(arrayList, printStream);
            return 0;
        }
        if (this.CROSSCHECK_LIBRARIES) {
            this.crossCheckLibraries(map, printStream);
            return 0;
        }
        return this.crossCheckReadGroups(map, printStream);
    }

    private void crossCheckSamples(List<Fingerprint> list, PrintStream printStream) {
        SortedMap<String, Fingerprint> sortedMap = FingerprintChecker.mergeFingerprintsBySample(list);
        SortedSet sortedSet = (SortedSet)sortedMap.keySet();
        printStream.print("\t");
        for (String string : sortedSet) {
            printStream.print(string);
            printStream.print("\t");
        }
        printStream.println();
        for (String string : sortedSet) {
            printStream.print(string);
            Fingerprint fingerprint = (Fingerprint)sortedMap.get(string);
            for (String string2 : sortedSet) {
                MatchResults matchResults = FingerprintChecker.calculateMatchResults(fingerprint, (Fingerprint)sortedMap.get(string2), this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE);
                printStream.print("\t");
                printStream.print(this.formatUtil.format(matchResults.getLOD()));
            }
            printStream.println();
        }
    }

    private void crossCheckLibraries(Map<SAMReadGroupRecord, Fingerprint> map, PrintStream printStream) {
        ArrayList<Fingerprint> arrayList = new ArrayList<Fingerprint>();
        for (SAMReadGroupRecord sAMReadGroupRecord : map.keySet()) {
            Fingerprint fingerprint = map.get(sAMReadGroupRecord);
            String string = sAMReadGroupRecord.getSample() + "::" + sAMReadGroupRecord.getLibrary();
            Fingerprint fingerprint2 = new Fingerprint(string, fingerprint.getSource(), fingerprint.getInfo());
            fingerprint2.putAll(fingerprint);
            arrayList.add(fingerprint2);
        }
        this.crossCheckSamples(arrayList, printStream);
    }

    private int crossCheckReadGroups(Map<SAMReadGroupRecord, Fingerprint> map, PrintStream printStream) {
        int n = 0;
        int n2 = 0;
        ArrayList<SAMReadGroupRecord> arrayList = new ArrayList<SAMReadGroupRecord>(map.keySet());
        ArrayList<String> arrayList2 = new ArrayList<String>();
        for (int i = 0; i < arrayList.size(); ++i) {
            SAMReadGroupRecord sAMReadGroupRecord = (SAMReadGroupRecord)arrayList.get(i);
            for (int j = i + 1; j < arrayList.size(); ++j) {
                SAMReadGroupRecord sAMReadGroupRecord2 = (SAMReadGroupRecord)arrayList.get(j);
                boolean bl = this.EXPECT_ALL_READ_GROUPS_TO_MATCH || sAMReadGroupRecord.getSample().equals(sAMReadGroupRecord2.getSample());
                MatchResults matchResults = FingerprintChecker.calculateMatchResults(map.get(sAMReadGroupRecord), map.get(sAMReadGroupRecord2), this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE);
                if (bl) {
                    if (matchResults.getLOD() < this.LOD_THRESHOLD) {
                        ++n;
                        arrayList2.add(this.getMatchDetails(UNEXPECTED_MISMATCH, matchResults, sAMReadGroupRecord, sAMReadGroupRecord2));
                        continue;
                    }
                    if (this.OUTPUT_ERRORS_ONLY) continue;
                    arrayList2.add(this.getMatchDetails(EXPECTED_MATCH, matchResults, sAMReadGroupRecord, sAMReadGroupRecord2));
                    continue;
                }
                if (matchResults.getLOD() > -this.LOD_THRESHOLD) {
                    ++n2;
                    arrayList2.add(this.getMatchDetails(UNEXPECTED_MATCH, matchResults, sAMReadGroupRecord, sAMReadGroupRecord2));
                    continue;
                }
                if (this.OUTPUT_ERRORS_ONLY) continue;
                arrayList2.add(this.getMatchDetails(EXPECTED_MISMATCH, matchResults, sAMReadGroupRecord, sAMReadGroupRecord2));
            }
        }
        if (!arrayList2.isEmpty()) {
            printStream.println("RESULT\tLOD_SCORE\tLOD_SCORE_TUMOR_NORMAL\tLOD_SCORE_NORMAL_TUMOR\tLEFT_RUN_BARCODE\tLEFT_LANE\tLEFT_MOLECULAR_BARCODE_SEQUENCE\tLEFT_LIBRARY\tLEFT_SAMPLE\tRIGHT_RUN_BARCODE\tRIGHT_LANE\tRIGHT_MOLECULAR_BARCODE_SEQUENCE\tRIGHT_LIBRARY\tRIGHT_SAMPLE");
            printStream.println(String.join((CharSequence)"\n", arrayList2));
        }
        if (n + n2 > 0) {
            this.log.info(new Object[]{"WARNING: At least two read groups did not relate as expected."});
            return this.EXIT_CODE_WHEN_MISMATCH;
        }
        this.log.info(new Object[]{"All read groups related as expected."});
        return 0;
    }

    private String getMatchDetails(String string, MatchResults matchResults, SAMReadGroupRecord sAMReadGroupRecord, SAMReadGroupRecord sAMReadGroupRecord2) {
        ArrayList<String> arrayList = new ArrayList<String>(4);
        arrayList.add(string);
        arrayList.add(this.formatUtil.format(matchResults.getLOD()));
        arrayList.add(this.formatUtil.format(matchResults.getLodTN()));
        arrayList.add(this.formatUtil.format(matchResults.getLodNT()));
        arrayList.add(this.getReadGroupDetails(sAMReadGroupRecord));
        arrayList.add(this.getReadGroupDetails(sAMReadGroupRecord2));
        return String.join((CharSequence)"\t", arrayList);
    }

    private String getReadGroupDetails(SAMReadGroupRecord sAMReadGroupRecord) {
        ArrayList<String> arrayList = new ArrayList<String>(5);
        String[] stringArray = sAMReadGroupRecord.getPlatformUnit().split("\\.");
        String string = "?";
        String string2 = "?";
        String string3 = "?";
        if (stringArray.length == 3 || stringArray.length == 2) {
            string = stringArray[0];
            string2 = stringArray[1];
            string3 = stringArray.length == 3 ? stringArray[2] : "";
        } else {
            this.log.error(new Object[]{"Unexpected format " + sAMReadGroupRecord.getPlatformUnit() + " for PU attribute"});
        }
        arrayList.add(string);
        arrayList.add(string2);
        arrayList.add(string3);
        arrayList.add(sAMReadGroupRecord.getLibrary());
        arrayList.add(sAMReadGroupRecord.getSample());
        return String.join((CharSequence)"\t", arrayList);
    }
}

