/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.ReservedTagConstants;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.fastq.FastqConstants;
import htsjdk.samtools.fastq.FastqReader;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.util.FastqQualityFormat;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Iso8601Date;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityEncodingDetector;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.SolexaQualityConverter;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.ReadDataManipulationProgramGroup;
import picard.nio.PicardHtsPath;

@CommandLineProgramProperties(summary="<p>Converts a FASTQ file to an unaligned BAM or SAM file.</p><p>Output read records will contain the original base calls and quality scores will be translated depending on the base quality score encoding: FastqSanger, FastqSolexa and FastqIllumina.</p><p>There are also arguments to provide values for SAM header and read attributes that are not present in FASTQ (e.g see RG or SM below).</p><h3>Inputs</h3><p>One FASTQ file name for single-end or two for pair-end sequencing input data. These files might be in gzip compressed format (when file name is ending with \".gz\").</p><p>Alternatively, for larger inputs you can provide a collection of FASTQ files indexed by their name (see USE_SEQUENCIAL_FASTQ for details below).</p><p>By default, this tool will try to guess the base quality score encoding. However you can indicate it explicitly using the QUALITY_FORMAT argument.</p><h3>Output</h3><p>A single unaligned BAM or SAM file. By default, the records are sorted by query (read) name.</p><h3>Usage examples</h3><h4>Example 1:</h4><p>Single-end sequencing FASTQ file conversion. All reads are annotated as belonging to the \"rg0013\" read group that in turn is part of the sample \"sample001\".</p><pre>java -jar picard.jar FastqToSam \\\n        F1=input_reads.fastq \\\n        O=unaligned_reads.bam \\\n        SM=sample001 \\\n        RG=rg0013</pre><h4>Example 2:</h4><p>Similar to example 1 above, but for paired-end sequencing.</p><pre>java -jar picard.jar FastqToSam \\\n       F1=forward_reads.fastq \\\n       F2=reverse_reads.fastq \\\n       O=unaligned_read_pairs.bam \\\n       SM=sample001 \\\n       RG=rg0013</pre><hr />", oneLineSummary="Converts a FASTQ file to an unaligned BAM or SAM file", programGroup=ReadDataManipulationProgramGroup.class)
@DocumentedFeature
public class FastqToSam
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Converts a FASTQ file to an unaligned BAM or SAM file";
    static final String USAGE_DETAILS = "<p>Output read records will contain the original base calls and quality scores will be translated depending on the base quality score encoding: FastqSanger, FastqSolexa and FastqIllumina.</p><p>There are also arguments to provide values for SAM header and read attributes that are not present in FASTQ (e.g see RG or SM below).</p><h3>Inputs</h3><p>One FASTQ file name for single-end or two for pair-end sequencing input data. These files might be in gzip compressed format (when file name is ending with \".gz\").</p><p>Alternatively, for larger inputs you can provide a collection of FASTQ files indexed by their name (see USE_SEQUENCIAL_FASTQ for details below).</p><p>By default, this tool will try to guess the base quality score encoding. However you can indicate it explicitly using the QUALITY_FORMAT argument.</p><h3>Output</h3><p>A single unaligned BAM or SAM file. By default, the records are sorted by query (read) name.</p><h3>Usage examples</h3><h4>Example 1:</h4><p>Single-end sequencing FASTQ file conversion. All reads are annotated as belonging to the \"rg0013\" read group that in turn is part of the sample \"sample001\".</p><pre>java -jar picard.jar FastqToSam \\\n        F1=input_reads.fastq \\\n        O=unaligned_reads.bam \\\n        SM=sample001 \\\n        RG=rg0013</pre><h4>Example 2:</h4><p>Similar to example 1 above, but for paired-end sequencing.</p><pre>java -jar picard.jar FastqToSam \\\n       F1=forward_reads.fastq \\\n       F2=reverse_reads.fastq \\\n       O=unaligned_read_pairs.bam \\\n       SM=sample001 \\\n       RG=rg0013</pre><hr />";
    private static final Log LOG = Log.getInstance(FastqToSam.class);
    @Argument(shortName="F1", doc="Input fastq file (optionally gzipped) for single end data, or first read in paired end data.")
    public PicardHtsPath FASTQ;
    @Argument(shortName="F2", doc="Input fastq file (optionally gzipped) for the second read of paired end data.", optional=true)
    public PicardHtsPath FASTQ2;
    @Argument(doc="Use sequential fastq files with the suffix <prefix>_###.fastq or <prefix>_###.fastq.gz.The files should be named:\n    <prefix>_001.<extension>, <prefix>_002.<extension>, ..., <prefix>_XYZ.<extension>\n The base files should be:\n    <prefix>_001.<extension>\n An example would be:\n    RUNNAME_S8_L005_R1_001.fastq\n    RUNNAME_S8_L005_R1_002.fastq\n    RUNNAME_S8_L005_R1_003.fastq\n    RUNNAME_S8_L005_R1_004.fastq\nRUNNAME_S8_L005_R1_001.fastq should be provided as FASTQ.", optional=true)
    public boolean USE_SEQUENTIAL_FASTQS = false;
    @Argument(shortName="V", doc="A value describing how the quality values are encoded in the input FASTQ file.  Either Solexa (phred scaling + 66), Illumina (phred scaling + 64) or Standard (phred scaling + 33).  If this value is not specified, the quality format will be detected automatically.", optional=true)
    public FastqQualityFormat QUALITY_FORMAT;
    @Argument(doc="Output BAM/SAM/CRAM file. ", shortName="O")
    public File OUTPUT;
    @Argument(shortName="RG", doc="Read group name")
    public String READ_GROUP_NAME = "A";
    @Argument(shortName="SM", doc="Sample name to insert into the read group header")
    public String SAMPLE_NAME;
    @Argument(shortName="LB", doc="The library name to place into the LB attribute in the read group header", optional=true)
    public String LIBRARY_NAME;
    @Argument(shortName="PU", doc="The platform unit (often run_barcode.lane) to insert into the read group header", optional=true)
    public String PLATFORM_UNIT;
    @Argument(shortName="PL", doc="The platform type (e.g. ILLUMINA, SOLID) to insert into the read group header", optional=true)
    public String PLATFORM;
    @Argument(shortName="CN", doc="The sequencing center from which the data originated", optional=true)
    public String SEQUENCING_CENTER;
    @Argument(shortName="PI", doc="Predicted median insert size, to insert into the read group header", optional=true)
    public Integer PREDICTED_INSERT_SIZE;
    @Argument(shortName="PG", doc="Program group to insert into the read group header.", optional=true)
    public String PROGRAM_GROUP;
    @Argument(shortName="PM", doc="Platform model to insert into the group header (free-form text providing further details of the platform/technology used)", optional=true)
    public String PLATFORM_MODEL;
    @Argument(doc="Comment(s) to include in the merged output file's header.", optional=true, shortName="CO")
    public List<String> COMMENT = new ArrayList<String>();
    @Argument(shortName="DS", doc="Inserted into the read group header", optional=true)
    public String DESCRIPTION;
    @Argument(shortName="DT", doc="Date the run was produced, to insert into the read group header", optional=true)
    public Iso8601Date RUN_DATE;
    @Argument(shortName="SO", doc="The sort order for the output BAM/SAM/CRAM file.")
    public SAMFileHeader.SortOrder SORT_ORDER = SAMFileHeader.SortOrder.queryname;
    @Argument(doc="Minimum quality allowed in the input fastq.  An exception will be thrown if a quality is less than this value.")
    public int MIN_Q = 0;
    @Argument(doc="Maximum quality allowed in the input fastq.  An exception will be thrown if a quality is greater than this value.")
    public int MAX_Q = 93;
    @Deprecated
    @Argument(doc="Deprecated (No longer used). If true and this is an unpaired fastq any occurrence of '/1' or '/2' will be removed from the end of a read name.")
    public Boolean STRIP_UNPAIRED_MATE_NUMBER = false;
    @Argument(doc="Allow (and ignore) empty lines")
    public Boolean ALLOW_AND_IGNORE_EMPTY_LINES = false;
    @Argument(doc="Allow empty input fastq")
    public Boolean ALLOW_EMPTY_FASTQ = false;
    private static final SolexaQualityConverter solexaQualityConverter = SolexaQualityConverter.getSingleton();

    public static FastqQualityFormat determineQualityFormat(FastqReader reader1, FastqReader reader2, FastqQualityFormat expectedQuality) {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        if (reader2 == null) {
            detector.add(10000L, new FastqReader[]{reader1});
        } else {
            detector.add(10000L, new FastqReader[]{reader1, reader2});
            reader2.close();
        }
        reader1.close();
        FastqQualityFormat qualityFormat = detector.generateBestGuess(QualityEncodingDetector.FileContext.FASTQ, expectedQuality);
        if (detector.isDeterminationAmbiguous()) {
            LOG.warn(new Object[]{"Making ambiguous determination about fastq's quality encoding; more than one format possible based on observed qualities."});
        }
        LOG.info(new Object[]{String.format("Auto-detected quality format as: %s.", qualityFormat)});
        return qualityFormat;
    }

    protected static List<Path> getSequentialFileList(Path baseFastq) {
        ArrayList<Path> files = new ArrayList<Path>();
        files.add(baseFastq);
        FastqConstants.FastqExtensions fastqExtensions = null;
        String suffix = null;
        for (FastqConstants.FastqExtensions ext : FastqConstants.FastqExtensions.values()) {
            suffix = "_001" + ext.getExtension();
            if (!baseFastq.toString().endsWith(suffix)) continue;
            fastqExtensions = ext;
            break;
        }
        if (null == fastqExtensions) {
            throw new PicardException(String.format("Could not parse the FASTQ extension (expected '_001' + '%s'): %s", Arrays.toString(FastqConstants.FastqExtensions.values()), baseFastq));
        }
        int idx = 2;
        while (true) {
            String fastq = baseFastq.toAbsolutePath().toString();
            fastq = String.format("%s_%03d%s", fastq.substring(0, fastq.length() - suffix.length()), idx, fastqExtensions.getExtension());
            try {
                IOUtil.assertFileIsReadable((Path)Paths.get(fastq, new String[0]));
            }
            catch (SAMException e) {
                break;
            }
            files.add(Paths.get(fastq, new String[0]));
            ++idx;
        }
        return files;
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((Path)this.FASTQ.toPath());
        if (this.FASTQ2 != null) {
            IOUtil.assertFileIsReadable((Path)this.FASTQ2.toPath());
        }
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        SAMFileHeader header = this.createSamFileHeader();
        SAMFileWriter writer = new SAMFileWriterFactory().makeWriter(header, false, this.OUTPUT, this.REFERENCE_SEQUENCE);
        FastqReader reader1 = this.fileToFastqReader(this.FASTQ.toPath());
        if (reader1.hasNext()) {
            this.QUALITY_FORMAT = FastqToSam.determineQualityFormat(reader1, this.FASTQ2 == null ? null : this.fileToFastqReader(this.FASTQ2.toPath()), this.QUALITY_FORMAT);
        } else if (this.ALLOW_EMPTY_FASTQ.booleanValue()) {
            LOG.warn(new Object[]{"Input FASTQ is empty, will write out SAM/BAM file with no reads."});
        } else {
            throw new PicardException("Input fastq is empty. Set ALLOW_EMPTY_FASTQ if you still want to write out a file with no reads.");
        }
        ArrayList<FastqReader> readers1 = new ArrayList<FastqReader>();
        ArrayList<FastqReader> readers2 = new ArrayList<FastqReader>();
        if (this.USE_SEQUENTIAL_FASTQS) {
            for (Path fastq : FastqToSam.getSequentialFileList(this.FASTQ.toPath())) {
                readers1.add(this.fileToFastqReader(fastq));
            }
            if (null != this.FASTQ2) {
                for (Path fastq : FastqToSam.getSequentialFileList(this.FASTQ2.toPath())) {
                    readers2.add(this.fileToFastqReader(fastq));
                }
                if (readers1.size() != readers2.size()) {
                    throw new PicardException(String.format("Found %d files for FASTQ and %d files for FASTQ2.", readers1.size(), readers2.size()));
                }
            }
        } else {
            readers1.add(this.fileToFastqReader(this.FASTQ.toPath()));
            if (this.FASTQ2 != null) {
                readers2.add(this.fileToFastqReader(this.FASTQ2.toPath()));
            }
        }
        for (int idx = 0; idx < readers1.size(); ++idx) {
            this.makeItSo((FastqReader)readers1.get(idx), readers2.isEmpty() ? null : (FastqReader)readers2.get(idx), writer);
        }
        for (FastqReader reader : readers1) {
            reader.close();
        }
        for (FastqReader reader : readers2) {
            reader.close();
        }
        writer.close();
        return 0;
    }

    public void makeItSo(FastqReader reader1, FastqReader reader2, SAMFileWriter writer) {
        int readCount = reader2 == null ? this.doUnpaired(reader1, writer) : this.doPaired(reader1, reader2, writer);
        LOG.info(new Object[]{"Processed " + readCount + " fastq reads"});
    }

    protected int doUnpaired(FastqReader freader, SAMFileWriter writer) {
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader.hasNext()) {
            FastqRecord frec = freader.next();
            SAMRecord srec = this.createSamRecord(writer.getFileHeader(), SequenceUtil.getSamReadNameFromFastqHeader((String)frec.getReadHeader()), frec, false);
            srec.setReadPairedFlag(false);
            writer.addAlignment(srec);
            progress.record(srec);
            ++readCount;
        }
        return readCount;
    }

    protected int doPaired(FastqReader freader1, FastqReader freader2, SAMFileWriter writer) {
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader1.hasNext() && freader2.hasNext()) {
            FastqRecord frec1 = freader1.next();
            FastqRecord frec2 = freader2.next();
            String frec1Name = SequenceUtil.getSamReadNameFromFastqHeader((String)frec1.getReadHeader());
            String frec2Name = SequenceUtil.getSamReadNameFromFastqHeader((String)frec2.getReadHeader());
            String baseName = this.getBaseName(frec1Name, frec2Name, freader1, freader2);
            SAMRecord srec1 = this.createSamRecord(writer.getFileHeader(), baseName, frec1, true);
            srec1.setFirstOfPairFlag(true);
            srec1.setSecondOfPairFlag(false);
            writer.addAlignment(srec1);
            progress.record(srec1);
            SAMRecord srec2 = this.createSamRecord(writer.getFileHeader(), baseName, frec2, true);
            srec2.setFirstOfPairFlag(false);
            srec2.setSecondOfPairFlag(true);
            writer.addAlignment(srec2);
            progress.record(srec2);
            ++readCount;
        }
        if (freader1.hasNext() || freader2.hasNext()) {
            throw new PicardException("Input paired fastq files must be the same length");
        }
        return readCount;
    }

    private FastqReader fileToFastqReader(Path path) throws PicardException {
        return new FastqReader(null, IOUtil.openFileForBufferedReading((Path)path), this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
    }

    private SAMRecord createSamRecord(SAMFileHeader header, String baseName, FastqRecord frec, boolean paired) {
        SAMRecord srec = new SAMRecord(header);
        srec.setReadName(baseName);
        srec.setReadString(frec.getReadString());
        srec.setReadUnmappedFlag(true);
        srec.setAttribute(ReservedTagConstants.READ_GROUP_ID, (Object)this.READ_GROUP_NAME);
        byte[] quals = StringUtil.stringToBytes((String)frec.getBaseQualityString());
        this.convertQuality(quals, this.QUALITY_FORMAT);
        for (byte qual : quals) {
            int uQual = qual & 0xFF;
            if (uQual >= this.MIN_Q && uQual <= this.MAX_Q) continue;
            throw new PicardException("Base quality " + uQual + " is not in the range " + this.MIN_Q + ".." + this.MAX_Q + " for read " + frec.getReadHeader());
        }
        srec.setBaseQualities(quals);
        if (paired) {
            srec.setReadPairedFlag(true);
            srec.setMateUnmappedFlag(true);
        }
        return srec;
    }

    public SAMFileHeader createSamFileHeader() {
        SAMReadGroupRecord rgroup = new SAMReadGroupRecord(this.READ_GROUP_NAME);
        rgroup.setSample(this.SAMPLE_NAME);
        if (this.LIBRARY_NAME != null) {
            rgroup.setLibrary(this.LIBRARY_NAME);
        }
        if (this.PLATFORM != null) {
            rgroup.setPlatform(this.PLATFORM);
        }
        if (this.PLATFORM_UNIT != null) {
            rgroup.setPlatformUnit(this.PLATFORM_UNIT);
        }
        if (this.SEQUENCING_CENTER != null) {
            rgroup.setSequencingCenter(this.SEQUENCING_CENTER);
        }
        if (this.PREDICTED_INSERT_SIZE != null) {
            rgroup.setPredictedMedianInsertSize(this.PREDICTED_INSERT_SIZE);
        }
        if (this.DESCRIPTION != null) {
            rgroup.setDescription(this.DESCRIPTION);
        }
        if (this.RUN_DATE != null) {
            rgroup.setRunDate((Date)this.RUN_DATE);
        }
        if (this.PLATFORM_MODEL != null) {
            rgroup.setPlatformModel(this.PLATFORM_MODEL);
        }
        if (this.PROGRAM_GROUP != null) {
            rgroup.setProgramGroup(this.PROGRAM_GROUP);
        }
        SAMFileHeader header = new SAMFileHeader();
        header.addReadGroup(rgroup);
        for (String comment : this.COMMENT) {
            header.addComment(comment);
        }
        header.setSortOrder(this.SORT_ORDER);
        return header;
    }

    void convertQuality(byte[] quals, FastqQualityFormat version) {
        switch (version) {
            case Standard: {
                SAMUtils.fastqToPhred((byte[])quals);
                break;
            }
            case Solexa: {
                solexaQualityConverter.convertSolexaQualityCharsToPhredBinary(quals);
                break;
            }
            case Illumina: {
                solexaQualityConverter.convertSolexa_1_3_QualityCharsToPhredBinary(quals);
            }
        }
    }

    String getBaseName(String readName1, String readName2, FastqReader freader1, FastqReader freader2) {
        String[] toks = this.getReadNameTokens(readName1, 1, freader1);
        String baseName1 = toks[0];
        String num1 = toks[1];
        toks = this.getReadNameTokens(readName2, 2, freader2);
        String baseName2 = toks[0];
        String num2 = toks[1];
        if (!baseName1.equals(baseName2)) {
            throw new PicardException(String.format("In paired mode, read name 1 (%s) does not match read name 2 (%s)", baseName1, baseName2));
        }
        boolean num1Blank = StringUtil.isBlank((String)num1);
        boolean num2Blank = StringUtil.isBlank((String)num2);
        if (num1Blank || num2Blank) {
            if (!num1Blank) {
                throw new PicardException(this.error(freader1, "Pair 1 number is missing (" + readName1 + "). Both pair numbers must be present or neither."));
            }
            if (!num2Blank) {
                throw new PicardException(this.error(freader2, "Pair 2 number is missing (" + readName2 + "). Both pair numbers must be present or neither."));
            }
        } else {
            if (!num1.equals("1")) {
                throw new PicardException(this.error(freader1, "Pair 1 number must be 1 (" + readName1 + ")"));
            }
            if (!num2.equals("2")) {
                throw new PicardException(this.error(freader2, "Pair 2 number must be 2 (" + readName2 + ")"));
            }
        }
        return baseName1;
    }

    private String[] getReadNameTokens(String readName, int pairNum, FastqReader freader) {
        if (readName.equals("")) {
            throw new PicardException(this.error(freader, "Pair read name " + pairNum + " cannot be empty: " + readName));
        }
        int idx = readName.lastIndexOf(47);
        String[] result = new String[2];
        if (idx == -1) {
            result[0] = readName;
            result[1] = null;
        } else {
            result[1] = readName.substring(idx + 1, readName.length());
            if (!result[1].equals("1") && !result[1].equals("2")) {
                result[0] = readName;
                result[1] = null;
            } else {
                result[0] = readName.substring(0, idx);
            }
        }
        return result;
    }

    private String error(FastqReader freader, String str) {
        return str + " at line " + freader.getLineNumber() + " in file " + freader.getFile().getAbsolutePath();
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.MIN_Q < 0) {
            return new String[]{"MIN_Q must be >= 0"};
        }
        if (this.MAX_Q > 93) {
            return new String[]{"MAX_Q must be <= 93"};
        }
        return null;
    }
}

