/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import com.google.common.annotations.VisibleForTesting;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.filter.AlignedFilter;
import htsjdk.samtools.filter.FilteringSamIterator;
import htsjdk.samtools.filter.IntervalKeepPairFilter;
import htsjdk.samtools.filter.JavascriptSamRecordFilter;
import htsjdk.samtools.filter.ReadNameFilter;
import htsjdk.samtools.filter.SamRecordFilter;
import htsjdk.samtools.filter.TagFilter;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineParser;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.ReadDataManipulationProgramGroup;

@CommandLineProgramProperties(summary="Subsets reads from a SAM/BAM/CRAM file by applying one of several filters.\nTakes a SAM/BAM/CRAM file and subsets it by either excluding or only including certain reads such as aligned or unaligned reads, specific reads based on a list of reads names, an interval list, by Tag Values (type Z / String values only), or using a JavaScript script.\n<br /><h3>Usage example:</h3><h4>Filter by queryname</h4><pre>java -jar picard.jar FilterSamReads \\<br />       I=input.bam \\ <br />       O=output.bam \\ <br />       READ_LIST_FILE=read_names.txt \\ <br />      FILTER=includeReadList</pre> <h4>Filter by interval</h4><pre>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       INTERVAL_LIST=regions.interval_list \\ <br/>      FILTER=includePairedIntervals</pre> <h4>Filter by Tag Value (type Z / String values only)</h4><pre>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       TAG=CR \\ <br/>      TAG_VALUE=TTTGTCATCTCGAGTA \\ <br/>      FILTER=includeTagValues</pre> <h4>Filter reads having a soft clip on the beginning of the read larger than 2 bases with a JavaScript script</h4><pre>cat <<EOF > script.js <br/>/** reads having a soft clip larger than 2 bases in beginning of read*/ <br/>function accept(rec) {   <br/>    if (rec.getReadUnmappedFlag()) return false; <br/>    var cigar = rec.getCigar(); <br/>    if (cigar == null) return false; <br/>    var ce = cigar.getCigarElement(0); <br/>    return ce.getOperator().name() == \"S\" && ce.length() > 2; <br/>} <br /><br />accept(record); <br/>EOF <br/><br/>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       JAVASCRIPT_FILE=script.js \\ <br/>      FILTER=includeJavascript</pre> ", oneLineSummary="Subsets reads from a SAM/BAM/CRAM file by applying one of several filters.", programGroup=ReadDataManipulationProgramGroup.class)
@DocumentedFeature
public class FilterSamReads
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Subsets reads from a SAM/BAM/CRAM file by applying one of several filters.";
    static final String USAGE_DETAILS = "\nTakes a SAM/BAM/CRAM file and subsets it by either excluding or only including certain reads such as aligned or unaligned reads, specific reads based on a list of reads names, an interval list, by Tag Values (type Z / String values only), or using a JavaScript script.\n<br /><h3>Usage example:</h3><h4>Filter by queryname</h4><pre>java -jar picard.jar FilterSamReads \\<br />       I=input.bam \\ <br />       O=output.bam \\ <br />       READ_LIST_FILE=read_names.txt \\ <br />      FILTER=includeReadList</pre> <h4>Filter by interval</h4><pre>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       INTERVAL_LIST=regions.interval_list \\ <br/>      FILTER=includePairedIntervals</pre> <h4>Filter by Tag Value (type Z / String values only)</h4><pre>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       TAG=CR \\ <br/>      TAG_VALUE=TTTGTCATCTCGAGTA \\ <br/>      FILTER=includeTagValues</pre> <h4>Filter reads having a soft clip on the beginning of the read larger than 2 bases with a JavaScript script</h4><pre>cat <<EOF > script.js <br/>/** reads having a soft clip larger than 2 bases in beginning of read*/ <br/>function accept(rec) {   <br/>    if (rec.getReadUnmappedFlag()) return false; <br/>    var cigar = rec.getCigar(); <br/>    if (cigar == null) return false; <br/>    var ce = cigar.getCigarElement(0); <br/>    return ce.getOperator().name() == \"S\" && ce.length() > 2; <br/>} <br /><br />accept(record); <br/>EOF <br/><br/>java -jar picard.jar FilterSamReads \\ <br />       I=input.bam \\ <br />       O=output.bam \\ <br />       JAVASCRIPT_FILE=script.js \\ <br/>      FILTER=includeJavascript</pre> ";
    private static final Log log = Log.getInstance(FilterSamReads.class);
    @Argument(doc="The SAM/BAM/CRAM file that will be filtered.", shortName="I")
    public File INPUT;
    @Argument(doc="Which filter to use.")
    public Filter FILTER = null;
    @Argument(doc="File containing reads that will be included in or excluded from the OUTPUT SAM/BAM/CRAM file, when using FILTER=includeReadList or FILTER=excludeReadList.", optional=true, shortName="RLF")
    public File READ_LIST_FILE;
    @Argument(doc="Interval List File containing intervals that will be included in the OUTPUT when using FILTER=includePairedIntervals", optional=true, shortName="IL")
    public File INTERVAL_LIST;
    @Argument(doc="The tag to select from input SAM/BAM", optional=true, shortName="T")
    public String TAG;
    @Argument(doc="The tag value(s) to filter by", optional=true, shortName="TV")
    public List<String> TAG_VALUE;
    @Argument(doc="SortOrder of the OUTPUT file, otherwise use the SortOrder of the INPUT file.", optional=true, shortName="SO")
    public SAMFileHeader.SortOrder SORT_ORDER;
    @Argument(doc="SAM/BAM/CRAM file for resulting reads.", shortName="O")
    public File OUTPUT;
    @Argument(shortName="JS", doc="Filters the INPUT with a javascript expression using the java javascript-engine, when using FILTER=includeJavascript.  The script puts the following variables in the script context: \n 'record' a SamRecord ( https://samtools.github.io/htsjdk/javadoc/htsjdk/htsjdk/samtools/SAMRecord.html ) and \n  'header' a SAMFileHeader ( https://samtools.github.io/htsjdk/javadoc/htsjdk/htsjdk/samtools/SAMFileHeader.html ).\n all the public members of SamRecord and SAMFileHeader are accessible. A record is accepted if the last value of the script evaluates to true.", optional=true)
    public File JAVASCRIPT_FILE = null;
    @Argument(doc="Create <OUTPUT>.reads file containing names of reads from INPUT and OUTPUT (for debugging purposes.)", optional=true)
    public boolean WRITE_READS_FILES = false;

    void setReferenceSequence(File referenceSequence) {
        this.REFERENCE_SEQUENCE = referenceSequence;
    }

    private void filterReads(FilteringSamIterator filteringIterator) {
        SAMFileHeader fileHeader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).getFileHeader(this.INPUT);
        SAMFileHeader.SortOrder inputSortOrder = fileHeader.getSortOrder();
        if (this.SORT_ORDER != null) {
            fileHeader.setSortOrder(this.SORT_ORDER);
        }
        if (this.FILTER == Filter.includePairedIntervals && fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new UnsupportedOperationException("Input must be coordinate sorted to use includePairedIntervals");
        }
        boolean presorted = inputSortOrder.equals((Object)fileHeader.getSortOrder());
        log.info(new Object[]{"Filtering [presorted=" + presorted + "] " + this.INPUT.getName() + " -> OUTPUT=" + this.OUTPUT.getName() + " [sortorder=" + fileHeader.getSortOrder().name() + "]"});
        SAMFileWriter outputWriter = new SAMFileWriterFactory().makeWriter(fileHeader, presorted, this.OUTPUT, this.REFERENCE_SEQUENCE);
        ProgressLogger progress = new ProgressLogger(log, 1000000, "Written");
        while (filteringIterator.hasNext()) {
            SAMRecord rec = filteringIterator.next();
            outputWriter.addAlignment(rec);
            progress.record(rec);
        }
        filteringIterator.close();
        outputWriter.close();
        log.info(new Object[]{new DecimalFormat("#,###").format(progress.getCount()) + " SAMRecords written to " + this.OUTPUT.getName()});
    }

    private void writeReadsFile(File samOrBamFile) throws IOException {
        File readsFile = new File(this.OUTPUT.getParentFile(), IOUtil.basename((File)samOrBamFile) + ".reads");
        IOUtil.assertFileIsWritable((File)readsFile);
        try (SamReader reader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(samOrBamFile);
             BufferedWriter bw = IOUtil.openFileForBufferedWriting((File)readsFile, (boolean)false);){
            for (SAMRecord rec : reader) {
                bw.write(rec.toString() + "\n");
            }
        }
        IOUtil.assertFileIsReadable((File)readsFile);
    }

    private List<Interval> getIntervalList(File intervalFile) throws IOException {
        IOUtil.assertFileIsReadable((File)intervalFile);
        return IntervalList.fromFile((File)intervalFile).getIntervals();
    }

    @Override
    protected int doWork() {
        try {
            IOUtil.assertFileIsReadable((File)this.INPUT);
            IOUtil.assertFileIsWritable((File)this.OUTPUT);
            if (this.WRITE_READS_FILES) {
                this.writeReadsFile(this.INPUT);
            }
            SamReader samReader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
            List<String> tagList = this.TAG_VALUE;
            this.filterReads(switch (this.FILTER) {
                case Filter.includeAligned -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new AlignedFilter(true), true);
                case Filter.excludeAligned -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new AlignedFilter(false), true);
                case Filter.includeReadList -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new ReadNameFilter(this.READ_LIST_FILE, true));
                case Filter.excludeReadList -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new ReadNameFilter(this.READ_LIST_FILE, false));
                case Filter.includeJavascript -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new JavascriptSamRecordFilter(this.JAVASCRIPT_FILE, samReader.getFileHeader()));
                case Filter.includePairedIntervals -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new IntervalKeepPairFilter(this.getIntervalList(this.INTERVAL_LIST)));
                case Filter.includeTagValues -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new TagFilter(this.TAG, tagList, Boolean.valueOf(true)));
                case Filter.excludeTagValues -> new FilteringSamIterator((Iterator)samReader.iterator(), (SamRecordFilter)new TagFilter(this.TAG, tagList, Boolean.valueOf(false)));
                default -> throw new UnsupportedOperationException(this.FILTER.name() + " has not been implemented!");
            });
            IOUtil.assertFileIsReadable((File)this.OUTPUT);
            if (this.WRITE_READS_FILES) {
                this.writeReadsFile(this.OUTPUT);
            }
            return 0;
        }
        catch (Exception e) {
            log.error((Throwable)e, new Object[]{"Failed to filter " + this.INPUT.getName()});
            if (this.OUTPUT.exists() && !this.OUTPUT.delete()) {
                log.warn(new Object[]{"Failed to delete possibly incomplete output file:" + this.OUTPUT.getAbsolutePath()});
            }
            return 1;
        }
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errors = new ArrayList<String>();
        if (this.INPUT.equals(this.OUTPUT)) {
            errors.add("INPUT file and OUTPUT file must differ!");
        }
        List<Filter> tagFilters = Arrays.asList(Filter.includeTagValues, Filter.excludeTagValues);
        this.checkInputs(Arrays.asList(Filter.includeReadList, Filter.excludeReadList), this.READ_LIST_FILE, "READ_LIST_FILE").ifPresent(errors::add);
        this.checkInputs(Collections.singletonList(Filter.includePairedIntervals), this.INTERVAL_LIST, "INTERVAL_LIST").ifPresent(errors::add);
        this.checkInputs(Collections.singletonList(Filter.includeJavascript), this.JAVASCRIPT_FILE, "JAVASCRIPT_FILE").ifPresent(errors::add);
        this.checkInputs(tagFilters, this.TAG, "TAG").ifPresent(errors::add);
        if (tagFilters.contains((Object)this.FILTER) && this.TAG_VALUE.isEmpty()) {
            log.warn(new Object[]{"Running FilterSamReads with a Tag Filter but no TAG_VALUE argument provided.  This will recreate the original input file i.e. not filter anything"});
        }
        if (!errors.isEmpty()) {
            return errors.toArray(new String[errors.size()]);
        }
        return super.customCommandLineValidation();
    }

    private Optional<String> checkInputs(List<Filter> filters, Object inputObject, String inputFileVariable) {
        if (filters.contains((Object)this.FILTER) && inputObject == null) {
            return Optional.of(String.format("%s must be specified when using FILTER=%s, but it was null.", new Object[]{inputFileVariable, this.FILTER}));
        }
        if (!filters.contains((Object)this.FILTER) && inputObject != null) {
            return Optional.of(String.format("%s may only be specified when using FILTER from %s, FILTER value: %s, %s value: %s", new Object[]{inputFileVariable, String.join((CharSequence)", ", filters.stream().map(Enum::toString).collect(Collectors.toList())), this.FILTER, inputFileVariable, inputObject}));
        }
        return Optional.empty();
    }

    @VisibleForTesting
    protected static enum Filter implements CommandLineParser.ClpEnum
    {
        includeAligned("Output aligned reads only. INPUT SAM/BAM/CRAM must be in queryname SortOrder. (Note: first and second of paired reads must both be aligned to be included in OUTPUT.)"),
        excludeAligned("Output Unmapped reads only. INPUT SAM/BAM/CRAM must be in queryname SortOrder. (Note: first and second of pair must both be aligned to be excluded from OUTPUT.)"),
        includeReadList("Output reads with names contained in READ_LIST_FILE. See READ_LIST_FILE for more detail."),
        excludeReadList("Output reads with names *not* contained in READ_LIST_FILE. See READ_LIST_FILE for more detail."),
        includeJavascript("Output reads that have been accepted by the JAVASCRIPT_FILE script, that is, reads for which the value of the script is true. See the JAVASCRIPT_FILE argument for more detail. "),
        includePairedIntervals("Output reads that overlap with an interval from INTERVAL_LIST (and their mate). INPUT must be coordinate sorted."),
        includeTagValues("Output reads that have a value of tag TAG that is contained in the values for TAG_VALUES"),
        excludeTagValues("Output reads that do not have a value of tag TAG that is contained in the values for TAG_VALUES");

        private final String description;

        private Filter(String description) {
            this.description = description;
        }

        public String getHelpDoc() {
            return this.description;
        }
    }
}

