/*
 * Decompiled with CFR 0.152.
 */
package picard.util;

import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.RuntimeIOException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.OtherProgramGroup;
import picard.util.CircularByteBuffer;

@CommandLineProgramProperties(oneLineSummary="Provides a large, FIFO buffer that can be used to buffer input and output streams between programs.", summary="Acts as a large memory buffer between processes that are connected with unix pipes for the case that one or more processes produces or consumes their input or output in bursts.  By inserting a large memory buffer between such processes each process can run at full speed and the bursts can be smoothed out by the memory buffer.\n \n<h3>Example</h3>\n  java -jar SamToFastq.jar \\\n     F=my.fastq \\\n     INTERLEAVE=true |\n  java -jar FifoBuffer |\n  bwa mem -t 8 \\dev\\stdin output.bam\n", programGroup=OtherProgramGroup.class)
@DocumentedFeature
public class FifoBuffer
extends CommandLineProgram {
    @Argument(doc="The size of the memory buffer in bytes.")
    public int BUFFER_SIZE = 0x20000000;
    @Argument(doc="The size, in bytes, to read/write atomically to the input and output streams.")
    public int IO_SIZE = 65536;
    @Argument(doc="How frequently, in seconds, to report debugging statistics. Set to zero for never.")
    public int DEBUG_FREQUENCY = 0;
    @Argument(doc="Name to use for Fifo in debugging statements.", optional=true)
    public String NAME;
    private final Log log = Log.getInstance(FifoBuffer.class);
    private final InputStream inputStream;
    private final PrintStream outputStream;

    public FifoBuffer(InputStream in, PrintStream out) {
        this.inputStream = in;
        this.outputStream = out;
        this.QUIET = true;
    }

    public FifoBuffer() {
        this(System.in, System.out);
    }

    @Override
    protected int doWork() {
        final CircularByteBuffer fifo = new CircularByteBuffer(this.BUFFER_SIZE);
        Thread input = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    byte[] buffer = new byte[FifoBuffer.this.IO_SIZE];
                    int read = 0;
                    while ((read = FifoBuffer.this.inputStream.read(buffer)) > -1) {
                        for (int start = 0; start < read; start += fifo.write(buffer, start, read - start)) {
                        }
                    }
                }
                catch (IOException ioe) {
                    throw new RuntimeIOException((Throwable)ioe);
                }
                finally {
                    fifo.close();
                }
            }
        });
        Thread output = new Thread(new Runnable(){

            @Override
            public void run() {
                byte[] buffer = new byte[FifoBuffer.this.IO_SIZE];
                int read = 0;
                while ((read = fifo.read(buffer, 0, buffer.length)) > 0 || !fifo.isClosed()) {
                    FifoBuffer.this.outputStream.write(buffer, 0, read);
                }
            }
        });
        try {
            if (this.DEBUG_FREQUENCY > 0) {
                Thread debug = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        NumberFormat pFmt = NumberFormat.getPercentInstance();
                        DecimalFormat iFmt = new DecimalFormat("#,##0");
                        while (true) {
                            int capacity = fifo.getCapacity();
                            int used = fifo.getBytesAvailableToRead();
                            double pct = (double)used / (double)capacity;
                            String name = FifoBuffer.this.NAME == null ? "" : FifoBuffer.this.NAME + " ";
                            FifoBuffer.this.log.info(new Object[]{"Fifo buffer ", name, "used ", iFmt.format(used), " / ", iFmt.format(capacity), " (", pFmt.format(pct), ")."});
                            try {
                                Thread.sleep(FifoBuffer.this.DEBUG_FREQUENCY * 1000);
                            }
                            catch (InterruptedException interruptedException) {
                            }
                        }
                    }
                });
                debug.setName("BufferDebugThread");
                debug.setDaemon(true);
                debug.start();
            }
            LoggingExceptionHandler inputExceptionHandler = new LoggingExceptionHandler();
            input.setUncaughtExceptionHandler(inputExceptionHandler);
            input.setName("Fifo Input Thread");
            input.start();
            LoggingExceptionHandler outputExceptionHandler = new LoggingExceptionHandler();
            output.setUncaughtExceptionHandler(new LoggingExceptionHandler());
            output.setName("Fifo Output Thread");
            output.start();
            input.join();
            output.join();
            if (inputExceptionHandler.throwable != null) {
                throw new PicardException("Exception on input thread.", inputExceptionHandler.throwable);
            }
            if (outputExceptionHandler.throwable != null) {
                throw new PicardException("Exception on output thread.", outputExceptionHandler.throwable);
            }
        }
        catch (InterruptedException ie) {
            throw new PicardException("Interrupted!", ie);
        }
        return 0;
    }

    class LoggingExceptionHandler
    implements Thread.UncaughtExceptionHandler {
        public Throwable throwable;

        LoggingExceptionHandler() {
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            this.throwable = e;
            FifoBuffer.this.log.error(e, new Object[]{"Exception caught on thread ", t.getName()});
        }
    }
}

