/*
 * Decompiled with CFR 0.152.
 */
package picard.vcf.MendelianViolations;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.Lazy;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.variant.variantcontext.VariantContextComparator;
import htsjdk.variant.variantcontext.writer.Options;
import htsjdk.variant.variantcontext.writer.VariantContextWriter;
import htsjdk.variant.variantcontext.writer.VariantContextWriterBuilder;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLineCount;
import htsjdk.variant.vcf.VCFHeaderLineType;
import htsjdk.variant.vcf.VCFInfoHeaderLine;
import java.io.File;
import java.lang.invoke.CallSite;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.VariantEvaluationProgramGroup;
import picard.pedigree.PedFile;
import picard.vcf.MendelianViolations.MendelianViolationDetector;
import picard.vcf.MendelianViolations.MendelianViolationMetrics;
import picard.vcf.processor.VariantProcessor;

@CommandLineProgramProperties(summary="Takes in VCF or BCF and a pedigree file and looks for high confidence calls where the genotype of the offspring is incompatible with the genotypes of the parents.  \nKey features:\n- Checks for regular MVs in diploid regions and invalid transmissions in haploid regions (using the declared gender of the offspring in the pedigree file to determine how to deal with the male and female chromosomes.)\n- Outputs metrics about the different kinds of MVs found.\n- Can output a per-trio VCF with violations; INFO field will indicate the type of violation in the MV field\n<h3>Example</h3>\n    java -jar picard.jar FindMendelianViolations\\\n         I=input.vcf \\\n         TRIO=pedigree.fam \\\n         O=report.mendelian_violation_metrics \\\n         MIN_DP=20\n<h3>Caveates</h3>\n<h4>Assumptions</h4>\nThe tool assumes the existence of FORMAT fields AD, DP, GT, GQ, and PL. \n<h4>Ignored Variants</h4>\nThis tool ignores variants that are:\n- Not SNPs\n- Filtered\n- Multiallelic (i.e., trio has more than 2 alleles)\n- Within the SKIP_CHROMS contigs\n<h4>PseudoAutosomal Region</h4>\nThis tool assumes that variants in the PAR will be mapped onto the female chromosome, and will treat variants in that region as as autosomal. The mapping to female requires that the PAR in the male chromosome be masked so that the aligner maps reads to single contig. This is normally done for the public releases of the human reference. The tool has default values for PAR that are sensible for humans on either build b37 or hg38.\n", oneLineSummary="Finds mendelian violations of all types within a VCF", programGroup=VariantEvaluationProgramGroup.class)
@DocumentedFeature
public class FindMendelianViolations
extends CommandLineProgram {
    @Argument(shortName="I", doc="Input VCF or BCF with genotypes.")
    public File INPUT;
    @Argument(shortName="PED", doc="File of Trio information in PED format (with no genotype columns).")
    public File TRIOS;
    @Argument(shortName="O", doc="Output metrics file.")
    public File OUTPUT;
    @Argument(shortName="GQ", doc="Minimum genotyping quality (or non-ref likelihood) to perform tests.")
    public int MIN_GQ = 30;
    @Argument(shortName="DP", doc="Minimum depth in each sample to consider possible mendelian violations.")
    public int MIN_DP = 0;
    @Argument(shortName="MINHET", doc="Minimum allele balance at sites that are heterozygous in the offspring.")
    public double MIN_HET_FRACTION = 0.3;
    @Argument(optional=true, doc="If provided, output per-family VCFs of mendelian violations into this directory.")
    public File VCF_DIR;
    @Argument(doc="List of chromosome names to skip entirely.")
    public Set<String> SKIP_CHROMS = CollectionUtil.makeSet((Object[])new String[]{"MT", "chrM"});
    @Argument(doc="List of possible names for male sex chromosome(s)")
    public Set<String> MALE_CHROMS = CollectionUtil.makeSet((Object[])new String[]{"Y", "chrY"});
    @Argument(doc="List of possible names for female sex chromosome(s)")
    public Set<String> FEMALE_CHROMS = CollectionUtil.makeSet((Object[])new String[]{"X", "chrX"});
    @Argument(doc="List of chr:start-end for pseudo-autosomal regions on the female sex chromosome. Defaults to HG19/b37 & HG38 coordinates.", shortName="PAR")
    public Set<String> PSEUDO_AUTOSOMAL_REGIONS = CollectionUtil.makeSet((Object[])new String[]{"X:60001-2699520", "X:154931044-155260560", "chrX:10001-2781479", "chrX:155701383-156030895"});
    @Argument(doc="The number of threads that will be used to collect the metrics. ")
    public int THREAD_COUNT = 1;
    @Argument(doc="If true then fields need to be delimited by a single tab. If false the delimiter is one or more whitespace characters. Note that tab mode does not strictly follow the PED spec")
    public boolean TAB_MODE = false;
    private final Log LOG = Log.getInstance(FindMendelianViolations.class);
    private final ProgressLogger progressLogger = new ProgressLogger(this.LOG, 10000, "variants analyzed");
    private final Lazy<VCFHeader> inputHeader = new Lazy(() -> {
        try (VCFFileReader in = new VCFFileReader(this.INPUT);){
            VCFHeader vCFHeader = in.getFileHeader();
            return vCFHeader;
        }
    });
    private final Lazy<PedFile> pedFile = new Lazy(() -> PedFile.fromFile(this.TRIOS, this.TAB_MODE));
    private final Lazy<Set<Interval>> parIntervals = new Lazy(() -> Collections.unmodifiableSet(this.parseIntervalLists(this.PSEUDO_AUTOSOMAL_REGIONS)));

    private Set<Interval> parseIntervalLists(Set<String> intervalLists) {
        HashSet<Interval> intervals = new HashSet<Interval>(intervalLists.size());
        for (String par : intervalLists) {
            String[] splits1 = par.split(":");
            String[] splits2 = splits1[1].split("-");
            intervals.add(new Interval(splits1[0], Integer.parseInt(splits2[0]), Integer.parseInt(splits2[1])));
        }
        return intervals;
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<CallSite> errors = new ArrayList<CallSite>();
        HashSet<String> sexChroms = new HashSet<String>(this.FEMALE_CHROMS);
        sexChroms.retainAll(this.MALE_CHROMS);
        if (!sexChroms.isEmpty()) {
            errors.add((CallSite)((Object)("The following chromosomes were listed as both male and female sex chromosomes: " + sexChroms)));
        }
        if (errors.isEmpty()) {
            return null;
        }
        return errors.toArray(new String[0]);
    }

    @Override
    protected int doWork() {
        boolean outputVcfs = this.VCF_DIR != null;
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsReadable((File)this.TRIOS);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        if (outputVcfs) {
            IOUtil.assertDirectoryIsWritable((File)this.VCF_DIR);
        }
        this.LOG.info(new Object[]{"Loading and filtering trios."});
        MendelianViolationDetector.Result result = VariantProcessor.Builder.generatingAccumulatorsBy(this::buildDetector).withInput(this.INPUT).combiningResultsBy(MendelianViolationDetector.Result::merge).multithreadingBy(this.THREAD_COUNT).build().process();
        MetricsFile metricsFile = this.getMetricsFile();
        for (MendelianViolationMetrics m : result.metrics()) {
            m.calculateDerivedFields();
            metricsFile.addMetric((MetricBase)m);
        }
        metricsFile.write(this.OUTPUT);
        this.writeAllViolations(result);
        return 0;
    }

    private void writeAllViolations(MendelianViolationDetector.Result result) {
        if (this.VCF_DIR != null) {
            this.LOG.info(new Object[]{String.format("Writing family violation VCFs to %s/", this.VCF_DIR.getAbsolutePath())});
            VariantContextComparator vcComparator = new VariantContextComparator((Collection)((VCFHeader)this.inputHeader.get()).getContigLines());
            LinkedHashSet<VCFInfoHeaderLine> headerLines = new LinkedHashSet<VCFInfoHeaderLine>(((VCFHeader)this.inputHeader.get()).getMetaDataInInputOrder());
            headerLines.add(new VCFInfoHeaderLine("MV", 1, VCFHeaderLineType.String, "Type of mendelian violation."));
            headerLines.add(new VCFInfoHeaderLine("AC_Orig", VCFHeaderLineCount.A, VCFHeaderLineType.Integer, "Original AC"));
            headerLines.add(new VCFInfoHeaderLine("AF_Orig", VCFHeaderLineCount.A, VCFHeaderLineType.Float, "Original AF"));
            headerLines.add(new VCFInfoHeaderLine("AN_Orig", 1, VCFHeaderLineType.Integer, "Original AN"));
            for (PedFile.PedTrio trio : ((PedFile)this.pedFile.get()).values()) {
                File outputFile = new File(this.VCF_DIR, IOUtil.makeFileNameSafe((String)(trio.getFamilyId() + ".vcf")));
                this.LOG.info(new Object[]{String.format("Writing %s violation VCF to %s", trio.getFamilyId(), outputFile.getAbsolutePath())});
                VariantContextWriter out = new VariantContextWriterBuilder().setOutputFile(outputFile).unsetOption(Options.INDEX_ON_THE_FLY).build();
                VCFHeader newHeader = new VCFHeader(headerLines, CollectionUtil.makeList((Object[])new String[]{trio.getMaternalId(), trio.getPaternalId(), trio.getIndividualId()}));
                TreeSet orderedViolations = new TreeSet(vcComparator);
                orderedViolations.addAll((Collection)result.violations().get(trio.getFamilyId()));
                out.writeHeader(newHeader);
                orderedViolations.forEach(arg_0 -> ((VariantContextWriter)out).add(arg_0));
                out.close();
            }
        }
    }

    private MendelianViolationDetector buildDetector() {
        return new MendelianViolationDetector((Set<String>)ImmutableSet.copyOf(this.SKIP_CHROMS), (Set<String>)ImmutableSet.copyOf(this.MALE_CHROMS), (Set<String>)ImmutableSet.copyOf(this.FEMALE_CHROMS), this.MIN_HET_FRACTION, this.MIN_GQ, this.MIN_DP, this.generateTrioMetricsBase(), (List<Interval>)ImmutableList.copyOf((Collection)((Collection)this.parIntervals.get())), this.progressLogger);
    }

    private List<MendelianViolationMetrics> generateTrioMetricsBase() {
        MendelianViolationMetrics m;
        ArrayList<MendelianViolationMetrics> metrics = new ArrayList<MendelianViolationMetrics>();
        for (PedFile.PedTrio trio : ((PedFile)this.pedFile.get()).values()) {
            m = new MendelianViolationMetrics();
            m.MOTHER = trio.getMaternalId();
            m.FATHER = trio.getPaternalId();
            m.OFFSPRING = trio.getIndividualId();
            m.FAMILY_ID = trio.getFamilyId();
            m.OFFSPRING_SEX = trio.getSex();
            metrics.add(m);
        }
        HashSet allSamples = new HashSet(((VCFHeader)this.inputHeader.get()).getSampleNamesInOrder());
        Iterator trioIterator = metrics.iterator();
        while (trioIterator.hasNext()) {
            m = (MendelianViolationMetrics)((Object)trioIterator.next());
            Set trio = CollectionUtil.makeSet((Object[])new String[]{m.MOTHER, m.FATHER, m.OFFSPRING});
            trio.removeAll(allSamples);
            if (trio.isEmpty()) continue;
            this.LOG.warn(new Object[]{"Removing trio due to the following missing samples in VCF: " + trio});
            trioIterator.remove();
        }
        return metrics;
    }
}

