/*
 * Decompiled with CFR 0.152.
 */
package redis.embedded;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.IntStream;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.embedded.Redis;
import redis.embedded.core.RedisShardedClusterBuilder;
import redis.embedded.error.RedisClusterSetupException;

public final class RedisShardedCluster
implements Redis {
    private static final String CLUSTER_IP = "127.0.0.1";
    private static final int MAX_NUMBER_OF_SLOTS_PER_CLUSTER = 16384;
    private static final Duration SLEEP_DURATION = Duration.ofMillis(300L);
    private static final long SLEEP_DURATION_IN_MILLIS = SLEEP_DURATION.toMillis();
    private final List<Redis> servers = new LinkedList<Redis>();
    private final Map<Integer, Set<Integer>> replicasPortsByMainNodePort = new LinkedHashMap<Integer, Set<Integer>>();
    private final Map<Integer, String> mainNodeIdsByPort = new LinkedHashMap<Integer, String>();
    private final Duration initializationTimeout;

    public RedisShardedCluster(List<Redis> servers, Map<Integer, Set<Integer>> replicasPortsByMainNodePort, Duration initializationTimeout) {
        this.servers.addAll(servers);
        this.replicasPortsByMainNodePort.putAll(replicasPortsByMainNodePort);
        this.initializationTimeout = initializationTimeout;
    }

    public static RedisShardedClusterBuilder newRedisCluster() {
        return new RedisShardedClusterBuilder();
    }

    @Override
    public boolean isActive() {
        for (Redis redis : this.servers) {
            if (redis.isActive()) continue;
            return false;
        }
        return true;
    }

    @Override
    public void start() throws IOException {
        for (Redis redis : this.servers) {
            redis.start();
        }
        this.linkReplicasAndShards();
    }

    @Override
    public void stop() throws IOException {
        for (Redis redis : this.servers) {
            redis.stop();
        }
    }

    @Override
    public List<Integer> ports() {
        return new ArrayList<Integer>(this.serverPorts());
    }

    public List<Redis> servers() {
        return new LinkedList<Redis>(this.servers);
    }

    public List<Integer> serverPorts() {
        ArrayList<Integer> ports = new ArrayList<Integer>();
        for (Redis redis : this.servers) {
            ports.addAll(redis.ports());
        }
        return ports;
    }

    public int getPort() {
        return this.ports().get(0);
    }

    private void linkReplicasAndShards() {
        try {
            Set<Integer> mainNodePorts = this.replicasPortsByMainNodePort.keySet();
            Integer clusterMeetTarget = mainNodePorts.iterator().next();
            this.meetMainNodes(clusterMeetTarget);
            this.setupReplicas(clusterMeetTarget);
            this.waitForClusterToBeInteractReady();
        }
        catch (RedisClusterSetupException e) {
            try {
                this.stop();
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
            throw new RuntimeException(e);
        }
    }

    private void meetMainNodes(Integer clusterMeetTarget) throws RedisClusterSetupException {
        LinkedList<Integer> shardsMainNodePorts = new LinkedList<Integer>(this.replicasPortsByMainNodePort.keySet());
        int slotsPerShard = 16384 / shardsMainNodePorts.size();
        for (int i = 0; i < shardsMainNodePorts.size(); ++i) {
            Integer port = (Integer)shardsMainNodePorts.get(i);
            int startSlot = i * slotsPerShard;
            int endSlot = i == shardsMainNodePorts.size() - 1 ? 16383 : startSlot + slotsPerShard - 1;
            try (Jedis jedis = new Jedis(CLUSTER_IP, port.intValue());){
                if (!port.equals(clusterMeetTarget)) {
                    jedis.clusterMeet(CLUSTER_IP, clusterMeetTarget.intValue());
                }
                String nodeId = jedis.clusterMyId();
                this.mainNodeIdsByPort.put(port, nodeId);
                jedis.clusterAddSlots(IntStream.range(startSlot, endSlot + 1).toArray());
                continue;
            }
            catch (Exception e) {
                throw new RedisClusterSetupException("Failed creating main node instance at port: " + port, e);
            }
        }
    }

    private void setupReplicas(Integer clusterMeetTarget) throws RedisClusterSetupException {
        for (Map.Entry<Integer, Set<Integer>> entry : this.replicasPortsByMainNodePort.entrySet()) {
            String mainNodeId = this.mainNodeIdsByPort.get(entry.getKey());
            Set<Integer> replicaPorts = entry.getValue();
            for (Integer replicaPort : replicaPorts) {
                try (Jedis jedis = new Jedis(CLUSTER_IP, replicaPort.intValue());){
                    jedis.clusterMeet(CLUSTER_IP, clusterMeetTarget.intValue());
                    this.waitForNodeToAppearInCluster(jedis, mainNodeId);
                    jedis.clusterReplicate(mainNodeId);
                    this.waitForClusterToHaveStatusOK(jedis);
                }
                catch (Exception e) {
                    throw new RedisClusterSetupException("Failed adding replica instance at port: " + replicaPort, e);
                }
            }
        }
    }

    private void waitForNodeToAppearInCluster(Jedis jedis, String nodeId) throws RedisClusterSetupException {
        boolean nodeReady = this.waitForPredicateToPass(() -> jedis.clusterNodes().contains(nodeId));
        if (!nodeReady) {
            throw new RedisClusterSetupException("Node was not ready before timeout");
        }
    }

    private void waitForClusterToHaveStatusOK(Jedis jedis) throws RedisClusterSetupException {
        boolean clusterIsReady = this.waitForPredicateToPass(() -> jedis.clusterInfo().contains("cluster_state:ok"));
        if (!clusterIsReady) {
            throw new RedisClusterSetupException("Cluster did not have status OK before timeout");
        }
    }

    private void waitForClusterToBeInteractReady() throws RedisClusterSetupException {
        boolean clusterIsReady = this.waitForPredicateToPass(() -> {
            Boolean bl;
            JedisCluster jc = new JedisCluster(new HostAndPort(CLUSTER_IP, this.getPort()));
            try {
                jc.get("someKey");
                bl = true;
            }
            catch (Throwable throwable) {
                try {
                    try {
                        jc.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    return false;
                }
            }
            jc.close();
            return bl;
        });
        if (!clusterIsReady) {
            throw new RedisClusterSetupException("Cluster was not stable before timeout");
        }
    }

    private boolean waitForPredicateToPass(Supplier<Boolean> predicate) throws RedisClusterSetupException {
        long maxWaitInMillis = this.initializationTimeout.toMillis();
        int waited = 0;
        boolean result = predicate.get();
        while (!result && (long)waited < maxWaitInMillis) {
            try {
                Thread.sleep(SLEEP_DURATION_IN_MILLIS);
            }
            catch (InterruptedException e) {
                throw new RedisClusterSetupException("Interrupted while waiting", e);
            }
            waited = (int)((long)waited + SLEEP_DURATION_IN_MILLIS);
            result = predicate.get();
        }
        return result;
    }
}

