package simplexml;

import simplexml.model.*;
import simplexml.utils.Interfaces.AccessDeserializers;

import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.*;

import static simplexml.utils.Constants.*;
import static simplexml.utils.Reflection.newObject;
import static simplexml.utils.Reflection.toName;
import static simplexml.utils.XML.unescapeXml;

public interface XmlReader extends AccessDeserializers {

    default <T> T domToObject(final Element node, final Class<T> clazz) {
        if (node == null) return null;
        final ObjectDeserializer c = getDeserializer(clazz);
        if (c != null) return c.convert(node.text, clazz);

        try {
            final T o = newObject(clazz);

            for (final Field f : clazz.getDeclaredFields()) {
                f.setAccessible(true);

                if (f.isAnnotationPresent(XmlTextNode.class)) {
                    f.set(o, textNodeToValue(f.getType(), node));
                    continue;
                }

                final String name = toName(f);
                if (f.isAnnotationPresent(XmlAttribute.class)) {
                    f.set(o, attributeToValue(f.getType(), name, node));
                    continue;
                }

                final Class<?> type = f.getType();
                if (Set.class.isAssignableFrom(type)) {
                    f.set(o, domToSet(getClassOfCollection(f), name, node));
                    continue;
                }
                if (List.class.isAssignableFrom(type)) {
                    f.set(o, domToList(getClassOfCollection(f), name, node));
                    continue;
                }
                if (type.isArray()) {
                    f.set(o, domToArray(f.getType().getComponentType(), name, node));
                    continue;
                }

                if (Map.class.isAssignableFrom(type)) {
                    f.set(o, domToMap((ParameterizedType) f.getGenericType(), name, node));
                    continue;
                }

                final String value = node.attributes.get(name);
                if (value != null) {
                    f.set(o, stringToValue(f.getType(), value));
                    continue;
                }

                f.set(o, domToObject(findChildForName(name, node), f.getType()));
            }

            return o;
        } catch ( IllegalAccessException | SecurityException | IllegalArgumentException e) {
            return null;
        }
    }


    default Object textNodeToValue(final Class<?> type, final Element node) throws IllegalAccessException {
        final ObjectDeserializer conv = getDeserializer(type);
        return (conv != null) ? conv.convert(node.text) : null;
    }
    default Object attributeToValue(final Class<?> type, final String name, final Element node) throws IllegalAccessException {
        final ObjectDeserializer conv = getDeserializer(type);
        if (conv == null) return null;
        final String value = node.attributes.get(name);
        if (value == null) return null;
        return conv.convert(value);
    }
    default Object stringToValue(final Class<?> type, final String value) {
        final ObjectDeserializer conv = getDeserializer(type);
        return (conv != null) ? conv.convert(value) : null;
    }
    default Set<Object> domToSet(final Class<?> type, final String name, final Element node) throws IllegalAccessException {
        final ObjectDeserializer elementConv = getDeserializer(type);

        final Set<Object> set = new HashSet<>();
        for (final Element n : node.children) {
            if (!n.name.equals(name)) continue;

            set.add( (elementConv == null) ? domToObject(n, type) : elementConv.convert(n.text));
        }
        return set;
    }
    default List<Object> domToList(final Class<?> type, final String name, final Element node) throws IllegalAccessException {
        final ObjectDeserializer elementConv = getDeserializer(type);

        final List<Object> list = new LinkedList<>();
        for (final Element n : node.children) {
            if (!n.name.equals(name)) continue;

            list.add( (elementConv == null) ? domToObject(n, type) : elementConv.convert(n.text));
        }
        return list;
    }
    default Object[] domToArray(final Class<?> type, final String name, final Element node) throws IllegalAccessException {
        final ObjectDeserializer elementConv = getDeserializer(type);

        final Object[] array = (Object[]) Array.newInstance(type, numChildrenWithName(name, node));
        int i = 0;
        for (final Element n : node.children) {
            if (n.name.equals(name)) {
                array[i] = (elementConv == null) ? domToObject(n, type) : elementConv.convert(n.text, type);
                i++;
            }
        }
        return array;
    }
    default Map<Object, Object> domToMap(final ParameterizedType type, final String name, final Element node) throws IllegalAccessException {
        final Element element = findChildForName(name, node);
        if (element == null) return null;

        final ObjectDeserializer convKey = getDeserializer((Class<?>)type.getActualTypeArguments()[0]);
        final ObjectDeserializer convVal = getDeserializer((Class<?>)type.getActualTypeArguments()[1]);

        final Map<Object, Object> map = new HashMap<>();
        for (final Element child : element.children) {
            map.put(convKey.convert(child.name), convVal.convert(child.text));
        }
        return map;
    }

    static Element findChildForName(final String name, final Element node) {
        for (final Element child : node.children) {
            if (name.equals(child.name))
                return child;
        }
        return null;
    }

    static int numChildrenWithName(final String name, final Element node) {
        int num = 0;
        for (final Element child : node.children) {
            if (name.equals(child.name)) num++;
        }
        return num;
    }

    static Class<?> getClassOfCollection(final Field f) {
        final ParameterizedType stringListType = (ParameterizedType) f.getGenericType();
        return (Class<?>) stringListType.getActualTypeArguments()[0];
    }

    static Element parseXML(final InputStreamReader in) throws IOException {
        final EventParser p = new EventParser();

        String str;
        while ((str = readLine(in, XML_TAG_START)) != null) {
            if (!str.isEmpty()) p.someText(unescapeXml(str.trim()));

            str = readLine(in, XML_TAG_END).trim();
            if (str.charAt(0) == XML_PROLOG) continue;

            if (str.charAt(0) == XML_SELF_CLOSING) p.endNode();
            else {
                final String name = getNameOfTag(str);
                if (str.length() == name.length()) {
                    p.startNode(str, new HashMap<>());
                    continue;
                }

                final int beginAttr = name.length();
                final int end = str.length();
                if (str.endsWith(FORWARD_SLASH)) {
                    p.startNode(name, parseAttributes(str.substring(beginAttr, end-1)));
                    p.endNode();
                } else {
                    p.startNode(name, parseAttributes(str.substring(beginAttr+1, end)));
                }
            }
        }

        return p.getRoot();
    }

    static String readLine(final InputStreamReader in, final char end) throws IOException {
        final List<Character> chars = new LinkedList<>();
        int data;
        while ((data = in.read()) != -1) {
            if (data == end) break;
            chars.add((char) data);
        }
        if (data == -1) return null;

        char[] value = new char[chars.size()];
        int i = 0;
        for (final Character c : chars) value[i++] = c;
        return new String(value);
    }

    static String getNameOfTag(final String tag) {
        int offset = 0;
        for (; offset < tag.length(); offset++) {
            if (tag.charAt(offset) == CHAR_SPACE || tag.charAt(offset) == CHAR_FORWARD_SLASH)
                break;
        }
        return tag.substring(0, offset);
    }

    static HashMap<String, String> parseAttributes(String input) {
        final HashMap<String, String> attributes = new HashMap<>();

        while (!input.isEmpty()) {
            int startName = indexOfNonWhitespaceChar(input, 0);
            if (startName == -1) break;
            int equals = input.indexOf(CHAR_EQUALS, startName+1);
            if (equals == -1) break;

            final String name = input.substring(startName, equals).trim();
            input = input.substring(equals+1);

            int startValue = indexOfNonWhitespaceChar(input, 0);
            if (startValue == -1) break;

            int endValue; final String value;
            if (input.charAt(startValue) == CHAR_DOUBLE_QUOTE) {
                startValue++;
                endValue = input.indexOf(CHAR_DOUBLE_QUOTE, startValue);
                if (endValue == -1) endValue = input.length()-1;
                value = input.substring(startValue, endValue).trim();
            } else {
                endValue = indexOfWhitespaceChar(input, startValue+1);
                if (endValue == -1) endValue = input.length()-1;
                value = input.substring(startValue, endValue+1).trim();
            }

            input = input.substring(endValue+1);

            attributes.put(name, unescapeXml(value));
        }

        return attributes;
    }

    static int indexOfNonWhitespaceChar(final String input, final int offset) {
        for (int i = offset; i < input.length(); i++) {
            final char at = input.charAt(i);
            if (at == ' ' || at == '\t' || at == '\n' || at == '\r') continue;
            return i;
        }
        return -1;
    }
    static int indexOfWhitespaceChar(final String input, final int offset) {
        for (int i = offset; i < input.length(); i++) {
            final char at = input.charAt(i);
            if (at == ' ' || at == '\t' || at == '\n' || at == '\r') return i;
        }
        return -1;
    }
}
