/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import com.sun.codemodel.ClassWriter;
import com.sun.codemodel.JDefinedClass;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.PackageElement;
import javax.tools.FileObject;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Generates source files for {@link Proxiable} types
 */
class Generator {
    private static final Logger LOGGER = Logger.getLogger(SmallproxProcessor.LOGGER_NAME);
    private final ProcessingEnvironment env;
    private final ProxyFactory proxyFactory;
    private OutputPackageProvider outputPackageProvider;

    public Generator(ProcessingEnvironment env, Map<String, Proxiable> proxiables) {
        this.env = env;
        if(env.getOptions().containsKey(SmallproxProcessor.OPTION_ENTITY_OUTPUT_PACKAGE_PROVIDER)) {
            String packageProviderName = env.getOptions().get(SmallproxProcessor.OPTION_ENTITY_OUTPUT_PACKAGE_PROVIDER);
            LOGGER.log(Level.INFO, "OutputPackageProvider specified: {0}", packageProviderName);
            try {
                this.outputPackageProvider = (OutputPackageProvider) Class.forName(packageProviderName).newInstance();
            } catch (Exception ex) {
                LOGGER.log(Level.SEVERE, "Couldn't instantiate OutputPackageProvider " + packageProviderName, ex);
                throw new RuntimeException(ex);
            }
        } else {
            this.outputPackageProvider = new SimpleOutputPackageProvider();
        }
        this.proxyFactory = new ProxyFactory(outputPackageProvider, proxiables);
    }

    public void generate(Proxiable entity) throws IOException {
        JDefinedClass proxyClass = proxyFactory.createProxy(entity);
        JavaFileObject file = createSourceFile(proxyClass, entity);
        write(proxyClass, file);
    }

    private JavaFileObject createSourceFile(JDefinedClass proxyClass, Proxiable entity) throws IOException {
        PackageElement packageElement = (PackageElement) entity.getElement().getEnclosingElement();
        String packageName = outputPackageProvider.outputPackageFor(packageElement.getQualifiedName().toString());
        String className = packageName + "." + proxyClass.name();
        return env.getFiler().createSourceFile(className, entity.getElement());
    }

    private void write(JDefinedClass proxyClass, FileObject destination) throws IOException {
        ClassWriter classWriter = new ClassWriter(destination.openWriter());
        try {
            classWriter.write(proxyClass);
        } finally {
            classWriter.close();
        }
    }
}
