/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.tools.Diagnostic;
import javax.tools.Diagnostic.Kind;
import java.util.Arrays;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

/**
 * Logs java.util.logging messages with the compiler.
 */
class MessagerAppender extends Handler {

    private static final SimpleFormatter FORMATTER = new SimpleFormatter();

    static MessagerAppender forEnvironment(ProcessingEnvironment env) {
        return new MessagerAppender(env);
    }
    private final Messager messager;

    private MessagerAppender(ProcessingEnvironment env) {
        this.messager = env.getMessager();
    }

    @Override
    public void publish(LogRecord record) {
        Diagnostic.Kind messageKind = kindOfMessage(record);
        if (messageKind != null) {
            Object[] parameters = record.getParameters();
            String messageTemplate = record.getMessage();
            if (parameters == null || parameters.length == 0) {
                messager.printMessage(messageKind, messageTemplate);
            } else {
                if (parameters[parameters.length - 1] instanceof Element) {
                    Element element = (Element) parameters[parameters.length - 1];
                    if (parameters.length == 1) {
                        messager.printMessage(messageKind, messageTemplate, element);
                    } else {
                        Object[] otherParams = Arrays.copyOfRange(parameters, 0, parameters.length - 1);
                        record.setParameters(otherParams);
                        String message = FORMATTER.formatMessage(record);
                        messager.printMessage(messageKind, message, element);
                    }
                } else {
                    String message = FORMATTER.formatMessage(record);
                    messager.printMessage(messageKind, message);
                }
            }
            if(record.getThrown() != null) {
                Throwable error = record.getThrown();
                messager.printMessage(Kind.ERROR, error.toString());
                
                for (StackTraceElement stackTraceElement : error.getStackTrace()) {
                    messager.printMessage(Kind.NOTE, stackTraceElement.toString());
                }
            }
        }
    }

    private Kind kindOfMessage(LogRecord record) {
        Level level = record.getLevel();
        if (atLeast(level, Level.SEVERE)) {
            return Kind.ERROR;
        } else if (atLeast(level, Level.WARNING)) {
            return Kind.MANDATORY_WARNING;
        } else if (atLeast(level, Level.INFO)) {
            return Kind.NOTE;
        }
        return null;
    }

    private boolean atLeast(Level a, Level b) {
        return a.intValue() >= b.intValue();
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() throws SecurityException {
    }
}
