/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import javax.lang.model.element.TypeElement;
import javax.persistence.Entity;
import javax.persistence.MappedSuperclass;
import java.util.*;

/**
 * Represents a proxiable type: either an entity or an embeddable type
 */
class Proxiable {

    private final String className;
    private final List<Method> methods = new LinkedList<Method>();
    private Proxiable parent;
    private final TypeElement element;

    public Proxiable(TypeElement e) {
        element = e;
        className = e.getQualifiedName().toString();
    }

    public TypeElement getElement() {
        return element;
    }

    public boolean isEntity() {
        return element.getAnnotation(Entity.class) != null
                || element.getAnnotation(MappedSuperclass.class) != null;
    }

    public static class Method {

        public static Builder named(String methodName) {
            return new Builder(methodName);
        }
        private String name;
        private Type returnType;
        private final List<Parameter> parameters;

        public Method(String name, Type returnType, List<Parameter> parameters) {
            this.name = name;
            this.returnType = returnType;
            this.parameters = new ArrayList<Parameter>(parameters);
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Type getReturnType() {
            return returnType;
        }

        public void setReturnType(Type returnType) {
            this.returnType = returnType;
        }

        public List<Parameter> getParameters() {
            return Collections.unmodifiableList(parameters);
        }

        public static class Builder {

            private final String name;
            private final List<Parameter> params = new LinkedList<Parameter>();
            private Type returnType;

            private Builder(String name) {
                this.name = name;
            }

            public Builder returning(Type returnType) {
                this.returnType = returnType;
                return this;
            }

            public Builder addParam(Type type, String name) {
                params.add(new Parameter(type, name));
                return this;
            }

            public Method build() {
                assert name != null && returnType != null;
                return new Method(name, returnType, params);
            }
        }

        public static class Parameter {

            private final Type type;
            private final String name;

            public Parameter(Type type, String name) {
                this.type = type;
                this.name = name;
            }

            public String getName() {
                return name;
            }

            public Type getType() {
                return type;
            }
        }
    }

    public static class Type {

        private final String name;
        private final List<Type> parameters = new LinkedList<Type>();

        public Type(String name, Type... parameters) {
            this.name = name;
            this.parameters.addAll(Arrays.asList(parameters));
        }

        public String getName() {
            return name;
        }

        public List<Type> getParameters() {
            return Collections.unmodifiableList(parameters);
        }
        
        public void addParameter(Type parameter) {
            parameters.add(parameter);
        }
    }

    public String getClassName() {
        return className;
    }

    public List<Method> getMethods() {
        return Collections.unmodifiableList(methods);
    }

    public void addMethod(Method method) {
        methods.add(method);
    }

    public Proxiable getParent() {
        return parent;
    }

    public void setParent(Proxiable parent) {
        this.parent = parent;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Proxiable other = (Proxiable) obj;
        return !((this.className == null) ? (other.className != null) : !this.className.equals(other.className));
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 23 * hash + (this.className != null ? this.className.hashCode() : 0);
        return hash;
    }

    @Override
    public String toString() {
        return String.format("Proxiable[%s]", className);
    }
}
