/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import com.github.danielbell.smallprox.Proxiable.Method;
import com.github.danielbell.smallprox.Proxiable.Type;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.*;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeKind;
import javax.lang.model.type.TypeMirror;
import javax.persistence.Embeddable;
import javax.persistence.Entity;
import javax.persistence.Transient;
import javax.tools.Diagnostic.Kind;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Collects all proxiable types
 */
class ProxiableVisitor extends ElementVisitorAdapter<Void, Proxiable> {

    private final Map<String, Proxiable> proxiables = new HashMap<String, Proxiable>();
    private final ProcessingEnvironment env;

    public ProxiableVisitor(ProcessingEnvironment env) {
        this.env = env;
    }

    public Map<String, Proxiable> getProxiables() {
        return Collections.unmodifiableMap(proxiables);
    }

    @Override
    public Void visitType(TypeElement typeElement, Proxiable child) {
        boolean hasProxiableChild = child != null;
        boolean isOrSuperclassesEntity = isEntityOrEmbeddable(typeElement) || hasProxiableChild;
        if (isOrSuperclassesEntity && isProxiable(typeElement)) {
            Proxiable proxiable = new Proxiable(typeElement);
            proxiables.put(proxiable.getClassName(), proxiable);
            if (hasProxiableChild) {
                child.setParent(proxiable);
            }
            for (Element element : typeElement.getEnclosedElements()) {
                element.accept(this, proxiable);
            }
            //Model the superclasses too
            Element superclass = asElement(typeElement.getSuperclass());
            superclass.accept(this, proxiable);
        }
        return null;
    }

    @Override
    public Void visitExecutable(ExecutableElement element, Proxiable enclosingEntity) {
        if (isNonstaticMethod(element)) {
            if (refersToTransientProperty(enclosingEntity.getElement(), element)) {
                return null;
            }

            String methodName = element.getSimpleName().toString();
            TypeMirror returnTypeMirror = element.getReturnType();
            Type returnType = asProxiableType(returnTypeMirror);
            Method.Builder method = Proxiable.Method.named(methodName).returning(returnType);

            List<? extends VariableElement> parameterElements = element.getParameters();
            for (VariableElement paramElement : parameterElements) {
                TypeMirror paramTypeMirror = paramElement.asType();
                Proxiable.Type paramType = asProxiableType(paramTypeMirror);
                String paramName = paramElement.getSimpleName().toString();
                method.addParam(paramType, paramName);
            }
            enclosingEntity.addMethod(method.build());
        }
        return null;
    }

    private static boolean isNonstaticMethod(ExecutableElement element) {
        return element.getKind() == ElementKind.METHOD && !element.getModifiers().contains(Modifier.STATIC);
    }

    private Proxiable.Type asProxiableType(TypeMirror typeMirror) {
        String typeName = getTypeName(typeMirror);
        //TODO: accommodate embedded generic return types: e.g. List<List<MyOtherProxy>>
        Proxiable.Type type = new Proxiable.Type(typeName);
        if (typeMirror.getKind() == TypeKind.DECLARED) {
            DeclaredType declared = (DeclaredType) typeMirror;
            List<? extends TypeMirror> typeArguments = declared.getTypeArguments();
            for (TypeMirror typeArgumentMirror : typeArguments) {
                Proxiable.Type arg = asProxiableType(typeArgumentMirror);
                type.addParameter(arg);
            }
        }
        return type;
    }

    private void log(String message, Element e) {
        env.getMessager().printMessage(Kind.WARNING, message, e);
    }

    private boolean isProxiable(TypeElement element) {
        boolean isNull = element == null;
        boolean isObject = element.getSuperclass().getKind() == TypeKind.NONE;
        return !(isNull || isObject);
    }

    private Element asElement(TypeMirror typeMirror) {
        return env.getTypeUtils().asElement(typeMirror);
    }

    private boolean isEntityOrEmbeddable(TypeElement typeElement) {
        return typeElement.getAnnotation(Entity.class) != null
                || typeElement.getAnnotation(Embeddable.class) != null;
    }

    private String getTypeName(TypeMirror returnType) {
        if (returnType.getKind() == TypeKind.DECLARED) {
            DeclaredType declaredType = (DeclaredType) returnType;
            TypeElement typeElement = (TypeElement) declaredType.asElement();
            Name binaryName = env.getElementUtils().getBinaryName(typeElement);
            return binaryName.toString();
        }
        return returnType.toString();
    }

    private boolean refersToTransientProperty(TypeElement enclosingElement, ExecutableElement methodElement) {
        if (methodElement.getAnnotation(Transient.class) != null) {
            return true;
        }

        String methodName = methodElement.getSimpleName().toString();
        JavaBeanMethod beanMethod = JavaBeanMethod.named(methodName);
        if (beanMethod.isValidJavaBeanMethod()) {
            Element field = getField(enclosingElement, beanMethod.getFieldName());
            if (field != null && field.getAnnotation(Transient.class) != null) {
                return true;
            }
            if (beanMethod.isSetter()) {
                final String fieldName = beanMethod.getFieldName();
                String getterName = "get" + Character.valueOf(fieldName.charAt(0)).toString().toUpperCase() + fieldName.substring(1);
                List<? extends Element> enclosedElements = enclosingElement.getEnclosedElements();
                for (Element enclosed : enclosedElements) {
                    if (enclosed.getKind() == ElementKind.METHOD && enclosed.getSimpleName().toString().equals(getterName)) {
                        if (enclosed.getAnnotation(Transient.class) != null) {
                            return true;
                        } else {
                            return false;
                        }
                    }
                }
            }
        }
        return false;
    }

    private Element getField(TypeElement type, String fieldName) {
        for (Element element : type.getEnclosedElements()) {
            if (element.getKind() == ElementKind.FIELD) {
                if (element.getSimpleName().toString().equals(fieldName)) {
                    return element;
                }
            }
        }
        return null;
    }
}
