/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import com.github.danielbell.smallprox.Proxiable.Type;
import com.sun.codemodel.*;

import javax.lang.model.element.PackageElement;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Creates CodeModel classes from {@link Proxiable} types.
 */
class ProxyFactory {

    private static final List<String> PRIMITIVES =
            Arrays.asList(void.class.getName(),
            char.class.getName(),
            byte.class.getName(),
            short.class.getName(),
            int.class.getName(),
            long.class.getName(),
            float.class.getName(),
            double.class.getName(),
            String.class.getName(),
            List.class.getName(),
            Set.class.getName(),
            Map.class.getName());
    private static final String PROXY_SUFFIX = "Proxy";
    private static final Logger LOGGER = Logger.getLogger(SmallproxProcessor.LOGGER_NAME);
    private final OutputPackageProvider outputPackageProvider;
    private final Map<String, Proxiable> proxiables;

    ProxyFactory(OutputPackageProvider outputPackageProvider, Map<String, Proxiable> proxiables) {
        this.outputPackageProvider = outputPackageProvider;
        this.proxiables = proxiables;
    }

    /**
     * Create a GWT RequestFactory proxy for a proxiable type
     *
     * @param proxiable the type for which to create the proxy
     * @return the newly created proxy
     */
    public JDefinedClass createProxy(Proxiable proxiable) {
        try {
            String proxyFqn = getProxifiedName(proxiable);
            //TODO: use the same model for all classes?
            JCodeModel cm = new JCodeModel();

            JDefinedClass proxyClass = cm._class(proxyFqn, ClassType.INTERFACE);

            annotateWithProxyAnnotation:
            {
                JClass proxyAnnotationType = cm.ref(GwtTypes.PROXY_FOR_TYPE);
                JClass proxiableType = cm.ref(proxiable.getClassName());
                JAnnotationUse proxyAnnotation = proxyClass.annotate(proxyAnnotationType);
                proxyAnnotation.param("value", proxiableType);
            }

            extendParentType:
            {
                Proxiable parent = proxiable.getParent();
                JClass parentClass;
                if (parent == null) {
                    if (proxiable.isEntity()) {
                        parentClass = cm.ref(GwtTypes.ENTITY_PROXY_TYPE);
                    } else {
                        //It's not an entity, so it should extend ValueProxy
                        parentClass = cm.ref(GwtTypes.VALUE_PROXY_TYPE);
                    }
                } else {
                    parentClass = cm._class(getProxifiedName(parent), ClassType.INTERFACE);
                }
                proxyClass._extends(parentClass);
            }

            createMethods:
            for (Proxiable.Method method : proxiable.getMethods()) {
                String methodName = method.getName();
                Type sourceReturnType = method.getReturnType();
                JClass returnType = asGenericCodeModelType(sourceReturnType, cm);
                JMethod proxyMethod = proxyClass.method(JMod.NONE, returnType, methodName);
                for (Proxiable.Method.Parameter parameter : method.getParameters()) {
                    Type sourceParamType = parameter.getType();
                    JClass paramType = asGenericCodeModelType(sourceParamType, cm);
                    String paramName = parameter.getName();
                    proxyMethod.param(paramType, paramName);
                }
            }
            return proxyClass;
        } catch (JClassAlreadyExistsException ex) {
            //Won't happen: we create a new JCodeModel for each class
            //TODO: should we use a shared JCodeModel?
            LOGGER.log(Level.WARNING, "Tried to create a class that was already created: {0}", new Object[]{ex.getMessage(), proxiable.getElement()});
            throw new AssertionError(ex);
        }
    }

    private JClass asGenericCodeModelType(Type type, JCodeModel cm) {
        String typeName = getMaybeProxyName(type.getName());
        JClass codeModelType = cm.ref(typeName);
        //TODO: accommodate embedded type args: e.g. List<List<MyProxy>>
        List<Type> typeParams = type.getParameters();
        for (Proxiable.Type typeArgument : typeParams) {
            JClass typeArg = asGenericCodeModelType(typeArgument, cm);
            codeModelType = codeModelType.narrow(typeArg);
        }
        return codeModelType;
    }

    private String getMaybeProxyName(String fqn) {
        if (!PRIMITIVES.contains(fqn) && proxiables.containsKey(fqn)) {
            return getProxifiedName(fqn);
        } else {
            return fqn;
        }
    }
    
    private String getProxifiedName(Proxiable entity) {
        String inputPackageName = ((PackageElement) entity.getElement().getEnclosingElement()).getQualifiedName().toString();
        String simpleName = entity.getElement().getSimpleName().toString();
        return getProxifiedName(inputPackageName, simpleName);
    }

    private String getProxifiedName(String className) {
        //TODO: this fails for inner classes. Does that matter?
        int lastDotIndex = className.lastIndexOf('.');
        if(lastDotIndex == -1) {
            return className + PROXY_SUFFIX;
        } else {
            String packageName = className.substring(0, lastDotIndex);
            String simpleName = className.substring(lastDotIndex + 1);
            return getProxifiedName(packageName, simpleName);
        }
    }
    
    private String getProxifiedName(String inputPackageName, String simpleName) {
        String outputPackageName = outputPackageProvider.outputPackageFor(inputPackageName);
        return outputPackageName + "." + simpleName + PROXY_SUFFIX;
    }
}
