/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.github.danielbell.smallprox;

import javax.annotation.processing.*;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;
import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Annotation processor that generates GWT RequestFactory proxies from a JPA
 * metamodel
 */
@SupportedAnnotationTypes({"javax.persistence.Entity", "javax.persistence.Embeddable"})
@SupportedSourceVersion(SourceVersion.RELEASE_6)
@SupportedOptions({SmallproxProcessor.OPTION_ENTITY_OUTPUT_PACKAGE_PROVIDER})
public class SmallproxProcessor extends AbstractProcessor {

    public static final String LOGGER_NAME = "Smallprox";
    private static final Logger LOGGER = Logger.getLogger(LOGGER_NAME);
    private static final String OPTION_PREFIX = "smallprox.";
    static final String OPTION_ENTITY_OUTPUT_PACKAGE_PROVIDER = OPTION_PREFIX + "outputPackageProvider";
    private static final boolean CLAIM_ANNOTATIONS = false;
    private ProcessingEnvironment env;

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        this.env = processingEnv;
        LOGGER.addHandler(MessagerAppender.forEnvironment(env));
        LOGGER.info("Initialising GWT RequestFactory proxy generator...");
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        Map<String, Proxiable> entities = getProxiables(roundEnv);
        generateProxySources(entities);

        return CLAIM_ANNOTATIONS;
    }

    private Map<String, Proxiable> getProxiables(RoundEnvironment roundEnv) {
        ProxiableVisitor entityVisitor = new ProxiableVisitor(env);
        Set<? extends Element> rootElements = roundEnv.getRootElements();
        for (Element root : rootElements) {
            root.accept(entityVisitor, null);
        }
        return entityVisitor.getProxiables();
    }

    private void generateProxySources(Map<String, Proxiable> proxiables) {
        Generator generator = new Generator(env, proxiables);
        for (Proxiable proxiable : proxiables.values()) {
            LOGGER.log(Level.INFO, "Generating GWT RequestFactory proxy for JPA type: {0}", new Object[]{proxiable, proxiable.getElement()});
            try {
                generator.generate(proxiable);
            } catch (IOException ex) {
                LOGGER.log(Level.SEVERE, "Could not create GWT RequestFactory proxy for source file: {0}", new Object[]{ex.getMessage(), proxiable.getElement()});
            }
        }
    }
}
