/**
 * Copyright (C) 2011 Daniel Bell <daniel.r.bell@gmail.com>
 *
 * This file is part of Smallprox.
 *
 * Smallprox is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Smallprox is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Smallprox.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.codemodel;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;

/**
 * Writes single classes to a specified {@link Writer}
 *
 * @see #write
 * @see #render
 */
public class ClassWriter {

    private final JFormatter formatter;
    private final Writer writer;

    /**
     * @param writer the writer to write all types to
     */
    public ClassWriter(Writer writer) {
        this.formatter = new JFormatter(writer);
        this.writer = writer;
    }

    /**
     * Write the specified type to the {@link Writer}
     *
     * @param type the type to write to the writer
     */
    public void write(JDefinedClass type) {
        formatter.write(type);
    }

    /**
     * Close the underlying writer
     *
     * @throws IOException if the writer throws an IOException while closing
     */
    public void close() throws IOException {
        writer.close();
    }

    /**
     * Render the specified type as a string
     *
     * @param type the type to render
     * @return the compilation unit, as a string
     */
    public static String render(JDefinedClass type) {
        Writer out = new StringWriter();
        new ClassWriter(out).write(type);
        return out.toString();
    }
}
