/*
 * Decompiled with CFR 0.152.
 */
package com.github.davidmoten.aws.maven;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.cloudformation.AmazonCloudFormation;
import com.amazonaws.services.cloudformation.AmazonCloudFormationClientBuilder;
import com.amazonaws.services.cloudformation.model.AmazonCloudFormationException;
import com.amazonaws.services.cloudformation.model.Capability;
import com.amazonaws.services.cloudformation.model.CreateStackRequest;
import com.amazonaws.services.cloudformation.model.DeleteStackRequest;
import com.amazonaws.services.cloudformation.model.DescribeStackEventsRequest;
import com.amazonaws.services.cloudformation.model.DescribeStackEventsResult;
import com.amazonaws.services.cloudformation.model.DescribeStacksRequest;
import com.amazonaws.services.cloudformation.model.ListStacksResult;
import com.amazonaws.services.cloudformation.model.Parameter;
import com.amazonaws.services.cloudformation.model.Stack;
import com.amazonaws.services.cloudformation.model.StackStatus;
import com.amazonaws.services.cloudformation.model.UpdateStackRequest;
import com.github.davidmoten.aws.maven.AwsKeyPair;
import com.github.davidmoten.aws.maven.Proxy;
import com.github.davidmoten.aws.maven.Util;
import com.google.common.base.Preconditions;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.maven.plugin.logging.Log;

final class CloudFormationDeployer {
    private final Log log;

    CloudFormationDeployer(Log log) {
        this.log = log;
    }

    public void deploy(AwsKeyPair keyPair, String region, String stackName, String templateBody, Map<String, String> parameters, int intervalSeconds, Proxy proxy) {
        boolean exists;
        long startTime = System.currentTimeMillis();
        Preconditions.checkArgument((intervalSeconds > 0 ? 1 : 0) != 0, (Object)"intervalSeconds must be greater than 0");
        Preconditions.checkArgument((intervalSeconds <= 600 ? 1 : 0) != 0, (Object)"intervalSeconds must be less than or equal to 600");
        AWSStaticCredentialsProvider credentials = new AWSStaticCredentialsProvider((AWSCredentials)new BasicAWSCredentials(keyPair.key, keyPair.secret));
        ClientConfiguration cc = Util.createConfiguration(proxy);
        AmazonCloudFormation cf = (AmazonCloudFormation)((AmazonCloudFormationClientBuilder)((AmazonCloudFormationClientBuilder)((AmazonCloudFormationClientBuilder)AmazonCloudFormationClientBuilder.standard().withCredentials((AWSCredentialsProvider)credentials)).withClientConfiguration(cc)).withRegion(region)).build();
        List<Parameter> params = this.buildParameters(parameters);
        this.displayStatusHistory(stackName, cf);
        int statusPollingIntervalMs = intervalSeconds * 1000;
        this.deleteFailedCreate(stackName, cf, statusPollingIntervalMs);
        try {
            exists = !cf.describeStacks(new DescribeStacksRequest().withStackName(stackName)).getStacks().isEmpty();
        }
        catch (AmazonCloudFormationException e) {
            exists = false;
        }
        if (!exists) {
            cf.createStack(new CreateStackRequest().withStackName(stackName).withTemplateBody(templateBody).withParameters(params).withCapabilities(new Capability[]{Capability.CAPABILITY_IAM}).withCapabilities(new Capability[]{Capability.CAPABILITY_NAMED_IAM}));
            this.log.info((CharSequence)"sent createStack command");
        } else {
            try {
                cf.updateStack(new UpdateStackRequest().withStackName(stackName).withTemplateBody(templateBody).withParameters(params).withCapabilities(new Capability[]{Capability.CAPABILITY_IAM}).withCapabilities(new Capability[]{Capability.CAPABILITY_NAMED_IAM}));
                this.log.info((CharSequence)"sent updateStack command");
            }
            catch (RuntimeException e) {
                this.log.info((CharSequence)e.getMessage());
                if (e.getMessage() != null && e.getMessage().contains("ValidationError") && e.getMessage().contains("No updates are to be performed")) {
                    return;
                }
                throw e;
            }
        }
        this.log.info((CharSequence)"");
        Status result = CloudFormationDeployer.waitForCompletion(cf, stackName, statusPollingIntervalMs, this.log);
        this.displayEvents(stackName, cf, startTime);
        if (!result.value.equals(StackStatus.CREATE_COMPLETE.toString()) && !result.value.equals(StackStatus.UPDATE_COMPLETE.toString())) {
            throw new RuntimeException("create/update failed: " + result);
        }
    }

    private void deleteFailedCreate(String stackName, AmazonCloudFormation cf, int statusPollingIntervalMs) {
        ListStacksResult r = cf.listStacks();
        r.getStackSummaries().stream().filter(x -> x.getStackName().equals(stackName)).limit(1L).filter(x -> StackStatus.ROLLBACK_COMPLETE.toString().equals(x.getStackStatus())).forEach(x -> {
            this.log.info((CharSequence)("Deleting stack with status " + x.getStackStatus()));
            cf.deleteStack(new DeleteStackRequest().withStackName(stackName));
            CloudFormationDeployer.waitForCompletion(cf, stackName, statusPollingIntervalMs, this.log);
        });
    }

    private List<Parameter> buildParameters(Map<String, String> parameters) {
        List<Object> params = parameters != null ? parameters.entrySet().stream().map(p -> new Parameter().withParameterKey((String)p.getKey()).withParameterValue((String)p.getValue())).collect(Collectors.toList()) : Collections.emptyList();
        return params;
    }

    private void displayStatusHistory(String stackName, AmazonCloudFormation cf) {
        this.log.info((CharSequence)"------------------------------");
        this.log.info((CharSequence)("Stack history - " + stackName));
        this.log.info((CharSequence)"------------------------------");
        ListStacksResult r = cf.listStacks();
        r.getStackSummaries().stream().filter(x -> x.getStackName().equals(stackName)).forEach(x -> {
            this.log.info((CharSequence)("id=" + x.getStackId()));
            this.log.info((CharSequence)("  status=" + x.getStackStatus()));
            this.log.info((CharSequence)("  created=" + x.getCreationTime()));
            this.log.info((CharSequence)("  update=" + x.getLastUpdatedTime()));
            this.log.info((CharSequence)("  deleted=" + x.getDeletionTime()));
        });
        this.log.info((CharSequence)"");
    }

    private void displayEvents(String stackName, AmazonCloudFormation cf, long sinceTime) {
        this.log.info((CharSequence)"------------------------------");
        this.log.info((CharSequence)("Event history - " + stackName));
        this.log.info((CharSequence)"------------------------------");
        DescribeStackEventsResult r = cf.describeStackEvents(new DescribeStackEventsRequest().withStackName(stackName));
        r.getStackEvents().stream().sorted((a, b) -> a.getTimestamp().compareTo(b.getTimestamp())).filter(x -> x.getTimestamp().getTime() >= sinceTime - TimeUnit.MINUTES.toMillis(1L)).forEach(x -> {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-ddd HH:mm:ss");
            this.log.info((CharSequence)(sdf.format(x.getTimestamp()) + " " + x.getResourceStatus() + " " + x.getResourceType()));
            if (x.getResourceStatusReason() != null) {
                this.log.info((CharSequence)("  reason=" + x.getResourceStatusReason()));
                if (x.getResourceProperties() != null) {
                    this.log.info((CharSequence)"  properties=\n");
                    this.log.info((CharSequence)Util.formatJson(x.getResourceProperties()));
                }
            }
        });
    }

    private static Status waitForCompletion(AmazonCloudFormation cf, String stackName, int intervalMs, Log log) {
        DescribeStacksRequest describeRequest = new DescribeStacksRequest().withStackName(stackName);
        String stackStatus = "Unknown";
        String stackReason = "";
        log.info((CharSequence)("waiting for action on  " + stackName));
        long t = System.currentTimeMillis();
        while (true) {
            List stacks;
            try {
                stacks = cf.describeStacks(describeRequest).getStacks();
            }
            catch (AmazonCloudFormationException e) {
                log.warn((CharSequence)e.getMessage());
                stacks = Collections.emptyList();
            }
            if (stacks.isEmpty()) {
                stackStatus = "NO_SUCH_STACK";
                stackReason = "Stack has been deleted";
                log.info((CharSequence)(CloudFormationDeployer.time(t) + " " + stackStatus));
                break;
            }
            Stack stack = (Stack)stacks.iterator().next();
            String ss = stack.getStackStatus();
            String sr = stack.getStackStatusReason();
            String msg = ss;
            if (sr != null) {
                msg = msg + " - " + sr;
            }
            log.info((CharSequence)(CloudFormationDeployer.time(t) + " " + msg));
            if (ss.equals(StackStatus.CREATE_COMPLETE.toString()) || ss.equals(StackStatus.CREATE_FAILED.toString()) || ss.equals(StackStatus.UPDATE_COMPLETE.toString()) || ss.equals(StackStatus.UPDATE_ROLLBACK_COMPLETE.toString()) || ss.equals(StackStatus.UPDATE_ROLLBACK_FAILED.toString()) || ss.equals(StackStatus.ROLLBACK_FAILED.toString()) || ss.equals(StackStatus.ROLLBACK_COMPLETE.toString()) || ss.equals(StackStatus.DELETE_FAILED.toString()) || ss.equals(StackStatus.DELETE_COMPLETE.toString())) {
                stackStatus = ss;
                stackReason = stack.getStackStatusReason();
                break;
            }
            try {
                Thread.sleep(intervalMs);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        return new Status(stackStatus, stackReason);
    }

    private static String time(long start) {
        long s = (System.currentTimeMillis() - start) / 1000L;
        long a = s / 60L;
        long b = s % 60L;
        return String.format("%02d:%02d", a, b);
    }

    private static class Status {
        final String value;
        final String reason;

        Status(String value, String reason) {
            this.value = value;
            this.reason = reason;
        }

        public String toString() {
            return "Status [value=" + this.value + ", reason=" + (this.reason == null ? "" : this.reason) + "]";
        }
    }
}

