/*
 * Decompiled with CFR 0.152.
 */
package com.ehsaniara.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CannedAccessControlList;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.ehsaniara.s3.EndpointProperty;
import com.ehsaniara.s3.KeyResolver;
import com.ehsaniara.s3.PathStyleEnabledProperty;
import com.ehsaniara.s3.Progress;
import com.ehsaniara.s3.ProgressFileInputStream;
import com.ehsaniara.s3.ProgressFileOutputStream;
import com.ehsaniara.s3.PublicReadProperty;
import com.ehsaniara.s3.S3Connect;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.IOUtils;
import org.apache.maven.wagon.ResourceDoesNotExistException;
import org.apache.maven.wagon.TransferFailedException;
import org.apache.maven.wagon.authentication.AuthenticationException;
import org.apache.maven.wagon.authentication.AuthenticationInfo;

public class S3StorageRepo {
    private static final Logger log = Logger.getLogger(S3StorageRepo.class.getName());
    private final String bucket;
    private final String baseDirectory;
    private final KeyResolver keyResolver = new KeyResolver();
    private AmazonS3 amazonS3;
    private PublicReadProperty publicReadProperty;

    public S3StorageRepo(String bucket, String baseDirectory, PublicReadProperty publicReadProperty) {
        this.bucket = bucket;
        this.baseDirectory = baseDirectory;
        this.publicReadProperty = publicReadProperty;
    }

    public void connect(AuthenticationInfo authenticationInfo, String region, EndpointProperty endpoint, PathStyleEnabledProperty pathStyle) throws AuthenticationException {
        this.amazonS3 = S3Connect.connect(authenticationInfo, region, endpoint, pathStyle);
    }

    public void copy(String resourceName, File destination, Progress progress) throws TransferFailedException, ResourceDoesNotExistException {
        String key = this.resolveKey(resourceName);
        try {
            S3Object s3Object;
            try {
                s3Object = this.amazonS3.getObject(this.bucket, key);
            }
            catch (AmazonS3Exception e) {
                throw new ResourceDoesNotExistException("Resource not exist");
            }
            destination.getParentFile().mkdirs();
            try (ProgressFileOutputStream outputStream = new ProgressFileOutputStream(destination, progress);
                 S3ObjectInputStream inputStream = s3Object.getObjectContent();){
                IOUtils.copy((InputStream)inputStream, (OutputStream)outputStream);
            }
        }
        catch (AmazonS3Exception | IOException e) {
            log.log(Level.SEVERE, "Could not transfer file", e);
            throw new TransferFailedException("Could not download resource " + key);
        }
    }

    public void put(File file, String destination, Progress progress) throws TransferFailedException {
        String key = this.resolveKey(destination);
        try (ProgressFileInputStream inputStream = new ProgressFileInputStream(file, progress);){
            PutObjectRequest putObjectRequest = new PutObjectRequest(this.bucket, key, (InputStream)inputStream, this.createContentLengthMetadata(file));
            this.applyPublicRead(putObjectRequest);
            this.amazonS3.putObject(putObjectRequest);
        }
        catch (AmazonS3Exception | IOException e) {
            log.log(Level.SEVERE, "Could not transfer file ", e);
            throw new TransferFailedException("Could not transfer file " + file.getName());
        }
    }

    private ObjectMetadata createContentLengthMetadata(File file) {
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength(file.length());
        return metadata;
    }

    public boolean newResourceAvailable(String resourceName, long timeStamp) throws ResourceDoesNotExistException {
        String key = this.resolveKey(resourceName);
        log.log(Level.FINER, String.format("Checking if new key %s exists", key));
        try {
            ObjectMetadata objectMetadata = this.amazonS3.getObjectMetadata(this.bucket, key);
            long updated = objectMetadata.getLastModified().getTime();
            return updated > timeStamp;
        }
        catch (AmazonS3Exception e) {
            log.log(Level.SEVERE, String.format("Could not find %s", key), e);
            throw new ResourceDoesNotExistException("Could not find key " + key);
        }
    }

    public List<String> list(String path) {
        String key = this.resolveKey(path);
        ObjectListing objectListing = this.amazonS3.listObjects(new ListObjectsRequest().withBucketName(this.bucket).withPrefix(key));
        ArrayList<String> objects = new ArrayList<String>();
        this.retrieveAllObjects(objectListing, objects);
        return objects;
    }

    private void applyPublicRead(PutObjectRequest putObjectRequest) {
        if (this.publicReadProperty.get()) {
            log.info("Public read was set to true");
            putObjectRequest.withCannedAcl(CannedAccessControlList.PublicRead);
        }
    }

    private void retrieveAllObjects(ObjectListing objectListing, List<String> objects) {
        objectListing.getObjectSummaries().forEach(os -> objects.add(os.getKey()));
        if (objectListing.isTruncated()) {
            ObjectListing nextObjectListing = this.amazonS3.listNextBatchOfObjects(objectListing);
            this.retrieveAllObjects(nextObjectListing, objects);
        }
    }

    public boolean exists(String resourceName) {
        String key = this.resolveKey(resourceName);
        try {
            this.amazonS3.getObjectMetadata(this.bucket, key);
            return true;
        }
        catch (AmazonS3Exception e) {
            return false;
        }
    }

    public void disconnect() {
        this.amazonS3 = null;
    }

    private String resolveKey(String path) {
        return this.keyResolver.resolve(this.baseDirectory, path);
    }

    public String getBucket() {
        return this.bucket;
    }

    public String getBaseDirectory() {
        return this.baseDirectory;
    }
}

