/*
 * MIT License
 * 
 * Copyright (c) 2018-2020 Fabio Lima
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package com.github.f4b6a3.uuid;

import java.util.UUID;

import com.github.f4b6a3.uuid.creator.nonstandard.PrefixCombCreator;
import com.github.f4b6a3.uuid.creator.nonstandard.ShortPrefixCombCreator;
import com.github.f4b6a3.uuid.creator.nonstandard.ShortSuffixCombCreator;
import com.github.f4b6a3.uuid.creator.nonstandard.SuffixCombCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.DceSecurityUuidCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.NameBasedMd5UuidCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.NameBasedSha1UuidCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.RandomBasedUuidCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.TimeBasedUuidCreator;
import com.github.f4b6a3.uuid.creator.rfc4122.TimeOrderedUuidCreator;
import com.github.f4b6a3.uuid.enums.UuidLocalDomain;
import com.github.f4b6a3.uuid.enums.UuidNamespace;
import com.github.f4b6a3.uuid.exception.InvalidUuidException;
import com.github.f4b6a3.uuid.exception.UuidCreatorException;
import com.github.f4b6a3.uuid.util.UuidConverter;

/**
 * Facade to the UUID factories.
 */
public final class UuidCreator {

	public static final UuidNamespace NAMESPACE_DNS = UuidNamespace.NAMESPACE_DNS;
	public static final UuidNamespace NAMESPACE_URL = UuidNamespace.NAMESPACE_URL;
	public static final UuidNamespace NAMESPACE_ISO_OID = UuidNamespace.NAMESPACE_ISO_OID;
	public static final UuidNamespace NAMESPACE_X500_DN = UuidNamespace.NAMESPACE_X500_DN;

	public static final UuidLocalDomain LOCAL_DOMAIN_PERSON = UuidLocalDomain.LOCAL_DOMAIN_PERSON;
	public static final UuidLocalDomain LOCAL_DOMAIN_GROUP = UuidLocalDomain.LOCAL_DOMAIN_GROUP;
	public static final UuidLocalDomain LOCAL_DOMAIN_ORG = UuidLocalDomain.LOCAL_DOMAIN_ORG;

	private static final UUID UUID_NIL = new UUID(0L, 0L);

	private UuidCreator() {
	}

	/**
	 * Returns a UUID from a byte array.
	 * 
	 * It validates the input byte array.
	 * 
	 * @param uuid a byte array
	 * @return a UUID
	 * @throws InvalidUuidException if invalid
	 */
	public static UUID fromBytes(byte[] uuid) {
		return UuidConverter.fromBytes(uuid);
	}

	/**
	 * Returns a UUID from a string.
	 * 
	 * It validates the input string.
	 * 
	 * It also accepts UUID strings without hyphens.
	 * 
	 * It's an alternative to {@link java.util.UUID#fromString(String)}.
	 * 
	 * @param uuid a string
	 * @return a UUID
	 * @throws InvalidUuidException if invalid
	 */
	public static UUID fromString(String uuid) {
		return UuidConverter.fromString(uuid);
	}

	/**
	 * Returns a Nil UUID.
	 * 
	 * The Nil UUID is special UUID that has all 128 bits set to zero.
	 * 
	 * @return a Nil UUID
	 */
	public static UUID getNil() {
		return UUID_NIL;
	}

	/**
	 * Returns a random UUID.
	 * 
	 * The random generator is {@link java.security.SecureRandom}.
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 4
	 * </pre>
	 * 
	 * @return a version 4 UUID
	 */
	public static UUID getRandomBased() {
		return RandomCreatorHolder.INSTANCE.create();
	}

	/**
	 * [Deprecated] Returns a random UUID generated by a fast random generator.
	 * 
	 * @deprecated This method will be removed in the next major version.
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 4
	 * </pre>
	 * 
	 * @return a version 4 UUID
	 */
	@Deprecated
	public static UUID getFastRandomBased() {
		return FastRandomCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a time-based UUID.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 1
	 * - Hardware address (MAC): NO (random)
	 * </pre>
	 * 
	 * @return a version 1 UUID
	 * @throws UuidCreatorException an overrun exception if too many UUIDs are
	 *                              requested within the same millisecond
	 */
	public static UUID getTimeBased() {
		return TimeBasedCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a time-based UUID with hardware address.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 1 
	 * - Hardware address (MAC): YES
	 * </pre>
	 * 
	 * @return a version 1 UUID
	 * @throws UuidCreatorException an overrun exception if too many UUIDs are
	 *                              requested within the same millisecond
	 */
	public static UUID getTimeBasedWithMac() {
		return TimeBasedWithMacCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a time-ordered UUID.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 6
	 * - Hardware address (MAC): NO (random)
	 * </pre>
	 * 
	 * @return a version 6 UUID
	 * @throws UuidCreatorException an overrun exception if too many UUIDs are
	 *                              requested within the same millisecond
	 */
	public static UUID getTimeOrdered() {
		return TimeOrderedCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a time-ordered UUID with hardware address.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 6
	 * - Hardware address (MAC): YES
	 * </pre>
	 * 
	 * @return a version 6 UUID
	 * @throws UuidCreatorException an overrun exception if too many UUIDs are
	 *                              requested within the same millisecond
	 */
	public static UUID getTimeOrderedWithMac() {
		return TimeOrderedWithMacCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a name-based UUID (MD5).
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: NO
	 * </pre>
	 * 
	 * @param name a name string
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(String name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(name);
	}

	/**
	 * Returns a name-based UUID (MD5).
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: NO
	 * </pre>
	 * 
	 * @param name a byte array
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(byte[] name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(name);
	}

	/**
	 * Returns a name-based UUID (MD5).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: YES (custom)
	 * </pre>
	 * 
	 * @param namespace a custom name space UUID
	 * @param name      a name string
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(UUID namespace, String name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (MD5).
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: YES (custom)
	 * </pre>
	 * 
	 * @param namespace a custom name space UUID
	 * @param name      a byte array
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(UUID namespace, byte[] name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (MD5).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: YES (predefined)
	 * </pre>
	 * 
	 * <pre>
	 * Name spaces predefined by RFC-4122 (Appendix C):
	 * 
	 * - NAMESPACE_DNS: Name string is a fully-qualified domain name;
	 * - NAMESPACE_URL: Name string is a URL;
	 * - NAMESPACE_ISO_OID: Name string is an ISO OID;
	 * - NAMESPACE_X500_DN: Name string is an X.500 DN (in DER or a text format).
	 * </pre>
	 * 
	 * See: {@link UuidNamespace}.
	 * 
	 * @param namespace a predefined name space enumeration
	 * @param name      a name string
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(UuidNamespace namespace, String name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (MD5).
	 * 
	 * <pre>
	 * Details: 
	 * - Version number: 3 
	 * - Hash Algorithm: MD5 
	 * - Name Space: YES (predefined)
	 * </pre>
	 * 
	 * <pre>
	 * Name spaces predefined by RFC-4122 (Appendix C):
	 * 
	 * - NAMESPACE_DNS: Name string is a fully-qualified domain name;
	 * - NAMESPACE_URL: Name string is a URL;
	 * - NAMESPACE_ISO_OID: Name string is an ISO OID;
	 * - NAMESPACE_X500_DN: Name string is an X.500 DN (in DER or a text format).
	 * </pre>
	 * 
	 * See: {@link UuidNamespace}.
	 * 
	 * @param namespace a predefined name space enumeration
	 * @param name      a byte array
	 * @return a version 3 UUID
	 */
	public static UUID getNameBasedMd5(UuidNamespace namespace, byte[] name) {
		return NameBasedMd5CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: NO
	 * </pre>
	 * 
	 * @param name a name string
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(String name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: NO
	 * </pre>
	 * 
	 * @param name a byte array
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(byte[] name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: YES (custom)
	 * </pre>
	 * 
	 * @param namespace a custom name space UUID
	 * @param name      a name string
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(UUID namespace, String name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: YES (custom)
	 * </pre>
	 * 
	 * @param namespace a custom name space UUID
	 * @param name      a byte array
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(UUID namespace, byte[] name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: YES (predefined)
	 * </pre>
	 * 
	 * <pre>
	 * Name spaces predefined by RFC-4122 (Appendix C):
	 * 
	 * - NAMESPACE_DNS: Name string is a fully-qualified domain name;
	 * - NAMESPACE_URL: Name string is a URL;
	 * - NAMESPACE_ISO_OID: Name string is an ISO OID;
	 * - NAMESPACE_X500_DN: Name string is an X.500 DN (in DER or a text format).
	 * </pre>
	 * 
	 * See: {@link UuidNamespace}.
	 * 
	 * @param namespace a predefined name space enumeration
	 * @param name      a name string
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(UuidNamespace namespace, String name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a name-based UUID (SHA1).
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 5 
	 * - Hash Algorithm: SHA1 
	 * - Name Space: YES (predefined)
	 * </pre>
	 * 
	 * <pre>
	 * Name spaces predefined by RFC-4122 (Appendix C):
	 * 
	 * - NAMESPACE_DNS: Name string is a fully-qualified domain name;
	 * - NAMESPACE_URL: Name string is a URL;
	 * - NAMESPACE_ISO_OID: Name string is an ISO OID;
	 * - NAMESPACE_X500_DN: Name string is an X.500 DN (in DER or a text format).
	 * </pre>
	 * 
	 * See: {@link UuidNamespace}.
	 * 
	 * @param namespace a predefined name space enumeration
	 * @param name      a byte array
	 * @return a version 5 UUID
	 */
	public static UUID getNameBasedSha1(UuidNamespace namespace, byte[] name) {
		return NameBasedSha1CreatorHolder.INSTANCE.create(namespace, name);
	}

	/**
	 * Returns a DCE Security UUID.
	 *
	 * See: {@link UuidLocalDomain}.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 2 
	 * - Hardware address (MAC): NO (random)
	 * </pre>
	 * 
	 * @param localDomain     a custom local domain byte
	 * @param localIdentifier a local identifier
	 * @return a version 2 UUID
	 */
	public static UUID getDceSecurity(byte localDomain, int localIdentifier) {
		return DceSecurityCreatorHolder.INSTANCE.create(localDomain, localIdentifier);
	}

	/**
	 * Returns a DCE Security UUID.
	 *
	 * See: {@link UuidLocalDomain}.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 2 
	 * - Hardware address (MAC): YES
	 * </pre>
	 * 
	 * @param localDomain     a custom local domain byte
	 * @param localIdentifier a local identifier
	 * @return a version 2 UUID
	 */
	public static UUID getDceSecurityWithMac(byte localDomain, int localIdentifier) {
		return DceSecurityWithMacCreatorHolder.INSTANCE.create(localDomain, localIdentifier);
	}

	/**
	 * Returns a DCE Security UUID.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 2 
	 * - Hardware address (MAC): NO (random)
	 * </pre>
	 * 
	 * <pre>
	 * Local domains predefined by DCE 1.1 Authentication and Security Services (Chapter 11):
	 * 
	 * - LOCAL_DOMAIN_PERSON: 0 (interpreted as POSIX UID domain);
	 * - LOCAL_DOMAIN_GROUP: 1 (interpreted as POSIX GID domain);
	 * - LOCAL_DOMAIN_ORG: 2.
	 * </pre>
	 * 
	 * See: {@link UuidLocalDomain}.
	 * 
	 * @param localDomain     a predefined local domain enumeration
	 * @param localIdentifier a local identifier
	 * @return a version 2 UUID
	 */
	public static UUID getDceSecurity(UuidLocalDomain localDomain, int localIdentifier) {
		return DceSecurityCreatorHolder.INSTANCE.create(localDomain, localIdentifier);
	}

	/**
	 * Returns a DCE Security UUID.
	 *
	 * <pre>
	 * Details: 
	 * - Version number: 2 
	 * - Hardware address (MAC): YES
	 * </pre>
	 * 
	 * <pre>
	 * Local domains predefined by DCE 1.1 Authentication and Security Services (Chapter 11):
	 * 
	 * - LOCAL_DOMAIN_PERSON: 0 (interpreted as POSIX UID domain);
	 * - LOCAL_DOMAIN_GROUP: 1 (interpreted as POSIX GID domain);
	 * - LOCAL_DOMAIN_ORG: 2.
	 * </pre>
	 * 
	 * See: {@link UuidLocalDomain}.
	 * 
	 * @param localDomain     a predefined local domain enumeration
	 * @param localIdentifier a local identifier
	 * @return a version 2 UUID
	 */
	public static UUID getDceSecurityWithMac(UuidLocalDomain localDomain, int localIdentifier) {
		return DceSecurityWithMacCreatorHolder.INSTANCE.create(localDomain, localIdentifier);
	}

	/**
	 * Returns a Prefix COMB.
	 * 
	 * The creation millisecond is a 6 bytes PREFIX is at the MOST significant bits.
	 * 
	 * Read: The Cost of GUIDs as Primary Keys
	 * http://www.informit.com/articles/article.aspx?p=25862
	 * 
	 * @return a GUID
	 */
	public static UUID getPrefixComb() {
		return PrefixCombCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a Suffix COMB.
	 * 
	 * The creation millisecond is a 6 bytes SUFFIX is at the LEAST significant
	 * bits.
	 * 
	 * Read: The Cost of GUIDs as Primary Keys
	 * http://www.informit.com/articles/article.aspx?p=25862
	 * 
	 * @return a GUID
	 */
	public static UUID getSuffixComb() {
		return SuffixCombCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns n Short Prefix COMB.
	 * 
	 * The creation minute is a 2 bytes PREFIX is at the MOST significant bits.
	 * 
	 * The prefix wraps around every ~45 days (2^16/60/24 = ~45).
	 * 
	 * Read: Sequential UUID Generators
	 * https://www.2ndquadrant.com/en/blog/sequential-uuid-generators/
	 * 
	 * @return a GUID
	 */
	public static UUID getShortPrefixComb() {
		return ShortPrefixCombCreatorHolder.INSTANCE.create();
	}

	/**
	 * Returns a Short Suffix COMB.
	 * 
	 * The creation minute is a 2 bytes SUFFIX is at the LEAST significant bits.
	 * 
	 * The suffix wraps around every ~45 days (2^16/60/24 = ~45).
	 * 
	 * Read: Sequential UUID Generators
	 * https://www.2ndquadrant.com/en/blog/sequential-uuid-generators/
	 * 
	 * @return a GUID
	 */
	public static UUID getShortSuffixComb() {
		return ShortSuffixCombCreatorHolder.INSTANCE.create();
	}

	/*
	 * Public static methods for creating FACTORIES of UUIDs
	 */

	/**
	 * Returns a {@link RandomBasedUuidCreator} that creates UUID version 4.
	 * 
	 * @return {@link RandomBasedUuidCreator}
	 */
	public static RandomBasedUuidCreator getRandomBasedCreator() {
		return new RandomBasedUuidCreator();
	}

	/**
	 * Returns a {@link TimeBasedUuidCreator} that creates UUID version 1.
	 * 
	 * @return {@link TimeBasedUuidCreator}
	 */
	public static TimeBasedUuidCreator getTimeBasedCreator() {
		return new TimeBasedUuidCreator();
	}

	/**
	 * Returns a {@link TimeOrderedUuidCreator} that creates UUID version 6.
	 * 
	 * @return {@link TimeOrderedUuidCreator}
	 */
	public static TimeOrderedUuidCreator getTimeOrderedCreator() {
		return new TimeOrderedUuidCreator();
	}

	/**
	 * Returns a {@link NameBasedMd5UuidCreator} that creates UUID version 3.
	 * 
	 * @return {@link NameBasedMd5UuidCreator}
	 */
	public static NameBasedMd5UuidCreator getNameBasedMd5Creator() {
		return new NameBasedMd5UuidCreator();
	}

	/**
	 * Returns a {@link NameBasedSha1UuidCreator} that creates UUID version 5.
	 * 
	 * @return {@link NameBasedSha1UuidCreator}
	 */
	public static NameBasedSha1UuidCreator getNameBasedSha1Creator() {
		return new NameBasedSha1UuidCreator();
	}

	/**
	 * Returns a {@link DceSecurityUuidCreator} that creates UUID version 2.
	 * 
	 * @return {@link DceSecurityUuidCreator}
	 */
	public static DceSecurityUuidCreator getDceSecurityCreator() {
		return new DceSecurityUuidCreator();
	}

	/**
	 * Returns a {@link SuffixCombCreator}.
	 * 
	 * Read: The Cost of GUIDs as Primary Keys
	 * http://www.informit.com/articles/article.aspx?p=25862
	 * 
	 * @return {@link SuffixCombCreator}
	 */
	public static SuffixCombCreator getSuffixCombCreator() {
		return new SuffixCombCreator();
	}

	/**
	 * Returns a {@link PrefixCombCreator}.
	 * 
	 * Read: The Cost of GUIDs as Primary Keys
	 * http://www.informit.com/articles/article.aspx?p=25862
	 * 
	 * @return {@link PrefixCombCreator}
	 */
	public static PrefixCombCreator getPrefixCombCreator() {
		return new PrefixCombCreator();
	}

	/**
	 * Returns a {@link ShortPrefixCombCreator}.
	 * 
	 * Read: Sequential UUID Generators
	 * https://www.2ndquadrant.com/en/blog/sequential-uuid-generators/
	 * 
	 * @return {@link ShortPrefixCombCreator}
	 */
	public static ShortPrefixCombCreator getShortPrefixCombCreator() {
		return new ShortPrefixCombCreator();
	}

	/**
	 * Returns a {@link ShortSuffixCombCreator}.
	 * 
	 * Read: Sequential UUID Generators
	 * https://www.2ndquadrant.com/en/blog/sequential-uuid-generators/
	 * 
	 * @return {@link ShortSuffixCombCreator}
	 */
	public static ShortSuffixCombCreator getShortSuffixCombCreator() {
		return new ShortSuffixCombCreator();
	}

	/*
	 * Private classes for lazy holders
	 */

	private static class RandomCreatorHolder {
		static final RandomBasedUuidCreator INSTANCE = getRandomBasedCreator();
	}

	private static class FastRandomCreatorHolder {
		static final RandomBasedUuidCreator INSTANCE = getRandomBasedCreator().withFastRandomGenerator();
	}

	private static class TimeOrderedCreatorHolder {
		static final TimeOrderedUuidCreator INSTANCE = getTimeOrderedCreator();
	}

	private static class TimeOrderedWithMacCreatorHolder {
		static final TimeOrderedUuidCreator INSTANCE = getTimeOrderedCreator().withMacNodeIdentifier();
	}

	private static class TimeBasedCreatorHolder {
		static final TimeBasedUuidCreator INSTANCE = getTimeBasedCreator();
	}

	private static class TimeBasedWithMacCreatorHolder {
		static final TimeBasedUuidCreator INSTANCE = getTimeBasedCreator().withMacNodeIdentifier();
	}

	private static class NameBasedMd5CreatorHolder {
		static final NameBasedMd5UuidCreator INSTANCE = getNameBasedMd5Creator();
	}

	private static class NameBasedSha1CreatorHolder {
		static final NameBasedSha1UuidCreator INSTANCE = getNameBasedSha1Creator();
	}

	private static class DceSecurityCreatorHolder {
		static final DceSecurityUuidCreator INSTANCE = getDceSecurityCreator();
	}

	private static class DceSecurityWithMacCreatorHolder {
		static final DceSecurityUuidCreator INSTANCE = getDceSecurityCreator().withMacNodeIdentifier();
	}

	private static class SuffixCombCreatorHolder {
		static final SuffixCombCreator INSTANCE = getSuffixCombCreator();
	}

	private static class PrefixCombCreatorHolder {
		static final PrefixCombCreator INSTANCE = getPrefixCombCreator();
	}

	private static class ShortPrefixCombCreatorHolder {
		static final ShortPrefixCombCreator INSTANCE = getShortPrefixCombCreator();
	}

	private static class ShortSuffixCombCreatorHolder {
		static final ShortSuffixCombCreator INSTANCE = getShortSuffixCombCreator();
	}
}
