package com.github.gigiosouza.japigeebundler;

import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.github.gigiosouza.japigeebundler.apigee.APIProxy;
import com.github.gigiosouza.japigeebundler.apigee.Resource;
import com.github.gigiosouza.japigeebundler.apigee.flows.proxyendpoint.ProxyEndpoint;
import com.github.gigiosouza.japigeebundler.apigee.flows.targetendpoint.TargetEndpoint;
import com.github.gigiosouza.japigeebundler.apigee.policies.Policy;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;

import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.ZonedDateTime;
import java.util.HashSet;
import java.util.Set;

@Data
@AllArgsConstructor
@NoArgsConstructor
public class APIBundler {

  private APIProxy apiProxy = new APIProxy();
  private Set<ProxyEndpoint> proxyEndpoints = new HashSet<>();
  private Set<TargetEndpoint> targetEndpoints = new HashSet<>();
  private Set<Policy> policies = new HashSet<>();
  private Set<Resource> resources = new HashSet<>();

  public void addProxyEndpoint(final ProxyEndpoint proxyEndpoint) {
    proxyEndpoints.add(proxyEndpoint);
  }

  public void addTargetEndpoint(final TargetEndpoint targetEndpoint) {
    targetEndpoints.add(targetEndpoint);
  }

  public void addPolicy(final Policy policy) {
    policies.add(policy);
  }

  public void addResource(final Resource resource) {
    resources.add(resource);
  }

  public void bundle(final String rootPath, final Boolean keepSrc) throws IOException {
    XmlMapper xmlMapper = XmlMapperFactory.getInstance();

    Path bundleRoot = Paths.get(rootPath);

    if (!Files.exists(bundleRoot)) {
      throw new InvalidPathException(rootPath, "Root path doesn't exist");
    }

    Path srcPath = bundleRoot.resolve(apiProxy.getName()).resolve("apiproxy");

    Files.createDirectories(srcPath);

    if (!proxyEndpoints.isEmpty()) {
      Path proxiesPath = srcPath.resolve("proxies");
      Files.createDirectory(proxiesPath);
      for (ProxyEndpoint p : proxyEndpoints) {
        apiProxy.addProxyEndpoint(p);
        Path proxyPath = Files.createFile(proxiesPath.resolve(p.getName() + ".xml"));
        xmlMapper.writeValue(proxyPath.toFile(), p);
      }
    }

    if (!targetEndpoints.isEmpty()) {
      Path targetsPath = srcPath.resolve("targets");
      Files.createDirectory(targetsPath);
      for (TargetEndpoint t : targetEndpoints) {
        apiProxy.addTargetEndpoint(t);
        Path targetPath = Files.createFile(targetsPath.resolve(t.getName() + ".xml"));
        xmlMapper.writeValue(targetPath.toFile(), t);
      }
    }

    if (!policies.isEmpty()) {
      Path policiesPath = srcPath.resolve("policies");
      Files.createDirectory(policiesPath);
      for (Policy p : policies) {
        apiProxy.addPolicy(p);
        Path policyPath = Files.createFile(policiesPath.resolve(p.getName() + ".xml"));
        xmlMapper.writeValue(policyPath.toFile(), p);
      }
    }

    if (!resources.isEmpty()) {
      Path resourcesPath = srcPath.resolve("resources");
      Files.createDirectory(resourcesPath);
      for (Resource r : resources) {
        apiProxy.addResource(r);
        Path resourcePath = resourcesPath.resolve(r.getType().name());
        if (!Files.exists(resourcePath)) {
          Files.createDirectory(resourcePath);
        }
        resourcePath = Files.createFile(resourcePath.resolve(r.getFile().getName()));
        xmlMapper.writeValue(resourcePath.toFile(), r);
      }
    }

    // TODO test if Instant.now().toEpochMilli() is a better option due to Zone Offset
    ZonedDateTime now = ZonedDateTime.now();
    apiProxy.setCreatedAt(now.toEpochSecond());
    apiProxy.setLastModifiedAt(now.toEpochSecond());
    apiProxy.setCreatedBy("javaapp");
    apiProxy.setLastModifiedBy("javaapp");
    apiProxy.setDisplayName(apiProxy.getName());
    apiProxy.setDescription("my first generated api proxy bundle");
    apiProxy.setRevision(1);

    Path apiProxyPath = Files.createFile(srcPath.resolve(apiProxy.getName() + ".xml"));
    xmlMapper.writeValue(apiProxyPath.toFile(), apiProxy);

    zipBundle(bundleRoot.resolve(apiProxy.getName() + ".zip"), srcPath.resolve("..").normalize());
    if (!keepSrc) {
      deleteSrc(srcPath.resolve("..").normalize());
    }
  }

  private Path zipBundle(Path bundleZip, Path srcPath) throws IOException {
    try (
      ZipArchiveOutputStream zipOut = new ZipArchiveOutputStream(bundleZip.toFile());
    ) {
      Files.walkFileTree(srcPath, new SimpleFileVisitor<Path>() {
        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attributes) {
          if (attributes.isSymbolicLink()) {
            return FileVisitResult.CONTINUE;
          }

          try (FileInputStream fis = new FileInputStream(file.toFile())) {

            Path targetFile = srcPath.relativize(file);
            ZipArchiveEntry zipEntry = new ZipArchiveEntry(targetFile.toString());

            zipOut.putArchiveEntry(zipEntry);

            byte[] buffer = new byte[1024];
            int len;
            while ((len = fis.read(buffer)) > 0) {
              zipOut.write(buffer, 0, len);
            }

            zipOut.closeArchiveEntry();
          } catch (IOException e) {
            e.printStackTrace();
          }
          return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) {
          System.err.printf("Unable to zip : %s%n%s%n", file, exc);
          return FileVisitResult.CONTINUE;
        }
      });
    }

    return bundleZip;
  }

  private void deleteSrc(Path src) throws IOException {
    if (src.toFile().isDirectory()) {
      String[] children = src.toFile().list();
      if (children != null) {
        for (String child : children) {
          deleteSrc(src.resolve(child));
        }
      }
    }
    Files.delete(src);
  }

}
