package com.github.gigiosouza.japigeebundler.apigee.policies;

import com.fasterxml.jackson.dataformat.xml.annotation.JacksonXmlProperty;
import com.github.gigiosouza.japigeebundler.apigee.commons.Properties;
import com.github.gigiosouza.japigeebundler.apigee.exception.InvalidPolicyConfiguration;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.Validator;
import jakarta.validation.constraints.NotNull;
import lombok.Data;

import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Data
public abstract class Policy {

  @JacksonXmlProperty(isAttribute = true)
  protected Boolean async;

  @JacksonXmlProperty(isAttribute = true)
  protected Boolean continueOnError;

  @JacksonXmlProperty(isAttribute = true)
  protected Boolean enabled;

  @NotNull
  @JacksonXmlProperty(isAttribute = true)
  protected String name;

  @NotNull
  @JacksonXmlProperty(localName = "DisplayName")
  protected String displayName;

  @JacksonXmlProperty(localName = "Properties")
  private Properties properties;

  protected Set<String> customValidation() {
    return null;
  }

  protected void defaultHeader() {
    this.setContinueOnError(false);
    this.setEnabled(true);
    this.setAsync(false);
  }

  public abstract void defaultValues();

  public void validate(Validator validator) throws InvalidPolicyConfiguration {
    Stream<String> violations = null;
    Set<String> customViolations = customValidation();

    if (customViolations != null && !customViolations.isEmpty()) {
      violations = customViolations.stream();
    }

    Set<ConstraintViolation<Policy>> validatorsViolations = validator.validate(this);
    if (!validatorsViolations.isEmpty()) {
      Stream<String> violationStream = validatorsViolations
        .stream()
        .map(v -> v.getPropertyPath() + ": " + v.getMessage());

      if (violations == null) {
        violations = violationStream;
      } else {
        violations = Stream.concat(violations, violationStream);
      }
    }

    if (violations != null) {
      throw new InvalidPolicyConfiguration(violations.collect(Collectors.joining("\n")));
    }
  }

}
