/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix.fp32;

import java.io.Serializable;
import java.nio.FloatBuffer;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.Diag;
import smile.math.blas.EVDJob;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.SVDJob;
import smile.math.blas.Side;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.fp32.IMatrix;
import smile.sort.QuickSort;
import smile.stat.distribution.Distribution;
import smile.stat.distribution.GaussianDistribution;

public class Matrix
extends IMatrix {
    private static final long serialVersionUID = 3L;
    private static final Logger logger = LoggerFactory.getLogger(Matrix.class);
    float[] A;
    int ld;
    int m;
    int n;
    UPLO uplo;
    Diag diag;

    public Matrix(int m, int n) {
        this(m, n, 0.0f);
    }

    public Matrix(int m, int n, float a) {
        if (m <= 0 || n <= 0) {
            throw new IllegalArgumentException(String.format("Invalid matrix size: %d x %d", m, n));
        }
        this.m = m;
        this.n = n;
        this.ld = Matrix.ld(m);
        this.A = new float[this.ld * n];
        if (a != 0.0f) {
            Arrays.fill(this.A, a);
        }
    }

    public Matrix(int m, int n, int ld, float[] A) {
        if (this.layout() == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (this.layout() == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        this.m = m;
        this.n = n;
        this.ld = ld;
        this.A = A;
    }

    public static Matrix of(float[][] A) {
        int m = A.length;
        int n = A[0].length;
        Matrix matrix = new Matrix(m, n);
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                matrix.set(i, j, A[i][j]);
            }
        }
        return matrix;
    }

    public static Matrix of(double[][] A) {
        int m = A.length;
        int n = A[0].length;
        Matrix matrix = new Matrix(m, n);
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                matrix.set(i, j, (float)A[i][j]);
            }
        }
        return matrix;
    }

    public static Matrix column(float[] A) {
        return Matrix.column(A, 0, A.length);
    }

    public static Matrix column(float[] A, int offset, int length) {
        Matrix matrix = new Matrix(length, 1, length, new float[length]);
        System.arraycopy(A, offset, matrix.A, 0, length);
        return matrix;
    }

    public static Matrix column(double[] A) {
        return Matrix.column(A, 0, A.length);
    }

    public static Matrix column(double[] A, int offset, int length) {
        Matrix matrix = new Matrix(length, 1, length, new float[length]);
        for (int i = 0; i < length; ++i) {
            matrix.A[i] = (float)A[i + offset];
        }
        return matrix;
    }

    public static Matrix row(float[] A) {
        return Matrix.row(A, 0, A.length);
    }

    public static Matrix row(float[] A, int offset, int length) {
        Matrix matrix = new Matrix(1, length, 1, new float[length]);
        System.arraycopy(A, offset, matrix.A, 0, length);
        return matrix;
    }

    public static Matrix row(double[] A) {
        return Matrix.row(A, 0, A.length);
    }

    public static Matrix row(double[] A, int offset, int length) {
        Matrix matrix = new Matrix(1, length, 1, new float[length]);
        for (int i = 0; i < length; ++i) {
            matrix.A[i] = (float)A[i + offset];
        }
        return matrix;
    }

    public static Matrix rand(int m, int n, Distribution distribution) {
        Matrix matrix = new Matrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, (float)distribution.rand());
            }
        }
        return matrix;
    }

    public static Matrix randn(int m, int n) {
        return Matrix.rand(m, n, GaussianDistribution.getInstance());
    }

    public static Matrix rand(int m, int n) {
        Matrix matrix = new Matrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, (float)MathEx.random());
            }
        }
        return matrix;
    }

    public static Matrix rand(int m, int n, float lo, float hi) {
        Matrix matrix = new Matrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, (float)MathEx.random(lo, hi));
            }
        }
        return matrix;
    }

    public static Matrix eye(int n) {
        return Matrix.diag(n, 1.0f);
    }

    public static Matrix eye(int m, int n) {
        return Matrix.diag(m, n, 1.0f);
    }

    public static Matrix diag(int n, float diag) {
        return Matrix.diag(n, n, diag);
    }

    public static Matrix diag(int m, int n, float diag) {
        Matrix D = new Matrix(m, n);
        int k = Math.min(m, n);
        for (int i = 0; i < k; ++i) {
            D.set(i, i, diag);
        }
        return D;
    }

    public static Matrix diag(float[] diag) {
        int n = diag.length;
        Matrix D = new Matrix(n, n);
        for (int i = 0; i < n; ++i) {
            D.set(i, i, diag[i]);
        }
        return D;
    }

    public static Matrix toeplitz(float[] a) {
        int n = a.length;
        Matrix toeplitz = new Matrix(n, n);
        toeplitz.uplo(UPLO.LOWER);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, a[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, a[j - i]);
            }
        }
        return toeplitz;
    }

    public static Matrix toeplitz(float[] kl, float[] ku) {
        if (kl.length != ku.length - 1) {
            throw new IllegalArgumentException(String.format("Invalid sub-diagonals and super-diagonals size: %d != %d - 1", kl.length, ku.length));
        }
        int n = kl.length;
        Matrix toeplitz = new Matrix(n, n);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, kl[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, ku[j - i]);
            }
        }
        return toeplitz;
    }

    @Override
    public int nrow() {
        return this.m;
    }

    @Override
    public int ncol() {
        return this.n;
    }

    @Override
    public long size() {
        return (long)this.m * (long)this.n;
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public int ld() {
        return this.ld;
    }

    public boolean isSymmetric() {
        return this.uplo != null && this.diag == null;
    }

    public Matrix uplo(UPLO uplo) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.uplo = uplo;
        return this;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public Matrix triangular(Diag diag) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.diag = diag;
        return this;
    }

    public Diag triangular() {
        return this.diag;
    }

    @Override
    public Matrix copy() {
        Matrix matrix;
        if (this.layout() == Layout.COL_MAJOR) {
            matrix = new Matrix(this.m, this.n, this.ld, (float[])this.A.clone());
        } else {
            matrix = new Matrix(this.m, this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j));
                }
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public float[][] toArray() {
        float[][] array = new float[this.m][this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                array[i][j] = this.get(i, j);
            }
        }
        return array;
    }

    public Matrix set(Matrix b) {
        this.m = b.m;
        this.n = b.n;
        this.diag = b.diag;
        this.uplo = b.uplo;
        if (this.layout() == b.layout()) {
            this.A = b.A;
            this.ld = b.ld;
        } else if (this.layout() == Layout.COL_MAJOR) {
            this.ld = Matrix.ld(this.m);
            this.A = new float[this.ld * this.n];
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, this.get(i, j));
                }
            }
        } else {
            this.ld = Matrix.ld(this.n);
            this.A = new float[this.ld * this.m];
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    this.set(i, j, this.get(i, j));
                }
            }
        }
        return this;
    }

    protected int index(int i, int j) {
        return j * this.ld + i;
    }

    @Override
    public float get(int i, int j) {
        return this.A[this.index(i, j)];
    }

    @Override
    public void set(int i, int j, float x) {
        this.A[this.index((int)i, (int)j)] = x;
    }

    public Matrix get(int[] rows, int[] cols) {
        Matrix sub = new Matrix(rows.length, cols.length);
        for (int j = 0; j < cols.length; ++j) {
            int col = cols[j];
            if (col < 0) {
                col = this.n + col;
            }
            for (int i = 0; i < rows.length; ++i) {
                int row = rows[i];
                if (row < 0) {
                    row = this.m + row;
                }
                sub.set(i, j, this.get(row, col));
            }
        }
        return sub;
    }

    public float[] row(int i) {
        float[] x = new float[this.n];
        if (i < 0) {
            i = this.m + i;
        }
        if (this.layout() == Layout.COL_MAJOR) {
            for (int j = 0; j < this.n; ++j) {
                x[j] = this.get(i, j);
            }
        } else {
            System.arraycopy(this.A, this.index(i, 0), x, 0, this.n);
        }
        return x;
    }

    public float[] col(int j) {
        float[] x = new float[this.m];
        if (j < 0) {
            j = this.n + j;
        }
        if (this.layout() == Layout.COL_MAJOR) {
            System.arraycopy(this.A, this.index(0, j), x, 0, this.m);
        } else {
            for (int i = 0; i < this.m; ++i) {
                x[i] = this.get(i, j);
            }
        }
        return x;
    }

    public Matrix rows(int ... rows) {
        Matrix x = new Matrix(rows.length, this.n);
        for (int i = 0; i < rows.length; ++i) {
            int row = rows[i];
            if (row < 0) {
                row = this.m + row;
            }
            if (this.layout() == Layout.COL_MAJOR) {
                for (int j = 0; j < this.n; ++j) {
                    x.set(i, j, this.get(row, j));
                }
                continue;
            }
            System.arraycopy(this.A, this.index(row, 0), x.A, x.index(i, 0), this.n);
        }
        return x;
    }

    public Matrix cols(int ... cols) {
        Matrix x = new Matrix(this.m, cols.length);
        for (int j = 0; j < cols.length; ++j) {
            int col = cols[j];
            if (col < 0) {
                col = this.n + col;
            }
            if (this.layout() == Layout.COL_MAJOR) {
                System.arraycopy(this.A, this.index(0, col), x.A, x.index(0, j), this.m);
                continue;
            }
            for (int i = 0; i < this.m; ++i) {
                x.set(i, j, this.get(i, col));
            }
        }
        return x;
    }

    public Matrix submatrix(int i, int j, int k, int l) {
        if (i < 0 || i >= this.m || k < i || k >= this.m || j < 0 || j >= this.n || l < j || l >= this.n) {
            throw new IllegalArgumentException(String.format("Invalid submatrix range (%d:%d, %d:%d) of %d x %d", i, k, j, l, this.m, this.n));
        }
        Matrix sub = new Matrix(k - i + 1, l - j + 1);
        for (int jj = j; jj <= l; ++jj) {
            for (int ii = i; ii <= k; ++ii) {
                sub.set(ii - i, jj - j, this.get(ii, jj));
            }
        }
        return sub;
    }

    public void fill(float x) {
        Arrays.fill(this.A, x);
    }

    public Matrix transpose() {
        return this.transpose(true);
    }

    public Matrix transpose(boolean share) {
        Matrix matrix;
        if (share) {
            matrix = this.layout() == Layout.ROW_MAJOR ? new Matrix(this.n, this.m, this.ld, this.A) : new RowMajor(this.n, this.m, this.ld, this.A);
        } else {
            matrix = new Matrix(this.n, this.m);
            for (int j = 0; j < this.m; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    matrix.set(i, j, this.get(j, i));
                }
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public boolean equals(Object o) {
        if (!(o instanceof Matrix)) {
            return false;
        }
        return this.equals((Matrix)o, 1.0E-7f);
    }

    public boolean equals(Matrix o, float epsilon) {
        if (this.m != o.m || this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), epsilon)) continue;
                return false;
            }
        }
        return true;
    }

    public float add(int i, int j, float b) {
        int n = this.index(i, j);
        float f = this.A[n] + b;
        this.A[n] = f;
        return f;
    }

    public float sub(int i, int j, float b) {
        int n = this.index(i, j);
        float f = this.A[n] - b;
        this.A[n] = f;
        return f;
    }

    public float mul(int i, int j, float b) {
        int n = this.index(i, j);
        float f = this.A[n] * b;
        this.A[n] = f;
        return f;
    }

    public float div(int i, int j, float b) {
        int n = this.index(i, j);
        float f = this.A[n] / b;
        this.A[n] = f;
        return f;
    }

    public Matrix addDiag(float b) {
        int l = Math.min(this.m, this.n);
        for (int i = 0; i < l; ++i) {
            int n = this.index(i, i);
            this.A[n] = this.A[n] + b;
        }
        return this;
    }

    public Matrix addDiag(float[] b) {
        int l = Math.min(this.m, this.n);
        if (b.length != l) {
            throw new IllegalArgumentException("Invalid diagonal array size: " + b.length);
        }
        for (int i = 0; i < l; ++i) {
            int n = this.index(i, i);
            this.A[n] = this.A[n] + b[i];
        }
        return this;
    }

    public Matrix add(float b) {
        int i = 0;
        while (i < this.A.length) {
            int n = i++;
            this.A[n] = this.A[n] + b;
        }
        return this;
    }

    public Matrix sub(float b) {
        int i = 0;
        while (i < this.A.length) {
            int n = i++;
            this.A[n] = this.A[n] - b;
        }
        return this;
    }

    public Matrix mul(float b) {
        int i = 0;
        while (i < this.A.length) {
            int n = i++;
            this.A[n] = this.A[n] * b;
        }
        return this;
    }

    public Matrix div(float b) {
        int i = 0;
        while (i < this.A.length) {
            int n = i++;
            this.A[n] = this.A[n] / b;
        }
        return this;
    }

    public Matrix add(Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                int n = i;
                this.A[n] = this.A[n] + B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.add(i, j, B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix sub(Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                int n = i;
                this.A[n] = this.A[n] - B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.sub(i, j, B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix mul(Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                int n = i;
                this.A[n] = this.A[n] * B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.mul(i, j, B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix div(Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                int n = i;
                this.A[n] = this.A[n] / B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.div(i, j, B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix add(float beta, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                int n = i;
                this.A[n] = this.A[n] + beta * B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.add(i, j, beta * B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix add(float alpha, float beta, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                this.A[i] = alpha * this.A[i] + beta * B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix add2(float alpha, float beta, Matrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == B.layout() && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                this.A[i] = alpha * this.A[i] + beta * B.A[i] * B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j) * B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix add(float alpha, Matrix A, float beta, Matrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        if (this.layout() == A.layout() && this.layout() == B.layout() && this.ld == A.ld && this.ld == B.ld) {
            for (int i = 0; i < this.A.length; ++i) {
                this.A[i] = alpha * A.A[i] + beta * B.A[i];
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, alpha * A.get(i, j) + beta * B.get(i, j));
                }
            }
        }
        return this;
    }

    public Matrix add(float alpha, float[] x, float[] y) {
        if (this.m != x.length || this.n != y.length) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.isSymmetric() && x == y) {
            BLAS.engine.syr(this.layout(), this.uplo, this.m, alpha, x, 1, this.A, this.ld);
        } else {
            BLAS.engine.ger(this.layout(), this.m, this.n, alpha, x, 1, y, 1, this.A, this.ld);
        }
        return this;
    }

    public Matrix replaceNaN(float x) {
        for (int i = 0; i < this.A.length; ++i) {
            if (!Double.isNaN(this.A[i])) continue;
            this.A[i] = x;
        }
        return this;
    }

    public float sum() {
        float s = 0.0f;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                s += this.get(i, j);
            }
        }
        return s;
    }

    public float norm1() {
        float f = 0.0f;
        for (int j = 0; j < this.n; ++j) {
            float s = 0.0f;
            for (int i = 0; i < this.m; ++i) {
                s += Math.abs(this.get(i, j));
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public float norm2() {
        return this.svd((boolean)false, (boolean)false).s[0];
    }

    public float norm() {
        return this.norm2();
    }

    public float normInf() {
        float[] f = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                f[n] = f[n] + Math.abs(this.get(i, j));
            }
        }
        return MathEx.max(f);
    }

    public float normFro() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                f = Math.hypot(f, this.get(i, j));
            }
        }
        return (float)f;
    }

    public float xAx(float[] x) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        if (this.n != x.length) {
            throw new IllegalArgumentException(String.format("Matrix: %d x %d, Vector: %d", this.m, this.n, x.length));
        }
        float[] Ax = this.mv(x);
        return MathEx.dot(x, Ax);
    }

    public float[] rowSums() {
        float[] x = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public float[] rowMeans() {
        float[] x = this.rowSums();
        int i = 0;
        while (i < this.m) {
            int n = i++;
            x[n] = x[n] / (float)this.n;
        }
        return x;
    }

    public float[] rowSds() {
        float[] x = new float[this.m];
        float[] x2 = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            int i = 0;
            while (i < this.m) {
                float a = this.get(i, j);
                int n = i;
                x[n] = x[n] + a;
                int n2 = i++;
                x2[n2] = x2[n2] + a * a;
            }
        }
        for (int i = 0; i < this.m; ++i) {
            float mu = x[i] / (float)this.n;
            float variance = Math.max(x2[i] / (float)this.n - mu * mu, 0.0f);
            x[i] = (float)Math.sqrt(variance);
        }
        return x;
    }

    public float[] colSums() {
        float[] x = new float[this.n];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public float[] colMeans() {
        float[] x = this.colSums();
        int j = 0;
        while (j < this.n) {
            int n = j++;
            x[n] = x[n] / (float)this.m;
        }
        return x;
    }

    public float[] colSds() {
        float[] x = new float[this.n];
        for (int j = 0; j < this.n; ++j) {
            float mu = 0.0f;
            float sumsq = 0.0f;
            for (int i = 0; i < this.m; ++i) {
                float a = this.get(i, j);
                mu += a;
                sumsq += a * a;
            }
            float variance = Math.max(sumsq / (float)this.m - (mu /= (float)this.m) * mu, 0.0f);
            x[j] = (float)Math.sqrt(variance);
        }
        return x;
    }

    public Matrix standardize() {
        float[] center = this.colMeans();
        float[] scale = this.colSds();
        return this.scale(center, scale);
    }

    public Matrix scale(float[] center, float[] scale) {
        if (center == null && scale == null) {
            throw new IllegalArgumentException("Both center and scale are null");
        }
        Matrix matrix = new Matrix(this.m, this.n);
        if (center == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) / scale[j]);
                }
            }
        } else if (scale == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) - center[j]);
                }
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, (this.get(i, j) - center[j]) / scale[j]);
                }
            }
        }
        return matrix;
    }

    public Matrix inverse() {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        Matrix lu = this.copy();
        Matrix inv = Matrix.eye(this.n);
        int[] ipiv = new int[this.n];
        if (this.isSymmetric()) {
            int info = LAPACK.engine.sysv(lu.layout(), this.uplo, this.n, this.n, lu.A, lu.ld, ipiv, inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("SYSV fails: " + info);
            }
        } else {
            int info = LAPACK.engine.gesv(lu.layout(), this.n, this.n, lu.A, lu.ld, ipiv, inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("GESV fails: " + info);
            }
        }
        return inv;
    }

    private void mv(Transpose trans, float alpha, FloatBuffer x, float beta, FloatBuffer y) {
        FloatBuffer A = FloatBuffer.wrap(this.A);
        if (this.uplo != null) {
            if (this.diag != null) {
                if ((double)alpha == 1.0 && beta == 0.0f && x == y) {
                    BLAS.engine.trmv(this.layout(), this.uplo, trans, this.diag, this.m, A, this.ld, y, 1);
                } else {
                    BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, A, this.ld, x, 1, beta, y, 1);
                }
            } else {
                BLAS.engine.symv(this.layout(), this.uplo, this.m, alpha, A, this.ld, x, 1, beta, y, 1);
            }
        } else {
            BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, A, this.ld, x, 1, beta, y, 1);
        }
    }

    @Override
    public void mv(Transpose trans, float alpha, float[] x, float beta, float[] y) {
        if (this.uplo != null) {
            if (this.diag != null) {
                if ((double)alpha == 1.0 && beta == 0.0f && x == y) {
                    BLAS.engine.trmv(this.layout(), this.uplo, trans, this.diag, this.m, this.A, this.ld, y, 1);
                } else {
                    BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
                }
            } else {
                BLAS.engine.symv(this.layout(), this.uplo, this.m, alpha, this.A, this.ld, x, 1, beta, y, 1);
            }
        } else {
            BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
        }
    }

    @Override
    public void mv(float[] work, int inputOffset, int outputOffset) {
        FloatBuffer xb = FloatBuffer.wrap(work, inputOffset, this.n);
        FloatBuffer yb = FloatBuffer.wrap(work, outputOffset, this.m);
        this.mv(Transpose.NO_TRANSPOSE, 1.0f, xb, 0.0f, yb);
    }

    @Override
    public void tv(float[] work, int inputOffset, int outputOffset) {
        FloatBuffer xb = FloatBuffer.wrap(work, inputOffset, this.m);
        FloatBuffer yb = FloatBuffer.wrap(work, outputOffset, this.n);
        this.mv(Transpose.TRANSPOSE, 1.0f, xb, 0.0f, yb);
    }

    public Matrix mm(Transpose transA, Matrix A, Transpose transB, Matrix B) {
        return this.mm(transA, A, transB, B, 1.0f, 0.0f);
    }

    public Matrix mm(Transpose transA, Matrix A, Transpose transB, Matrix B, float alpha, float beta) {
        if (A.isSymmetric() && transB == Transpose.NO_TRANSPOSE && B.layout() == this.layout()) {
            BLAS.engine.symm(this.layout(), Side.LEFT, A.uplo, this.m, this.n, alpha, A.A, A.ld, B.A, B.ld, beta, this.A, this.ld);
        } else if (B.isSymmetric() && transA == Transpose.NO_TRANSPOSE && A.layout() == this.layout()) {
            BLAS.engine.symm(this.layout(), Side.RIGHT, B.uplo, this.m, this.n, alpha, B.A, B.ld, A.A, A.ld, beta, this.A, this.ld);
        } else {
            if (this.layout() != A.layout()) {
                transA = Matrix.flip(transA);
                A = A.transpose();
            }
            if (this.layout() != B.layout()) {
                transB = Matrix.flip(transB);
                B = B.transpose();
            }
            int k = transA == Transpose.NO_TRANSPOSE ? A.n : A.m;
            BLAS.engine.gemm(this.layout(), transA, transB, this.m, this.n, k, alpha, A.A, A.ld, B.A, B.ld, beta, this.A, this.ld);
        }
        return this;
    }

    public Matrix ata() {
        Matrix C = new Matrix(this.n, this.n);
        C.mm(Transpose.TRANSPOSE, this, Transpose.NO_TRANSPOSE, this);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public Matrix aat() {
        Matrix C = new Matrix(this.m, this.m);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.TRANSPOSE, this);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public static Matrix adb(Transpose transA, Matrix A, float[] D, Transpose transB, Matrix B) {
        Matrix AD;
        int m = A.m;
        int n = A.n;
        if (transA == Transpose.NO_TRANSPOSE) {
            AD = new Matrix(m, n);
            for (int j = 0; j < n; ++j) {
                float dj = D[j];
                for (int i = 0; i < m; ++i) {
                    AD.set(i, j, dj * A.get(i, j));
                }
            }
        } else {
            AD = new Matrix(n, m);
            for (int j = 0; j < m; ++j) {
                float dj = D[j];
                for (int i = 0; i < n; ++i) {
                    AD.set(i, j, dj * A.get(j, i));
                }
            }
        }
        return transB == Transpose.NO_TRANSPOSE ? AD.mm(B) : AD.mt(B);
    }

    public Matrix mm(Matrix B) {
        if (this.n != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.m, B.n);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.NO_TRANSPOSE, B);
        return C;
    }

    public Matrix mt(Matrix B) {
        if (this.n != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.m, B.m);
        C.mm(Transpose.NO_TRANSPOSE, this, Transpose.TRANSPOSE, B);
        return C;
    }

    public Matrix tm(Matrix B) {
        if (this.m != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.n, B.n);
        C.mm(Transpose.TRANSPOSE, this, Transpose.NO_TRANSPOSE, B);
        return C;
    }

    public Matrix tt(Matrix B) {
        if (this.m != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        Matrix C = new Matrix(this.n, B.m);
        C.mm(Transpose.TRANSPOSE, this, Transpose.TRANSPOSE, B);
        return C;
    }

    public LU lu() {
        return this.lu(false);
    }

    public LU lu(boolean overwrite) {
        Matrix lu = overwrite ? this : this.copy();
        int[] ipiv = new int[Math.min(this.m, this.n)];
        int info = LAPACK.engine.getrf(lu.layout(), lu.m, lu.n, lu.A, lu.ld, ipiv);
        if (info < 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        lu.uplo = null;
        return new LU(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        return this.cholesky(false);
    }

    public Cholesky cholesky(boolean overwrite) {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        Matrix lu = overwrite ? this : this.copy();
        int info = LAPACK.engine.potrf(lu.layout(), lu.uplo, lu.n, lu.A, lu.ld);
        if (info != 0) {
            logger.error("LAPACK POTRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK POTRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public QR qr() {
        return this.qr(false);
    }

    public QR qr(boolean overwrite) {
        Matrix qr = overwrite ? this : this.copy();
        float[] tau = new float[Math.min(this.m, this.n)];
        int info = LAPACK.engine.geqrf(qr.layout(), qr.m, qr.n, qr.A, qr.ld, tau);
        if (info != 0) {
            logger.error("LAPACK GEQRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEQRF error code: " + info);
        }
        qr.uplo = null;
        return new QR(qr, tau);
    }

    public SVD svd() {
        return this.svd(true, false);
    }

    public SVD svd(boolean vectors, boolean overwrite) {
        Matrix W;
        int k = Math.min(this.m, this.n);
        float[] s = new float[k];
        Matrix matrix = W = overwrite ? this : this.copy();
        if (vectors) {
            Matrix U = new Matrix(this.m, k);
            Matrix VT = new Matrix(k, this.n);
            int info = LAPACK.engine.gesdd(W.layout(), SVDJob.COMPACT, W.m, W.n, W.A, W.ld, s, U.A, U.ld, VT.A, VT.ld);
            if (info != 0) {
                logger.error("LAPACK GESDD with COMPACT error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK GESDD error code: " + info);
            }
            return new SVD(s, U, VT.transpose());
        }
        Matrix U = new Matrix(1, 1);
        Matrix VT = new Matrix(1, 1);
        int info = LAPACK.engine.gesdd(W.layout(), SVDJob.NO_VECTORS, W.m, W.n, W.A, W.ld, s, U.A, U.ld, VT.A, VT.ld);
        if (info != 0) {
            logger.error("LAPACK GESDD with NO_VECTORS error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GESDD error code: " + info);
        }
        return new SVD(this.m, this.n, s);
    }

    public EVD eigen() {
        return this.eigen(false, true, false);
    }

    public EVD eigen(boolean vl, boolean vr, boolean overwrite) {
        Matrix eig;
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        Matrix matrix = eig = overwrite ? this : this.copy();
        if (this.isSymmetric()) {
            float[] w = new float[this.n];
            int info = LAPACK.engine.syevd(eig.layout(), vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, eig.uplo, this.n, eig.A, eig.ld, w);
            if (info != 0) {
                logger.error("LAPACK SYEV error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK SYEV error code: " + info);
            }
            eig.uplo = null;
            return new EVD(w, vr ? eig : null);
        }
        float[] wr = new float[this.n];
        float[] wi = new float[this.n];
        Matrix Vl = vl ? new Matrix(this.n, this.n) : new Matrix(1, 1);
        Matrix Vr = vr ? new Matrix(this.n, this.n) : new Matrix(1, 1);
        int info = LAPACK.engine.geev(eig.layout(), vl ? EVDJob.VECTORS : EVDJob.NO_VECTORS, vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, this.n, eig.A, eig.ld, wr, wi, Vl.A, Vl.ld, Vr.A, Vr.ld);
        if (info != 0) {
            logger.error("LAPACK GEEV error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEEV error code: " + info);
        }
        return new EVD(wr, wi, vl ? Vl : null, vr ? Vr : null);
    }

    private static class RowMajor
    extends Matrix {
        RowMajor(int m, int n, int ld, float[] A) {
            super(m, n, ld, A);
        }

        @Override
        public Layout layout() {
            return Layout.ROW_MAJOR;
        }

        @Override
        protected int index(int i, int j) {
            return i * this.ld + j;
        }
    }

    public static class SVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final int m;
        public final int n;
        public final float[] s;
        public final Matrix U;
        public final Matrix V;
        private transient Matrix Ur;

        public SVD(int m, int n, float[] s) {
            this.m = m;
            this.n = n;
            this.s = s;
            this.U = null;
            this.V = null;
        }

        public SVD(float[] s, Matrix U, Matrix V) {
            this.m = U.m;
            this.n = V.m;
            this.s = s;
            this.U = U;
            this.V = V;
        }

        public Matrix diag() {
            Matrix S = new Matrix(this.m, this.n);
            for (int i = 0; i < this.s.length; ++i) {
                S.set(i, i, this.s[i]);
            }
            return S;
        }

        public float norm() {
            return this.s[0];
        }

        private float rcond() {
            return (float)(0.5 * Math.sqrt(this.m + this.n + 1) * (double)this.s[0] * MathEx.EPSILON);
        }

        public int rank() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            int r = 0;
            float tol = this.rcond();
            for (float si : this.s) {
                if (!(si > tol)) continue;
                ++r;
            }
            return r;
        }

        public int nullity() {
            return Math.min(this.m, this.n) - this.rank();
        }

        public float condition() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            return this.s[0] <= 0.0f || this.s[this.s.length - 1] <= 0.0f ? Float.POSITIVE_INFINITY : this.s[0] / this.s[this.s.length - 1];
        }

        public Matrix range() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.U == null) {
                throw new IllegalStateException("The left singular vectors are not available.");
            }
            int r = this.rank();
            if (r == 0) {
                return null;
            }
            Matrix R = new Matrix(this.m, r);
            for (int j = 0; j < r; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    R.set(i, j, this.U.get(i, j));
                }
            }
            return R;
        }

        public Matrix nullspace() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.V == null) {
                throw new IllegalStateException("The right singular vectors are not available.");
            }
            int nr = this.nullity();
            if (nr == 0) {
                return null;
            }
            Matrix N = new Matrix(this.n, nr);
            for (int j = 0; j < nr; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    N.set(i, j, this.V.get(i, this.n - j - 1));
                }
            }
            return N;
        }

        public Matrix pinv() {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            int k = this.s.length;
            float[] sigma = new float[k];
            int r = this.rank();
            for (int i = 0; i < r; ++i) {
                sigma[i] = 1.0f / this.s[i];
            }
            return Matrix.adb(Transpose.NO_TRANSPOSE, this.V, sigma, Transpose.TRANSPOSE, this.U);
        }

        public float[] solve(float[] b) {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            if (b.length != this.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.m, this.n, b.length));
            }
            int r = this.rank();
            if (this.Ur == null) {
                this.Ur = r == this.U.ncol() ? this.U : this.U.submatrix(0, 0, this.m - 1, r - 1);
            }
            float[] Utb = new float[this.s.length];
            this.Ur.tv(b, Utb);
            for (int i = 0; i < r; ++i) {
                int n = i;
                Utb[n] = Utb[n] / this.s[i];
            }
            return this.V.mv(Utb);
        }
    }

    public static class LU
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final Matrix lu;
        public final int[] ipiv;
        public final int info;

        public LU(Matrix lu, int[] ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public float det() {
            int j;
            int m = this.lu.m;
            int n = this.lu.n;
            if (m != n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", m, n));
            }
            float d = 1.0f;
            for (j = 0; j < n; ++j) {
                d *= this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv[j]) continue;
                d = -d;
            }
            return d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            Matrix x = Matrix.column(b);
            this.solve(x);
            return x.A;
        }

        public void solve(Matrix B) {
            if (this.lu.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.lu.m, this.lu.n));
            }
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.getrs(this.lu.layout(), Transpose.NO_TRANSPOSE, this.lu.n, B.n, this.lu.A, this.lu.ld, this.ipiv, B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }

    public static class Cholesky
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final Matrix lu;

        public Cholesky(Matrix lu) {
            if (lu.nrow() != lu.ncol()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public float det() {
            int n = this.lu.n;
            float d = 1.0f;
            for (int i = 0; i < n; ++i) {
                d *= this.lu.get(i, i);
            }
            return d * d;
        }

        public float logdet() {
            int n = this.lu.n;
            float d = 0.0f;
            for (int i = 0; i < n; ++i) {
                d += (float)Math.log(this.lu.get(i, i));
            }
            return 2.0f * d;
        }

        public Matrix inverse() {
            Matrix inv = Matrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            Matrix x = Matrix.column(b);
            this.solve(x);
            return x.A;
        }

        public void solve(Matrix B) {
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.potrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.A, this.lu.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }

    public static class QR
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final Matrix qr;
        public final float[] tau;

        public QR(Matrix qr, float[] tau) {
            this.qr = qr;
            this.tau = tau;
        }

        public Cholesky CholeskyOfAtA() {
            int n = this.qr.n;
            Matrix L = new Matrix(n, n);
            for (int i = 0; i < n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    L.set(i, j, this.qr.get(j, i));
                }
            }
            L.uplo(UPLO.LOWER);
            return new Cholesky(L);
        }

        public Matrix R() {
            int n = this.qr.n;
            Matrix R = Matrix.diag(this.tau);
            for (int i = 0; i < n; ++i) {
                for (int j = i; j < n; ++j) {
                    R.set(i, j, this.qr.get(i, j));
                }
            }
            return R;
        }

        public Matrix Q() {
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            Matrix Q = this.qr.copy();
            int info = LAPACK.engine.orgqr(this.qr.layout(), m, n, k, Q.A, this.qr.ld, this.tau);
            if (info != 0) {
                logger.error("LAPACK ORGRQ error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK ORGRQ error code: " + info);
            }
            return Q;
        }

        public float[] solve(float[] b) {
            if (b.length != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.qr.m, this.qr.n, b.length));
            }
            Matrix x = Matrix.column(b);
            this.solve(x);
            return Arrays.copyOf(x.A, this.qr.n);
        }

        public void solve(Matrix B) {
            if (B.m != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.qr.nrow(), this.qr.nrow(), B.nrow(), B.ncol()));
            }
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            int info = LAPACK.engine.ormqr(this.qr.layout(), Side.LEFT, Transpose.TRANSPOSE, B.nrow(), B.ncol(), k, this.qr.A, this.qr.ld, this.tau, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK ORMQR error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK ORMQR error code: " + info);
            }
            info = LAPACK.engine.trtrs(this.qr.layout(), UPLO.UPPER, Transpose.NO_TRANSPOSE, Diag.NON_UNIT, this.qr.n, B.n, this.qr.A, this.qr.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK TRTRS error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK TRTRS error code: " + info);
            }
        }
    }

    public static class EVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final float[] wr;
        public final float[] wi;
        public final Matrix Vl;
        public final Matrix Vr;

        public EVD(float[] w, Matrix V) {
            this.wr = w;
            this.wi = null;
            this.Vl = V;
            this.Vr = V;
        }

        public EVD(float[] wr, float[] wi, Matrix Vl, Matrix Vr) {
            this.wr = wr;
            this.wi = wi;
            this.Vl = Vl;
            this.Vr = Vr;
        }

        public Matrix diag() {
            Matrix D = Matrix.diag(this.wr);
            if (this.wi != null) {
                int n = this.wr.length;
                for (int i = 0; i < n; ++i) {
                    if (this.wi[i] > 0.0f) {
                        D.set(i, i + 1, this.wi[i]);
                        continue;
                    }
                    if (!(this.wi[i] < 0.0f)) continue;
                    D.set(i, i - 1, this.wi[i]);
                }
            }
            return D;
        }

        public EVD sort() {
            int n = this.wr.length;
            float[] w = new float[n];
            if (this.wi != null) {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i] + this.wi[i] * this.wi[i]);
                }
            } else {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i]);
                }
            }
            int[] index = QuickSort.sort(w);
            float[] wr2 = new float[n];
            for (int j = 0; j < n; ++j) {
                wr2[j] = this.wr[index[j]];
            }
            float[] wi2 = null;
            if (this.wi != null) {
                wi2 = new float[n];
                for (int j = 0; j < n; ++j) {
                    wi2[j] = this.wi[index[j]];
                }
            }
            Matrix Vl2 = null;
            if (this.Vl != null) {
                int m = this.Vl.m;
                Vl2 = new Matrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i = 0; i < m; ++i) {
                        Vl2.set(i, j, this.Vl.get(i, index[j]));
                    }
                }
            }
            Matrix Vr2 = null;
            if (this.Vr != null) {
                int m = this.Vr.m;
                Vr2 = new Matrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i = 0; i < m; ++i) {
                        Vr2.set(i, j, this.Vr.get(i, index[j]));
                    }
                }
            }
            return new EVD(wr2, wi2, Vl2, Vr2);
        }
    }
}

