/*
 * Decompiled with CFR 0.152.
 */
package smile.math.matrix;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import smile.math.MathEx;
import smile.math.blas.BLAS;
import smile.math.blas.Diag;
import smile.math.blas.EVDJob;
import smile.math.blas.LAPACK;
import smile.math.blas.Layout;
import smile.math.blas.SVDJob;
import smile.math.blas.Side;
import smile.math.blas.Transpose;
import smile.math.blas.UPLO;
import smile.math.matrix.SMatrix;
import smile.sort.QuickSort;
import smile.stat.distribution.Distribution;
import smile.stat.distribution.GaussianDistribution;

public class FloatMatrix
extends SMatrix {
    private static final long serialVersionUID = 2L;
    private static final Logger logger = LoggerFactory.getLogger(FloatMatrix.class);
    transient FloatBuffer A;
    transient int ld;
    int m;
    int n;
    UPLO uplo;
    Diag diag;

    public FloatMatrix(int m, int n) {
        this(m, n, 0.0f);
    }

    public FloatMatrix(int m, int n, float a) {
        if (m <= 0 || n <= 0) {
            throw new IllegalArgumentException(String.format("Invalid matrix size: %d x %d", m, n));
        }
        this.m = m;
        this.n = n;
        this.ld = FloatMatrix.ld(m);
        float[] array = new float[this.ld * n];
        if ((double)a != 0.0) {
            Arrays.fill(array, a);
        }
        this.A = FloatBuffer.wrap(array);
    }

    public FloatMatrix(int m, int n, float[][] A) {
        this(m, n);
        for (int i = 0; i < m; ++i) {
            for (int j = 0; j < n; ++j) {
                this.set(i, j, A[i][j]);
            }
        }
    }

    public FloatMatrix(float[][] A) {
        this(A.length, A[0].length, A);
    }

    public FloatMatrix(float[] A) {
        this(A, 0, A.length);
    }

    public FloatMatrix(float[] A, int offset, int length) {
        this.m = length;
        this.n = 1;
        this.ld = length;
        this.A = FloatBuffer.wrap(A, offset, length);
    }

    public FloatMatrix(int m, int n, int ld, FloatBuffer A) {
        if (this.layout() == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (this.layout() == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        this.m = m;
        this.n = n;
        this.ld = ld;
        this.A = A;
    }

    public static FloatMatrix of(Layout layout, int m, int n) {
        if (layout == Layout.COL_MAJOR) {
            int ld = FloatMatrix.ld(m);
            int size = ld * n;
            return FloatMatrix.of(layout, m, n, ld, FloatBuffer.allocate(size));
        }
        int ld = FloatMatrix.ld(n);
        int size = ld * m;
        return FloatMatrix.of(layout, m, n, ld, FloatBuffer.allocate(size));
    }

    public static FloatMatrix of(Layout layout, int m, int n, int ld, FloatBuffer A) {
        if (layout == Layout.COL_MAJOR && ld < m) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for COL_MAJOR: %d < %d", ld, m));
        }
        if (layout == Layout.ROW_MAJOR && ld < n) {
            throw new IllegalArgumentException(String.format("Invalid leading dimension for ROW_MAJOR: %d < %d", ld, n));
        }
        if (layout == Layout.COL_MAJOR) {
            return new FloatMatrix(m, n, ld, A);
        }
        return new FloatMatrix(m, n, ld, A){

            @Override
            public Layout layout() {
                return Layout.ROW_MAJOR;
            }

            @Override
            protected int index(int i, int j) {
                return i * this.ld + j + this.A.position();
            }

            @Override
            public FloatMatrix transpose() {
                return new FloatMatrix(this.n, this.m, this.ld, this.A);
            }
        };
    }

    public static FloatMatrix eye(int n) {
        FloatMatrix matrix = new FloatMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            matrix.set(i, i, 1.0f);
        }
        return matrix;
    }

    public static FloatMatrix eye(int m, int n) {
        FloatMatrix matrix = new FloatMatrix(m, n);
        int k = Math.min(m, n);
        for (int i = 0; i < k; ++i) {
            matrix.set(i, i, 1.0f);
        }
        return matrix;
    }

    public static FloatMatrix randn(int m, int n) {
        return FloatMatrix.rand(m, n, GaussianDistribution.getInstance());
    }

    public static FloatMatrix rand(int m, int n, Distribution distribution) {
        FloatMatrix matrix = new FloatMatrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, (float)distribution.rand());
            }
        }
        return matrix;
    }

    public static FloatMatrix rand(int m, int n, float lo, float hi) {
        FloatMatrix matrix = new FloatMatrix(m, n);
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < m; ++i) {
                matrix.set(i, j, (float)MathEx.random(lo, hi));
            }
        }
        return matrix;
    }

    public static FloatMatrix diag(float[] diag) {
        int n = diag.length;
        FloatMatrix D = new FloatMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            D.set(i, i, diag[i]);
        }
        return D;
    }

    public static FloatMatrix toeplitz(float[] a) {
        int n = a.length;
        FloatMatrix toeplitz = new FloatMatrix(n, n);
        toeplitz.uplo(UPLO.LOWER);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, a[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, a[j - i]);
            }
        }
        return toeplitz;
    }

    public static FloatMatrix toeplitz(float[] kl, float[] ku) {
        if (kl.length != ku.length - 1) {
            throw new IllegalArgumentException(String.format("Invalid subdiagonals and superdiagonals size: %d != %d - 1", kl.length, ku.length));
        }
        int n = kl.length;
        FloatMatrix toeplitz = new FloatMatrix(n, n);
        for (int i = 0; i < n; ++i) {
            int j;
            for (j = 0; j < i; ++j) {
                toeplitz.set(i, j, kl[i - j]);
            }
            for (j = i; j < n; ++j) {
                toeplitz.set(i, j, ku[j - i]);
            }
        }
        return toeplitz;
    }

    private static int ld(int n) {
        int elementSize = 4;
        if (n <= 256 / elementSize) {
            return n;
        }
        return ((n * elementSize + 511) / 512 * 512 + 64) / elementSize;
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        out.defaultWriteObject();
        if (this.layout() == Layout.COL_MAJOR) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    out.writeFloat(this.get(i, j));
                }
            }
        } else {
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    out.writeFloat(this.get(i, j));
                }
            }
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.A = FloatBuffer.wrap(new float[this.m * this.n]);
        if (this.layout() == Layout.COL_MAJOR) {
            this.ld = this.m;
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    this.set(i, j, in.readFloat());
                }
            }
        } else {
            this.ld = this.n;
            for (int i = 0; i < this.m; ++i) {
                for (int j = 0; j < this.n; ++j) {
                    this.set(i, j, in.readFloat());
                }
            }
        }
    }

    @Override
    public int nrows() {
        return this.m;
    }

    @Override
    public int ncols() {
        return this.n;
    }

    @Override
    public long size() {
        return this.m * this.n;
    }

    public Layout layout() {
        return Layout.COL_MAJOR;
    }

    public int ld() {
        return this.ld;
    }

    public boolean isSubmatrix() {
        return this.A.position() != 0 || this.A.limit() != this.A.capacity();
    }

    public boolean isSymmetric() {
        return this.uplo != null && this.diag == null;
    }

    public FloatMatrix uplo(UPLO uplo) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.uplo = uplo;
        return this;
    }

    public UPLO uplo() {
        return this.uplo;
    }

    public FloatMatrix triangular(Diag diag) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        this.diag = diag;
        return this;
    }

    public Diag triangular() {
        return this.diag;
    }

    public FloatMatrix clone() {
        FloatMatrix matrix = new FloatMatrix(this.m, this.n);
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                matrix.set(i, j, this.get(i, j));
            }
        }
        if (this.m == this.n) {
            matrix.uplo(this.uplo);
            matrix.triangular(this.diag);
        }
        return matrix;
    }

    public float[][] toArray() {
        float[][] array = new float[this.m][this.n];
        for (int i = 0; i < this.m; ++i) {
            for (int j = 0; j < this.n; ++j) {
                array[i][j] = this.get(i, j);
            }
        }
        return array;
    }

    public double[] row(int i) {
        double[] x = new double[this.n];
        for (int j = 0; j < this.n; ++j) {
            x[j] = this.get(i, j);
        }
        return x;
    }

    public double[] col(int j) {
        double[] x = new double[this.m];
        for (int i = 0; i < this.m; ++i) {
            x[i] = this.get(i, j);
        }
        return x;
    }

    public FloatMatrix row(int ... rows) {
        FloatMatrix x = new FloatMatrix(rows.length, this.n);
        for (int i = 0; i < rows.length; ++i) {
            int row = rows[i];
            for (int j = 0; j < this.n; ++j) {
                x.set(i, j, this.get(row, j));
            }
        }
        return x;
    }

    public FloatMatrix col(int ... cols) {
        FloatMatrix x = new FloatMatrix(this.m, cols.length);
        for (int j = 0; j < cols.length; ++j) {
            int col = cols[j];
            for (int i = 0; i < this.m; ++i) {
                x.set(i, j, this.get(i, col));
            }
        }
        return x;
    }

    public FloatMatrix submatrix(int i, int j, int k, int l) {
        if (i < 0 || i >= this.m || k < i || k >= this.m || j < 0 || j >= this.n || l < j || l >= this.n) {
            throw new IllegalArgumentException(String.format("Invalid submatrix range (%d:%d, %d:%d) of %d x %d", i, k, j, l, this.m, this.n));
        }
        int offset = this.index(i, j);
        int length = this.index(k, l) - offset + 1;
        FloatBuffer B = FloatBuffer.wrap(this.A.array(), offset, length);
        return FloatMatrix.of(this.layout(), k - i + 1, l - j + 1, this.ld, B);
    }

    public void fill(float x) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, x);
            }
        }
    }

    public FloatMatrix transpose() {
        return FloatMatrix.of(Layout.ROW_MAJOR, this.n, this.m, this.ld, this.A);
    }

    public boolean equals(Object o) {
        if (o == null || !(o instanceof FloatMatrix)) {
            return false;
        }
        return this.equals((FloatMatrix)o, 1.0E-7f);
    }

    public boolean equals(FloatMatrix o, float eps) {
        if (this.m != o.m || this.n != o.n) {
            return false;
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (MathEx.isZero(this.get(i, j) - o.get(i, j), eps)) continue;
                return false;
            }
        }
        return true;
    }

    protected int index(int i, int j) {
        return j * this.ld + i + this.A.position();
    }

    @Override
    public float get(int i, int j) {
        return this.A.get(this.index(i, j));
    }

    @Override
    public FloatMatrix set(int i, int j, float x) {
        this.A.put(this.index(i, j), x);
        return this;
    }

    public FloatMatrix set(int i, int j, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.set(i + ii, j + jj, B.get(ii, jj));
            }
        }
        return this;
    }

    public float add(int i, int j, float b) {
        int k = this.index(i, j);
        float y = this.A.get(k) + b;
        this.A.put(k, y);
        return y;
    }

    public float sub(int i, int j, float b) {
        int k = this.index(i, j);
        float y = this.A.get(k) - b;
        this.A.put(k, y);
        return y;
    }

    public float mul(int i, int j, float b) {
        int k = this.index(i, j);
        float y = this.A.get(k) * b;
        this.A.put(k, y);
        return y;
    }

    public float div(int i, int j, float b) {
        int k = this.index(i, j);
        float y = this.A.get(k) / b;
        this.A.put(k, y);
        return y;
    }

    public FloatMatrix add(float b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, b);
            }
        }
        return this;
    }

    public FloatMatrix sub(float b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, b);
            }
        }
        return this;
    }

    public FloatMatrix mul(float b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, b);
            }
        }
        return this;
    }

    public FloatMatrix div(float b) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, b);
            }
        }
        return this;
    }

    public FloatMatrix add(int i, int j, float alpha, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.add(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public FloatMatrix sub(int i, int j, float alpha, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.sub(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public FloatMatrix mul(int i, int j, float alpha, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.mul(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public FloatMatrix div(int i, int j, float alpha, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.div(i + ii, j + jj, alpha * B.get(ii, jj));
            }
        }
        return this;
    }

    public FloatMatrix add(FloatMatrix B) {
        return this.add(1.0f, B);
    }

    public FloatMatrix sub(FloatMatrix B) {
        return this.sub(1.0f, B);
    }

    public FloatMatrix mul(FloatMatrix B) {
        return this.mul(1.0f, B);
    }

    public FloatMatrix div(FloatMatrix B) {
        return this.div(1.0f, B);
    }

    public FloatMatrix add(float alpha, FloatMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.add(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix sub(float alpha, FloatMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.sub(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix mul(float alpha, FloatMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.mul(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix div(float alpha, FloatMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.div(i, j, alpha * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix add(float alpha, FloatMatrix A, float beta, FloatMatrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) + beta * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix sub(float alpha, FloatMatrix A, float beta, FloatMatrix B) {
        return this.add(alpha, A, -beta, B);
    }

    public FloatMatrix mul(float alpha, FloatMatrix A, FloatMatrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix div(float alpha, FloatMatrix A, FloatMatrix B) {
        if (this.m != A.m || this.n != A.n) {
            throw new IllegalArgumentException("Matrix A is not of same size.");
        }
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * A.get(i, j) / B.get(i, j));
            }
        }
        return this;
    }

    public double add(int i, int j, float alpha, float beta) {
        int k = this.index(i, j);
        float y = alpha * this.A.get(k) + beta;
        this.A.put(k, y);
        return y;
    }

    public FloatMatrix add(int i, int j, float alpha, float beta, FloatMatrix B) {
        for (int jj = 0; jj < B.n; ++jj) {
            for (int ii = 0; ii < B.m; ++ii) {
                this.add(i + ii, j + jj, alpha, beta * B.get(ii, jj));
            }
        }
        return this;
    }

    public FloatMatrix add(float alpha, float beta, FloatMatrix B) {
        if (this.m != B.m || this.n != B.n) {
            throw new IllegalArgumentException("Matrix B is not of same size.");
        }
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                this.set(i, j, alpha * this.get(i, j) + beta * B.get(i, j));
            }
        }
        return this;
    }

    public FloatMatrix add(float alpha, float[] x, float[] y) {
        if (this.m != x.length || this.n != y.length) {
            throw new IllegalArgumentException("Matrix is not of same size.");
        }
        if (this.isSymmetric() && x == y) {
            BLAS.engine.syr(this.layout(), this.uplo, this.m, alpha, FloatBuffer.wrap(x), 1, this.A, this.ld);
        } else {
            BLAS.engine.ger(this.layout(), this.m, this.n, alpha, FloatBuffer.wrap(x), 1, FloatBuffer.wrap(y), 1, this.A, this.ld);
        }
        return this;
    }

    public FloatMatrix replaceNaN(float x) {
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                if (!Float.isNaN(this.get(i, j))) continue;
                this.set(i, j, x);
            }
        }
        return this;
    }

    public float sum() {
        float s = 0.0f;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                s += this.get(i, j);
            }
        }
        return s;
    }

    public float norm1() {
        float f = 0.0f;
        for (int j = 0; j < this.n; ++j) {
            float s = 0.0f;
            for (int i = 0; i < this.m; ++i) {
                s += Math.abs(this.get(i, j));
            }
            f = Math.max(f, s);
        }
        return f;
    }

    public float norm2() {
        return this.svd((boolean)false, (boolean)false).s[0];
    }

    public float norm() {
        return this.norm2();
    }

    public float normInf() {
        float[] f = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                f[n] = f[n] + Math.abs(this.get(i, j));
            }
        }
        return MathEx.max(f);
    }

    public float normFro() {
        double f = 0.0;
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                f = Math.hypot(f, this.get(i, j));
            }
        }
        return (float)f;
    }

    public float xAx(float[] x) {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        if (this.n != x.length) {
            throw new IllegalArgumentException(String.format("Matrix: %d x %d, Vector: %d", this.m, this.n, x.length));
        }
        int n = x.length;
        float s = 0.0f;
        for (int j = 0; j < n; ++j) {
            for (int i = 0; i < n; ++i) {
                s += this.get(i, j) * x[i] * x[j];
            }
        }
        return s;
    }

    public float[] rowSums() {
        float[] x = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public float[] rowMeans() {
        float[] x = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = i;
                x[n] = x[n] + this.get(i, j);
            }
        }
        int i = 0;
        while (i < this.m) {
            int n = i++;
            x[n] = x[n] / (float)this.n;
        }
        return x;
    }

    public float[] rowSds() {
        float[] x = new float[this.m];
        float[] x2 = new float[this.m];
        for (int j = 0; j < this.n; ++j) {
            int i = 0;
            while (i < this.m) {
                float a = this.get(i, j);
                int n = i;
                x[n] = x[n] + a;
                int n2 = i++;
                x2[n2] = x2[n2] + a * a;
            }
        }
        for (int i = 0; i < this.m; ++i) {
            float mu = x[i] / (float)this.n;
            x[i] = (float)Math.sqrt(x2[i] / (float)this.n - mu * mu);
        }
        return x;
    }

    public float[] colSums() {
        float[] x = new float[this.n];
        for (int j = 0; j < this.n; ++j) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
        }
        return x;
    }

    public float[] colMeans() {
        float[] x = new float[this.n];
        int j = 0;
        while (j < this.n) {
            for (int i = 0; i < this.m; ++i) {
                int n = j;
                x[n] = x[n] + this.get(i, j);
            }
            int n = j++;
            x[n] = x[n] / (float)this.m;
        }
        return x;
    }

    public float[] colSds() {
        float[] x = new float[this.n];
        for (int j = 0; j < this.n; ++j) {
            float mu = 0.0f;
            float sumsq = 0.0f;
            for (int i = 0; i < this.m; ++i) {
                float a = this.get(i, j);
                mu += a;
                sumsq += a * a;
            }
            x[j] = (float)Math.sqrt(sumsq / (float)this.m - (mu /= (float)this.m) * mu);
        }
        return x;
    }

    public FloatMatrix scale() {
        float[] center = this.colMeans();
        float[] scale = this.colSds();
        return this.scale(center, scale);
    }

    public FloatMatrix scale(float[] center, float[] scale) {
        if (center == null && scale == null) {
            throw new IllegalArgumentException("Both center and scale are null");
        }
        FloatMatrix matrix = new FloatMatrix(this.m, this.n);
        if (center == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) / scale[j]);
                }
            }
        } else if (scale == null) {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, this.get(i, j) - center[j]);
                }
            }
        } else {
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    matrix.set(i, j, (this.get(i, j) - center[j]) / scale[j]);
                }
            }
        }
        return matrix;
    }

    public FloatMatrix inverse() {
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        FloatMatrix lu = this.clone();
        FloatMatrix inv = FloatMatrix.eye(this.n);
        int[] ipiv = new int[this.n];
        if (this.isSymmetric()) {
            int info = LAPACK.engine.sysv(lu.layout(), this.uplo, this.n, this.n, lu.A, lu.ld, IntBuffer.wrap(ipiv), inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("SYSV fails: " + info);
            }
        } else {
            int info = LAPACK.engine.gesv(lu.layout(), this.n, this.n, lu.A, lu.ld, IntBuffer.wrap(ipiv), inv.A, inv.ld);
            if (info != 0) {
                throw new ArithmeticException("GESV fails: " + info);
            }
        }
        return inv;
    }

    public void mv(Transpose trans, float alpha, FloatBuffer x, float beta, FloatBuffer y) {
        if (this.uplo != null) {
            if (this.diag != null) {
                if ((double)alpha == 1.0 && (double)beta == 0.0 && x == y) {
                    BLAS.engine.trmv(this.layout(), this.uplo, trans, this.diag, this.m, this.A, this.ld, y, 1);
                } else {
                    BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
                }
            } else {
                BLAS.engine.symv(this.layout(), this.uplo, this.m, alpha, this.A, this.ld, x, 1, beta, y, 1);
            }
        } else {
            BLAS.engine.gemv(this.layout(), trans, this.m, this.n, alpha, this.A, this.ld, x, 1, beta, y, 1);
        }
    }

    @Override
    public void mv(Transpose trans, float alpha, float[] x, float beta, float[] y) {
        this.mv(trans, alpha, FloatBuffer.wrap(x), beta, FloatBuffer.wrap(y));
    }

    @Override
    public void mv(float[] work, int inputOffset, int outputOffset) {
        FloatBuffer xb = FloatBuffer.wrap(work, inputOffset, this.n);
        FloatBuffer yb = FloatBuffer.wrap(work, outputOffset, this.m);
        this.mv(Transpose.NO_TRANSPOSE, 1.0f, xb, 0.0f, yb);
    }

    @Override
    public void tv(float[] work, int inputOffset, int outputOffset) {
        FloatBuffer xb = FloatBuffer.wrap(work, inputOffset, this.m);
        FloatBuffer yb = FloatBuffer.wrap(work, outputOffset, this.n);
        this.mv(Transpose.TRANSPOSE, 1.0f, xb, 0.0f, yb);
    }

    private Transpose flip(Transpose trans) {
        return trans == Transpose.NO_TRANSPOSE ? Transpose.TRANSPOSE : Transpose.NO_TRANSPOSE;
    }

    public void mm(Transpose transA, Transpose transB, float alpha, FloatMatrix B, float beta, FloatMatrix C) {
        if (this.isSymmetric() && transB == Transpose.NO_TRANSPOSE && B.layout() == C.layout()) {
            BLAS.engine.symm(C.layout(), Side.LEFT, this.uplo, C.m, C.n, alpha, this.A, this.ld, B.A, B.ld, beta, C.A, C.ld);
        } else if (B.isSymmetric() && transA == Transpose.NO_TRANSPOSE && this.layout() == C.layout()) {
            BLAS.engine.symm(C.layout(), Side.RIGHT, B.uplo, C.m, C.n, alpha, B.A, B.ld, this.A, this.ld, beta, C.A, C.ld);
        } else {
            if (C.layout() != this.layout()) {
                transA = this.flip(transA);
            }
            if (C.layout() != B.layout()) {
                transB = this.flip(transB);
            }
            int k = transA == Transpose.NO_TRANSPOSE ? this.n : this.m;
            BLAS.engine.gemm(this.layout(), transA, transB, C.m, C.n, k, alpha, this.A, this.ld, B.A, B.ld, beta, C.A, C.ld);
        }
    }

    public FloatMatrix ata() {
        FloatMatrix C = new FloatMatrix(this.n, this.n);
        this.mm(Transpose.TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0f, this, 0.0f, C);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public FloatMatrix aat() {
        FloatMatrix C = new FloatMatrix(this.m, this.m);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, 1.0f, this, 0.0f, C);
        C.uplo(UPLO.LOWER);
        return C;
    }

    public FloatMatrix adb(Transpose transA, Transpose transB, FloatMatrix B, float[] diag) {
        FloatMatrix C;
        if (transA == Transpose.NO_TRANSPOSE) {
            C = new FloatMatrix(this.m, this.n);
            for (int j = 0; j < this.n; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    C.set(i, j, diag[j] * this.get(i, j));
                }
            }
        } else {
            C = new FloatMatrix(this.n, this.m);
            for (int j = 0; j < this.m; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    C.set(i, j, diag[j] * this.get(j, i));
                }
            }
        }
        return transB == Transpose.NO_TRANSPOSE ? C.mm(B) : C.mt(B);
    }

    public FloatMatrix mm(FloatMatrix B) {
        if (this.n != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        FloatMatrix C = new FloatMatrix(this.m, B.n);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0f, B, 0.0f, C);
        return C;
    }

    public FloatMatrix mt(FloatMatrix B) {
        if (this.n != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        FloatMatrix C = new FloatMatrix(this.m, B.m);
        this.mm(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, 1.0f, B, 0.0f, C);
        return C;
    }

    public FloatMatrix tm(FloatMatrix B) {
        if (this.m != B.m) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B: %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        FloatMatrix C = new FloatMatrix(this.n, B.n);
        this.mm(Transpose.TRANSPOSE, Transpose.NO_TRANSPOSE, 1.0f, B, 0.0f, C);
        return C;
    }

    public FloatMatrix tt(FloatMatrix B) {
        if (this.m != B.n) {
            throw new IllegalArgumentException(String.format("Matrix multiplication A' * B': %d x %d vs %d x %d", this.m, this.n, B.m, B.n));
        }
        FloatMatrix C = new FloatMatrix(this.n, B.m);
        this.mm(Transpose.TRANSPOSE, Transpose.TRANSPOSE, 1.0f, B, 0.0f, C);
        return C;
    }

    public LU lu() {
        return this.lu(false);
    }

    public LU lu(boolean overwrite) {
        FloatMatrix lu = overwrite ? this : this.clone();
        int[] ipiv = new int[Math.min(this.m, this.n)];
        int info = LAPACK.engine.getrf(lu.layout(), lu.m, lu.n, lu.A, lu.ld, IntBuffer.wrap(ipiv));
        if (info < 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        return new LU(lu, ipiv, info);
    }

    public Cholesky cholesky() {
        return this.cholesky(false);
    }

    public Cholesky cholesky(boolean overwrite) {
        if (this.uplo == null) {
            throw new IllegalArgumentException("The matrix is not symmetric");
        }
        FloatMatrix lu = overwrite ? this : this.clone();
        int info = LAPACK.engine.potrf(lu.layout(), lu.uplo, lu.n, lu.A, lu.ld);
        if (info != 0) {
            logger.error("LAPACK GETRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GETRF error code: " + info);
        }
        return new Cholesky(lu);
    }

    public QR qr() {
        return this.qr(false);
    }

    public QR qr(boolean overwrite) {
        FloatMatrix qr = overwrite ? this : this.clone();
        float[] tau = new float[Math.min(this.m, this.n)];
        int info = LAPACK.engine.geqrf(qr.layout(), qr.m, qr.n, qr.A, qr.ld, FloatBuffer.wrap(tau));
        if (info != 0) {
            logger.error("LAPACK GEQRF error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEQRF error code: " + info);
        }
        return new QR(qr, tau);
    }

    public SVD svd() {
        return this.svd(true, false);
    }

    public SVD svd(boolean vectors, boolean overwrite) {
        FloatMatrix W;
        int k = Math.min(this.m, this.n);
        float[] s = new float[k];
        FloatMatrix floatMatrix = W = overwrite ? this : this.clone();
        if (vectors) {
            FloatMatrix U = new FloatMatrix(this.m, k);
            FloatMatrix VT = new FloatMatrix(k, this.n);
            int info = LAPACK.engine.gesdd(W.layout(), SVDJob.COMPACT, W.m, W.n, W.A, W.ld, FloatBuffer.wrap(s), U.A, U.ld, VT.A, VT.ld);
            if (info != 0) {
                logger.error("LAPACK GESDD error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK GESDD error code: " + info);
            }
            return new SVD(s, U, VT.transpose());
        }
        FloatMatrix U = new FloatMatrix(1, 1);
        FloatMatrix VT = new FloatMatrix(1, 1);
        int info = LAPACK.engine.gesdd(W.layout(), SVDJob.NO_VECTORS, W.m, W.n, W.A, W.ld, FloatBuffer.wrap(s), U.A, U.ld, VT.A, VT.ld);
        if (info != 0) {
            logger.error("LAPACK GESDD error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GESDD error code: " + info);
        }
        return new SVD(this.m, this.n, s);
    }

    public EVD eigen() {
        return this.eigen(false, true, false);
    }

    public EVD eigen(boolean vl, boolean vr, boolean overwrite) {
        FloatMatrix eig;
        if (this.m != this.n) {
            throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.m, this.n));
        }
        FloatMatrix floatMatrix = eig = overwrite ? this : this.clone();
        if (this.isSymmetric()) {
            float[] w = new float[this.n];
            int info = LAPACK.engine.syevd(eig.layout(), vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, eig.uplo, this.n, eig.A, eig.ld, FloatBuffer.wrap(w));
            if (info != 0) {
                logger.error("LAPACK SYEV error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK SYEV error code: " + info);
            }
            return new EVD(w, vr ? eig : null);
        }
        float[] wr = new float[this.n];
        float[] wi = new float[this.n];
        FloatMatrix Vl = vl ? new FloatMatrix(this.n, this.n) : new FloatMatrix(1, 1);
        FloatMatrix Vr = vr ? new FloatMatrix(this.n, this.n) : new FloatMatrix(1, 1);
        int info = LAPACK.engine.geev(eig.layout(), vl ? EVDJob.VECTORS : EVDJob.NO_VECTORS, vr ? EVDJob.VECTORS : EVDJob.NO_VECTORS, this.n, eig.A, eig.ld, FloatBuffer.wrap(wr), FloatBuffer.wrap(wi), Vl.A, Vl.ld, Vr.A, Vr.ld);
        if (info != 0) {
            logger.error("LAPACK GEEV error code: {}", (Object)info);
            throw new ArithmeticException("LAPACK GEEV error code: " + info);
        }
        float[] w = new float[2 * this.n];
        System.arraycopy(wr, 0, w, 0, this.n);
        System.arraycopy(wi, 0, w, this.n, this.n);
        return new EVD(wr, wi, vl ? Vl : null, vr ? Vr : null);
    }

    public static class QR
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final FloatMatrix qr;
        public final float[] tau;

        public QR(FloatMatrix qr, float[] tau) {
            this.qr = qr;
            this.tau = tau;
        }

        public Cholesky CholeskyOfAtA() {
            int n = this.qr.n;
            FloatMatrix L = new FloatMatrix(n, n);
            for (int i = 0; i < n; ++i) {
                for (int j = 0; j <= i; ++j) {
                    L.set(i, j, this.qr.get(j, i));
                }
            }
            L.uplo(UPLO.LOWER);
            return new Cholesky(L);
        }

        public FloatMatrix R() {
            int n = this.qr.n;
            FloatMatrix R = FloatMatrix.diag(this.tau);
            for (int i = 0; i < n; ++i) {
                for (int j = i + 1; j < n; ++j) {
                    R.set(i, j, this.qr.get(i, j));
                }
            }
            return R;
        }

        public FloatMatrix Q() {
            int m = this.qr.m;
            int n = this.qr.n;
            FloatMatrix Q = new FloatMatrix(m, n);
            for (int k = n - 1; k >= 0; --k) {
                Q.set(k, k, 1.0f);
                for (int j = k; j < n; ++j) {
                    int i;
                    if (this.qr.get(k, k) == 0.0f) continue;
                    float s = 0.0f;
                    for (i = k; i < m; ++i) {
                        s += this.qr.get(i, k) * Q.get(i, j);
                    }
                    s = -s / this.qr.get(k, k);
                    for (i = k; i < m; ++i) {
                        Q.add(i, j, s * this.qr.get(i, k));
                    }
                }
            }
            return Q;
        }

        public float[] solve(float[] b) {
            if (b.length != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.qr.m, this.qr.n, b.length));
            }
            float[] y = (float[])b.clone();
            this.solve(new FloatMatrix(y));
            float[] x = new float[this.qr.n];
            System.arraycopy(y, 0, x, 0, x.length);
            return x;
        }

        public void solve(FloatMatrix B) {
            if (B.m != this.qr.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.qr.nrows(), this.qr.nrows(), B.nrows(), B.ncols()));
            }
            int m = this.qr.m;
            int n = this.qr.n;
            int k = Math.min(m, n);
            int info = LAPACK.engine.ormqr(this.qr.layout(), Side.LEFT, Transpose.TRANSPOSE, B.nrows(), B.ncols(), k, this.qr.A, this.qr.ld, FloatBuffer.wrap(this.tau), B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK ORMQR error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK ORMQR error code: " + info);
            }
            info = LAPACK.engine.trtrs(this.qr.layout(), UPLO.UPPER, Transpose.NO_TRANSPOSE, Diag.NON_UNIT, this.qr.n, B.n, this.qr.A, this.qr.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK TRTRS error code: {}", (Object)info);
                throw new IllegalArgumentException("LAPACK TRTRS error code: " + info);
            }
        }
    }

    public static class Cholesky
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final FloatMatrix lu;

        public Cholesky(FloatMatrix lu) {
            if (lu.nrows() != lu.ncols()) {
                throw new UnsupportedOperationException("Cholesky constructor on a non-square matrix");
            }
            this.lu = lu;
        }

        public float det() {
            double d = 1.0;
            for (int i = 0; i < this.lu.n; ++i) {
                d *= (double)this.lu.get(i, i);
            }
            return (float)(d * d);
        }

        public float logdet() {
            int n = this.lu.n;
            double d = 0.0;
            for (int i = 0; i < n; ++i) {
                d += Math.log(this.lu.get(i, i));
            }
            return (float)(2.0 * d);
        }

        public FloatMatrix inverse() {
            FloatMatrix inv = FloatMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            float[] x = (float[])b.clone();
            this.solve(new FloatMatrix(x));
            return x;
        }

        public void solve(FloatMatrix B) {
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            int info = LAPACK.engine.potrs(this.lu.layout(), this.lu.uplo, this.lu.n, B.n, this.lu.A, this.lu.ld, B.A, B.ld);
            if (info != 0) {
                logger.error("LAPACK POTRS error code: {}", (Object)info);
                throw new ArithmeticException("LAPACK POTRS error code: " + info);
            }
        }
    }

    public static class LU
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final FloatMatrix lu;
        public final int[] ipiv;
        public final int info;

        public LU(FloatMatrix lu, int[] ipiv, int info) {
            this.lu = lu;
            this.ipiv = ipiv;
            this.info = info;
        }

        public boolean isSingular() {
            return this.info > 0;
        }

        public float det() {
            int j;
            int m = this.lu.m;
            int n = this.lu.n;
            if (m != n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", m, n));
            }
            double d = 1.0;
            for (j = 0; j < n; ++j) {
                d *= (double)this.lu.get(j, j);
            }
            for (j = 0; j < n; ++j) {
                if (j + 1 == this.ipiv[j]) continue;
                d = -d;
            }
            return (float)d;
        }

        public FloatMatrix inverse() {
            FloatMatrix inv = FloatMatrix.eye(this.lu.n);
            this.solve(inv);
            return inv;
        }

        public float[] solve(float[] b) {
            float[] x = (float[])b.clone();
            this.solve(new FloatMatrix(x));
            return x;
        }

        public void solve(FloatMatrix B) {
            if (this.lu.m != this.lu.n) {
                throw new IllegalArgumentException(String.format("The matrix is not square: %d x %d", this.lu.m, this.lu.n));
            }
            if (B.m != this.lu.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x %d", this.lu.m, this.lu.n, B.m, B.n));
            }
            if (this.lu.layout() != B.layout()) {
                throw new IllegalArgumentException("The matrix layout is inconsistent.");
            }
            if (this.info > 0) {
                throw new RuntimeException("The matrix is singular.");
            }
            int ret = LAPACK.engine.getrs(this.lu.layout(), Transpose.NO_TRANSPOSE, this.lu.n, B.n, this.lu.A, this.lu.ld, IntBuffer.wrap(this.ipiv), B.A, B.ld);
            if (ret != 0) {
                logger.error("LAPACK GETRS error code: {}", (Object)ret);
                throw new ArithmeticException("LAPACK GETRS error code: " + ret);
            }
        }
    }

    public static class EVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final float[] wr;
        public final float[] wi;
        public final FloatMatrix Vl;
        public final FloatMatrix Vr;

        public EVD(float[] w, FloatMatrix V) {
            this.wr = w;
            this.wi = null;
            this.Vl = V;
            this.Vr = V;
        }

        public EVD(float[] wr, float[] wi, FloatMatrix Vl, FloatMatrix Vr) {
            this.wr = wr;
            this.wi = wi;
            this.Vl = Vl;
            this.Vr = Vr;
        }

        public FloatMatrix diag() {
            FloatMatrix D = FloatMatrix.diag(this.wr);
            if (this.wi != null) {
                int n = this.wr.length;
                for (int i = 0; i < n; ++i) {
                    if (this.wi[i] > 0.0f) {
                        D.set(i, i + 1, this.wi[i]);
                        continue;
                    }
                    if (!(this.wi[i] < 0.0f)) continue;
                    D.set(i, i - 1, this.wi[i]);
                }
            }
            return D;
        }

        public EVD sort() {
            int i;
            int n = this.wr.length;
            float[] w = new float[n];
            if (this.wi != null) {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i] + this.wi[i] * this.wi[i]);
                }
            } else {
                for (i = 0; i < n; ++i) {
                    w[i] = -(this.wr[i] * this.wr[i]);
                }
            }
            int[] index = QuickSort.sort(w);
            float[] wr2 = new float[n];
            for (int j = 0; j < n; ++j) {
                wr2[j] = this.wr[index[j]];
            }
            float[] wi2 = null;
            if (this.wi != null) {
                wi2 = new float[n];
                for (int j = 0; j < n; ++j) {
                    wi2[j] = this.wi[index[j]];
                }
            }
            FloatMatrix Vl2 = null;
            if (this.Vl != null) {
                int m = this.Vl.m;
                Vl2 = new FloatMatrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i2 = 0; i2 < m; ++i2) {
                        Vl2.set(i2, j, this.Vl.get(i2, index[j]));
                    }
                }
            }
            FloatMatrix Vr2 = null;
            if (this.Vr != null) {
                int m = this.Vr.m;
                Vr2 = new FloatMatrix(m, n);
                for (int j = 0; j < n; ++j) {
                    for (int i3 = 0; i3 < m; ++i3) {
                        Vr2.set(i3, j, this.Vr.get(i3, index[j]));
                    }
                }
            }
            return new EVD(wr2, wi2, Vl2, Vr2);
        }
    }

    public static class SVD
    implements Serializable {
        private static final long serialVersionUID = 2L;
        public final int m;
        public final int n;
        public final float[] s;
        public final FloatMatrix U;
        public final FloatMatrix V;

        public SVD(int m, int n, float[] s) {
            this.m = m;
            this.n = n;
            this.s = s;
            this.U = null;
            this.V = null;
        }

        public SVD(float[] s, FloatMatrix U, FloatMatrix V) {
            this.m = U.m;
            this.n = V.m;
            this.s = s;
            this.U = U;
            this.V = V;
        }

        public FloatMatrix diag() {
            FloatMatrix S = new FloatMatrix(this.U.m, this.V.m);
            for (int i = 0; i < this.s.length; ++i) {
                S.set(i, i, this.s[i]);
            }
            return S;
        }

        public double norm() {
            return this.s[0];
        }

        private float rcond() {
            return 0.5f * (float)Math.sqrt(this.m + this.n + 1) * this.s[0] * MathEx.FLOAT_EPSILON;
        }

        public int rank() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            int r = 0;
            float tol = this.rcond();
            for (int i = 0; i < this.s.length; ++i) {
                if (!(this.s[i] > tol)) continue;
                ++r;
            }
            return r;
        }

        public int nullity() {
            return Math.min(this.m, this.n) - this.rank();
        }

        public float condition() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            return this.s[0] <= 0.0f || this.s[this.s.length - 1] <= 0.0f ? Float.POSITIVE_INFINITY : this.s[0] / this.s[this.s.length - 1];
        }

        public FloatMatrix range() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.U == null) {
                throw new IllegalStateException("The left singular vectors are not available.");
            }
            int r = this.rank();
            if (r == 0) {
                return null;
            }
            FloatMatrix R = new FloatMatrix(this.m, r);
            for (int j = 0; j < r; ++j) {
                for (int i = 0; i < this.m; ++i) {
                    R.set(i, j, this.U.get(i, j));
                }
            }
            return R;
        }

        public FloatMatrix nullspace() {
            if (this.s.length != Math.min(this.m, this.n)) {
                throw new UnsupportedOperationException("The operation cannot be called on a partial SVD.");
            }
            if (this.V == null) {
                throw new IllegalStateException("The right singular vectors are not available.");
            }
            int nr = this.nullity();
            if (nr == 0) {
                return null;
            }
            FloatMatrix N = new FloatMatrix(this.n, nr);
            for (int j = 0; j < nr; ++j) {
                for (int i = 0; i < this.n; ++i) {
                    N.set(i, j, this.V.get(i, this.n - j - 1));
                }
            }
            return N;
        }

        public FloatMatrix pinv() {
            int k = this.s.length;
            float[] sigma = new float[k];
            int r = this.rank();
            for (int i = 0; i < r; ++i) {
                sigma[i] = 1.0f / this.s[i];
            }
            return this.V.adb(Transpose.NO_TRANSPOSE, Transpose.TRANSPOSE, this.U, sigma);
        }

        public float[] solve(float[] b) {
            if (this.U == null || this.V == null) {
                throw new IllegalStateException("The singular vectors are not available.");
            }
            if (b.length != this.m) {
                throw new IllegalArgumentException(String.format("Row dimensions do not agree: A is %d x %d, but B is %d x 1", this.m, this.n, b.length));
            }
            int r = this.rank();
            float[] Utb = new float[this.s.length];
            this.U.submatrix(0, 0, this.m - 1, r - 1).tv(b, Utb);
            for (int i = 0; i < r; ++i) {
                int n = i;
                Utb[n] = Utb[n] / this.s[i];
            }
            return this.V.mv(Utb);
        }
    }
}

