package com.github.houbb.hash.api;

/**
 * @author binbin.hou
 * @since 0.0.2
 */
public interface IHashContext {

    /**
     * Returns the source data that will be hashed by a {@link IHash}. For example, this might be a
     * {@code ByteSource} representation of a password, or file, etc.
     *
     * @return the source data that will be hashed by a {@link IHash}.
     */
    byte[] source();

    /**
     * Returns a salt to be used by the {@link IHash} during hash computation, or {@code null} if no salt is
     * provided as part of the request.
     *
     * Note that a {@code null} value does not necessarily mean a salt won't be used at all - it just
     * means that the request didn't include one.  The servicing {@link IHash} is free to provide a salting
     * strategy for a request, even if the request did not specify one.
     *
     * @return a salt to be used by the {@link IHash} during hash computation, or {@code null} if no salt is
     *         provided as part of the request.
     */
    byte[] salt();

    /**
     * Returns the number of requested hash iterations to be performed when computing the final {@code Hash} result.
     * A non-positive (0 or less) indicates that the {@code HashService}'s default iteration configuration should
     * be used.  A positive value overrides the {@code HashService}'s configuration for a single request.
     *
     * Note that a {@code HashService} is free to ignore this number if it determines the number is not sufficient
     * to meet a desired level of security.
     *
     * @return the number of requested hash iterations to be performed when computing the final {@code Hash} result.
     */
    int times();

    /**
     * Returns the name of the hash algorithm the {@code HashService} should use when computing the Hash, or
     * {@code null} if the default algorithm configuration of the {@code HashService} should be used.  A non-null value
     * overrides the {@code HashService}'s configuration for a single request.
     *
     * Note that a {@code HashService} is free to ignore this value if it determines that the algorithm is not
     * sufficient to meet a desired level of security.
     *
     * @return the name of the hash algorithm the {@code HashService} should use when computing the Hash, or
     *         {@code null} if the default algorithm configuration of the {@code HashService} should be used.
     */
    String algorithmName();

}
