package com.github.houbb.hash.core.hash.result;

import com.github.houbb.hash.api.IHash;
import com.github.houbb.hash.api.IHashContext;
import com.github.houbb.hash.api.IHashResult;
import com.github.houbb.hash.exception.HashRuntimeException;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * @author binbin.hou
 * @since 0.0.2
 */
public class Hash implements IHash {

    public Hash() {
    }

    @Override
    public IHashResult hash(IHashContext context) {
        byte[] hashed = this.hash(context.source(), context.salt(), context.times(), context.algorithmName());
        return HashResult.newInstance().hashed(hashed);
    }

    /**
     * Returns the JDK MessageDigest instance to use for executing the hash.
     *
     * @param algorithmName the algorithm to use for the hash, provided by subclasses.
     * @return the MessageDigest object for the specified {@code algorithm}.
     */
    protected MessageDigest getDigest(String algorithmName){
        try {
            return MessageDigest.getInstance(algorithmName);
        } catch (NoSuchAlgorithmException e) {
            String msg = "No native '" + algorithmName + "' MessageDigest instance available on the current JVM.";
            throw new HashRuntimeException(msg, e);
        }
    }

    /**
     * Hashes the specified byte array using the given {@code salt} for the specified number of times.
     *
     * @param bytes          the bytes to hash
     * @param salt           the salt to use for the initial hash
     * @param hashIterations the number of times the the {@code bytes} will be hashed (for attack resiliency).
     * @param algorithmName 算法
     * @return the hashed bytes.
     * @since 0.0.2
     */
    protected byte[] hash(byte[] bytes, byte[] salt, int hashIterations,
                          String algorithmName)  {
        MessageDigest digest = getDigest(algorithmName);
        if (salt != null) {
            digest.reset();
            digest.update(salt);
        }
        byte[] hashed = digest.digest(bytes);
        //already hashed once above
        int times = hashIterations - 1;
        //iterate remaining number:
        for (int i = 0; i < times; i++) {
            digest.reset();
            hashed = digest.digest(hashed);
        }
        return hashed;
    }

}
