package com.github.houbb.hash.util;

import com.github.houbb.hash.api.IHashResultHandler;
import com.github.houbb.hash.bs.HashBs;
import com.github.houbb.hash.constant.enums.HashType;
import com.github.houbb.hash.core.handler.HashResultHandlers;
import com.github.houbb.heaven.util.codec.CodecSupport;

/**
 * 哈希编码工具类
 * @author binbin.hou
 * @since 0.0.1
 */
public final class HashHelper {

    private HashHelper(){}

    /**
     * hash 计算
     * @param text 文本
     * @return 结果
     * @since 0.0.2
     */
    public static String hash(final String text) {
        return hash(HashType.MD5, text);
    }

    /**
     * hash 计算
     * @param type 类型
     * @param text 文本
     * @return 结果
     * @since 0.0.2
     */
    public static String hash(final HashType type,
                             final String text) {
        return hash(type, text, HashResultHandlers.hex());
    }

    /**
     * hash 计算
     * @param type 类型
     * @param text 文本
     * @param hashResultHandler 处理类
     * @param <T> 反省
     * @return 结果
     * @since 0.0.2
     */
    public static <T> T hash(final HashType type,
                             final String text,
                             final IHashResultHandler<T> hashResultHandler) {
        return hash(type, text, null, hashResultHandler);
    }

    /**
     * hash 计算
     * @param type 类型
     * @param text 文本
     * @param salt 盐
     * @param hashResultHandler 处理类
     * @param <T> 反省
     * @return 结果
     * @since 0.0.2
     */
    public static <T> T hash(final HashType type,
                             final String text, final String salt,
                             final IHashResultHandler<T> hashResultHandler) {
        return hash(type, text, salt, 1, hashResultHandler);
    }

    /**
     * hash 计算
     * @param type 类型
     * @param text 文本
     * @param salt 盐
     * @param times 次数
     * @param hashResultHandler 处理类
     * @param <T> 反省
     * @return 结果
     * @since 0.0.2
     */
    public static <T> T hash(final HashType type,
                             final String text, final String salt,
                             final int times, final IHashResultHandler<T> hashResultHandler) {
        return HashBs.newInstance()
                .type(type)
                .source(CodecSupport.toBytes(text))
                .salt(CodecSupport.toBytes(salt))
                .times(times)
                .execute(hashResultHandler);
    }

}
