package com.github.houbb.load.balance.api.impl;

import com.github.houbb.heaven.annotation.ThreadSafe;
import com.github.houbb.heaven.support.filter.IFilter;
import com.github.houbb.heaven.support.handler.IHandler;
import com.github.houbb.heaven.util.lang.MathUtil;
import com.github.houbb.heaven.util.util.CollectionUtil;
import com.github.houbb.load.balance.api.ILoadBalanceContext;
import com.github.houbb.load.balance.support.server.IServer;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 指定权重的轮训策略
 *
 * @author binbin.hou
 * @since 0.0.1
 */
@ThreadSafe
public class LoadBalanceWeightRoundRobbin<T extends IServer> extends AbstractLoadBalance<T> {

    /**
     * 位移指针
     * @since 0.0.1
     */
    private final AtomicLong indexHolder = new AtomicLong();

    @Override
    protected T doSelect(ILoadBalanceContext<T> context) {
        List<T> servers = context.servers();
        List<T> actualList = buildActualList(servers);

        long index = indexHolder.getAndIncrement();

        // 基于真实的列表构建
        int actual = (int) (index % actualList.size());
        return actualList.get(actual);
    }

    /**
     * 初始化
     * @param serverList 服务列表
     * @since 0.0.1
     */
    private List<T> buildActualList(final List<T> serverList) {
        final List<T> actualList = new ArrayList<>();

        //1. 过滤掉权重为 0 的机器
        List<T> notZeroServers = CollectionUtil.filterList(serverList, new IFilter<T>() {
            @Override
            public boolean filter(IServer iServer) {
                return iServer.weight() <= 0;
            }
        });

        //2. 获取权重列表
        List<Integer> weightList = CollectionUtil.toList(notZeroServers, new IHandler<T, Integer>() {
            @Override
            public Integer handle(T iServer) {
                return iServer.weight();
            }
        });

        //3. 获取最大的权重
        int maxDivisor = MathUtil.ngcd(weightList);

        //4. 重新计算构建基于权重的列表
        for(T server : notZeroServers) {
            int weight = server.weight();

            int times = weight / maxDivisor;
            for(int i = 0; i < times; i++) {
                actualList.add(server);
            }
        }

        return actualList;
    }

}
