package com.github.houbb.property.bs;

import com.github.houbb.heaven.constant.CharsetConst;
import com.github.houbb.heaven.support.instance.impl.Instances;
import com.github.houbb.heaven.util.common.ArgUtil;
import com.github.houbb.heaven.util.io.FileUtil;
import com.github.houbb.heaven.util.io.StreamUtil;
import com.github.houbb.heaven.util.lang.ObjectUtil;
import com.github.houbb.heaven.util.lang.StringUtil;
import com.github.houbb.heaven.util.lang.reflect.ClassUtil;
import com.github.houbb.heaven.util.util.MapUtil;
import com.github.houbb.property.api.IPropertyReader;
import com.github.houbb.property.core.IProperty;
import com.github.houbb.property.core.impl.SimpleProperty;
import com.github.houbb.property.exception.PropertyRuntimeException;
import com.github.houbb.property.support.bean.impl.BeanToMap;
import com.github.houbb.property.support.bean.impl.MapToBean;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 属性引导类
 *
 * @author binbin.hou
 * @since 0.0.1
 */
public class PropertyBs implements IPropertyReader {

    /**
     * 内置路径
     * @since 0.0.5
     */
    private static final String DEFAULT_PATH = "$COM_GITHUB_HOUBB_PROPERTY_DEFAULT$";

    /**
     * 实例 map
     * （1）key: 文件路径
     * （2）value: 对应的实例
     *
     * @since 0.0.1
     */
    private static final Map<String, PropertyBs> INSTANCE_MAP = new ConcurrentHashMap<>();

    /**
     * 文件路径
     *
     * @since 0.0.1
     */
    private final String path;

    /**
     * 默认的属性处理类
     *
     * @since 0.0.1
     */
    private final IProperty property = new SimpleProperty();

    /**
     * 文件字符集合
     *
     * @since 0.0.1
     */
    private String charset = CharsetConst.UTF8;

    /**
     * 私有化构造器
     *
     * @param path 文件路径
     * @since 0.0.1
     */
    private PropertyBs(String path) {
        this.path = path;
    }

    /**
     * 获取实例
     * （1）如果已很创建，则直接返回
     * （2）如果不存在，则创建，并且执行一次文件信息加载。
     * <p>
     * 后期可以考虑支持判断文件是否存在，添加一个属性。暂时不支持。
     *
     * @param propertyPath 文件路径
     * @return 实例
     * @since 0.0.1
     */
    public static PropertyBs getInstance(final String propertyPath) {
        ArgUtil.notEmpty(propertyPath, "propertyPath");

        PropertyBs instance = INSTANCE_MAP.get(propertyPath);
        if (ObjectUtil.isNotNull(instance)) {
            return instance;
        }


        PropertyBs newInstance = new PropertyBs(propertyPath);
        // 如果不是使用内置才进行加载
        if(!propertyPath.equals(DEFAULT_PATH)) {
            newInstance.load();
        }

        INSTANCE_MAP.put(propertyPath, newInstance);
        return newInstance;
    }

    /**
     * 获取默认内置实例信息
     * @return 默认实例
     * @since 0.0.5
     * @see #flush(String) 刷新到磁盘必须指定路径
     */
    public static PropertyBs getInstance() {
        return getInstance(DEFAULT_PATH);
    }

    /**
     * 设置文件编码集
     *
     * @param charset 文件编码集
     * @return this
     * @since 0.0.1
     */
    public PropertyBs charset(String charset) {
        ArgUtil.notEmpty(charset, "charset");

        this.charset = charset;
        return this;
    }

    /**
     * 获取对应的值
     *
     * @param key 指定的 key
     * @return 结果
     * @since 0.0.1
     */
    @Override
    public String get(final String key) {
        ArgUtil.notEmpty(key, "key");
        return property.getAttrString(key);
    }

    /**
     * 获取对应的值 如果没有则使用 defaultValue;
     * （1）没有：当值为 null 的时候
     *
     * @param key          指定的 key
     * @param defaultValue 默认值
     * @return 结果
     * @since 0.0.1
     */
    @Override
    public String getOrDefault(final String key, final String defaultValue) {
        ArgUtil.notEmpty(key, "key");

        String value = this.get(key);
        if (ObjectUtil.isNotNull(value)) {
            return value;
        }
        return defaultValue;
    }

    @Override
    public Boolean getBool(String key) {
        return StringUtil.toBool(get(key));
    }

    @Override
    public Byte getByte(String key) {
        return StringUtil.toByte(get(key));
    }

    @Override
    public Character getChar(String key) {
        return StringUtil.toChar(get(key));
    }

    @Override
    public Short getShort(String key) {
        return StringUtil.toShort(get(key));
    }

    @Override
    public Integer getInt(String key) {
        return StringUtil.toInt(get(key));
    }

    @Override
    public Long getLong(String key) {
        return StringUtil.toLong(get(key));
    }

    @Override
    public Float getFloat(String key) {
        return StringUtil.toFloat(get(key));
    }

    @Override
    public Double getDouble(String key) {
        return StringUtil.toDouble(get(key));
    }

    @Override
    public Date getDate(String key, String dateFormat) {
        return StringUtil.toDate(get(key), dateFormat);
    }

    @Override
    public Date getDate(String key) {
        return StringUtil.toDate(get(key));
    }

    @Override
    public BigInteger getBigInteger(String key) {
        return StringUtil.toBigInteger(get(key));
    }

    @Override
    public BigDecimal getBigDecimal(String key) {
        return StringUtil.toBigDecimal(get(key));
    }

    /**
     * 设置对应的值
     * 1. 有则设置，无则添加
     * 2. 此时并不会刷新到磁盘中。
     *
     * @param key   属性的键
     * @param value 属性的值
     * @return this
     * @since 0.0.1
     */
    public PropertyBs set(final String key, final String value) {
        ArgUtil.notEmpty(key, "key");

        property.putAttr(key, value);
        return this;
    }

    /**
     * 移除对应的值
     *
     * @param key 属性的键
     * @return this
     * @since 0.0.2
     */
    public PropertyBs remove(final String key) {
        ArgUtil.notEmpty(key, "key");

        property.removeAttr(key);
        return this;
    }

    /**
     * 刷新内存中的值到磁盘中
     *
     * @return this
     * @since 0.0.1
     */
    public synchronized PropertyBs flush() {
        return this.flush(this.path);
    }

    /**
     * 刷新内存中的值到磁盘中
     *
     * @param path 指定输出文件路径信息
     * @return this
     * @since 0.0.1
     */
    public synchronized PropertyBs flush(final String path) {
        try {
            if(DEFAULT_PATH.equals(path)) {
                throw new PropertyRuntimeException("Forbidden write into the default path "
                + DEFAULT_PATH);
            }


            OutputStream outputStream = new FileOutputStream(path);
            property.flush(outputStream, charset);
            return this;
        } catch (FileNotFoundException e) {
            throw new PropertyRuntimeException(e);
        }
    }

    /**
     * 加载磁盘中信息到内存中
     *
     * @return this
     * @since 0.0.1
     */
    private synchronized PropertyBs load() {
        final InputStream inputStream = StreamUtil.getInputStream(path);
        this.property.load(inputStream, CharsetConst.UTF8);
        return this;
    }

    /**
     * 设置 map 的信息到 property 中
     *
     * @param map map
     * @return this
     * @since 0.0.2
     */
    public PropertyBs set(final Map<String, String> map) {
        if (MapUtil.isEmpty(map)) {
            return this;
        }

        this.property.putAttr(map);
        return this;
    }

    /**
     * 设置 bean 的信息到 property 中
     *
     * @param bean 对象
     * @return this
     * @since 0.0.2
     */
    public PropertyBs set(final Object bean) {
        Map<String, String> valueMap = Instances.singleton(BeanToMap.class)
                .beanToMap(bean);

        return this.set(valueMap);
    }

    /**
     * 返回 map 信息
     * （1）注意防御编程
     *
     * @return map 集合信息
     * @since 0.0.2
     */
    public Map<String, String> asMap() {
        return this.property.asMap();
    }

    /**
     * 转换为配置
     * @return 转换为配置文件
     * @since 0.0.8
     */
    public Properties asProperties() {
        return property.asProperties();
    }

    /**
     * 设置 property 的信息到 bean 中
     *
     * @param bean 对象
     * @return 对象信息
     * @since 0.0.2
     */
    public Object asBean(final Object bean) {
        ArgUtil.notNull(bean, "bean");

        final Map<String, String> valueMap = this.asMap();
        Instances.singleton(MapToBean.class)
                .mapToBean(valueMap, bean);

        return this;
    }


    /**
     * 设置 property 的信息到 bean 中
     *
     * @param tClass 类型
     * @return 对象信息
     * @since 0.0.8
     */
    @SuppressWarnings("all")
    public <T> T asBean(final Class<T> tClass) {
        ArgUtil.notNull(tClass, "tClass");

        T instance = ClassUtil.newInstance(tClass);

        return (T) asBean(instance);
    }

}
