package com.github.houbb.property.core.impl;

import com.github.houbb.heaven.support.attr.impl.AttributeContext;
import com.github.houbb.heaven.util.guava.Guavas;
import com.github.houbb.heaven.util.io.StreamUtil;
import com.github.houbb.heaven.util.lang.ObjectUtil;
import com.github.houbb.heaven.util.util.CollectionUtil;
import com.github.houbb.property.core.IProperty;
import com.github.houbb.property.exception.PropertyRuntimeException;
import com.github.houbb.property.support.properties.OrderedProperties;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * 属性接口适配器
 *
 * （1） 学习 Jdk {@link java.util.Properties} 属性工具类
 * （2） 改进 {@link AttributeContext} 提供 protected 级别方法，便于拓展。
 * （3）考虑并发安全，添加 {@link java.util.concurrent.locks.ReadWriteLock} 读写锁。
 *
 * @author binbin.hou
 * @since 0.0.1
 */
public class SimpleProperty extends AttributeContext implements IProperty {

    @Override
    public synchronized IProperty load(InputStream inputStream, String charset) {
        try {
            Properties properties = new Properties();
            InputStreamReader inputStreamReader = new InputStreamReader(inputStream, charset);
            properties.load(inputStreamReader);

            // 加载到内存中
            final int size = properties.keySet().size();
            Map<String, Object> map = new HashMap<>(size);
            for(Object key : properties.keySet()) {
                map.put(String.valueOf(key), properties.get(key));
            }
            super.putAttrMap(map);

            return this;
        } catch (IOException e) {
            throw new PropertyRuntimeException(e);
        } finally {
            StreamUtil.closeStream(inputStream);
        }
    }

    @Override
    public synchronized IProperty flush(OutputStream outputStream, String charset) {
        Properties properties = new OrderedProperties();

        try {
            // 获取内存中的集合信息
            for(Map.Entry<String, Object> entry : super.entrySet()) {
                properties.put(entry.getKey(), entry.getValue());
            }

            // 写入到指定的文件中
            properties.store(outputStream, null);

            return this;
        } catch (IOException e) {
            throw new PropertyRuntimeException(e);
        } finally {
            StreamUtil.closeStream(outputStream);
        }
    }

    @Override
    public IProperty putAttr(Map<String, String> map) {
        super.putAttrMap(map);
        return this;
    }

    @Override
    public Map<String, String> asMap() {
        Set<Map.Entry<String, Object>> entrySet = super.entrySet();

        if(CollectionUtil.isEmpty(entrySet)) {
            return Guavas.newHashMap();
        }

        Map<String, String> resultMap = Guavas.newHashMap(entrySet.size());
        for(Map.Entry<String, Object> entry : entrySet) {
            String value = ObjectUtil.objectToString(entry.getValue());
            resultMap.put(entry.getKey(), value);
        }
        return resultMap;
    }

    @Override
    public Properties asProperties() {
        Properties properties = new Properties();

        Set<Map.Entry<String, Object>> entrySet = super.entrySet();
        for(Map.Entry<String, Object> entry : entrySet) {
            String value = ObjectUtil.objectToString(entry.getValue());
            properties.setProperty(entry.getKey(), value);
        }
        return properties;
    }

}
