package com.github.houbb.property.support.converter.factory;

import com.github.houbb.heaven.annotation.ThreadSafe;
import com.github.houbb.heaven.util.common.ArgUtil;
import com.github.houbb.heaven.util.lang.ObjectUtil;
import com.github.houbb.heaven.util.lang.reflect.ClassUtil;
import com.github.houbb.property.support.converter.IValueConverter;
import com.github.houbb.property.support.converter.impl.DefaultValueConverter;

import java.util.HashMap;
import java.util.Map;

/**
 * 值转换类工厂
 * @author binbin.hou
 * @since 0.0.2
 */
public final class ValueConverterFactory {

    private ValueConverterFactory(){}

    /**
     * 转换类 map
     * @since 0.0.2
     */
    private static final Map<Class<? extends IValueConverter>, IValueConverter> CONVERTER_MAP = new HashMap<>();

    static {
        synchronized (CONVERTER_MAP) {
            CONVERTER_MAP.put(DefaultValueConverter.class, new DefaultValueConverter());
        }
    }

    /**
     * 注册值转换类
     * @param converterClass 转换类信息
     * @param valueConverter 转换类实现
     * @since 0.0.2
     */
    private static synchronized void register(final Class<? extends IValueConverter> converterClass,
                                             final IValueConverter valueConverter) {
        ArgUtil.notNull(converterClass, "converterClass");
        ArgUtil.notNull(valueConverter, "valueConverter");

        synchronized (CONVERTER_MAP) {
            CONVERTER_MAP.put(converterClass, valueConverter);
        }
    }

    /**
     * 获取转换类实现
     * @param clazz 值
     * @return 值转换类
     * @since 0.0.2
     */
    public static IValueConverter getConverter(final Class<? extends IValueConverter> clazz) {
        IValueConverter valueConverter = CONVERTER_MAP.get(clazz);
        if(ObjectUtil.isNotNull(valueConverter)) {
            return valueConverter;
        }

        valueConverter = ClassUtil.newInstance(clazz);

        // 注册线程安全的实现类
        if(clazz.isAnnotationPresent(ThreadSafe.class)) {
            register(clazz, valueConverter);
        }
        return valueConverter;
    }

}
