package com.github.houbb.property.support.converter.impl;

import com.github.houbb.heaven.annotation.ThreadSafe;
import com.github.houbb.heaven.util.common.ArgUtil;
import com.github.houbb.heaven.util.lang.ObjectUtil;
import com.github.houbb.heaven.util.lang.reflect.ClassTypeUtil;
import com.github.houbb.heaven.util.lang.reflect.PrimitiveUtil;
import com.github.houbb.property.support.converter.IFieldValueContext;
import com.github.houbb.property.support.converter.IPropertyValueContext;
import com.github.houbb.property.support.converter.IValueConverter;
import com.github.houbb.property.support.converter.impl.aggregation.*;
import com.github.houbb.property.support.converter.impl.base.*;
import com.github.houbb.property.support.converter.impl.math.BigDecimalValueConverter;
import com.github.houbb.property.support.converter.impl.math.BigIntegerValueConverter;
import com.github.houbb.property.support.converter.impl.util.CurrencyValueConverter;
import com.github.houbb.property.support.converter.impl.util.DateValueConverter;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.*;

/**
 * 默认值转换类
 * @author binbin.hou
 * @since 0.0.2
 */
@ThreadSafe
public class DefaultValueConverter implements IValueConverter {

    /**
     * 数据类型与转换类
     * @since 0.0.5
     */
    private static final Map<Class, IValueConverter> CONVERTER_MAP = new IdentityHashMap<>();

    static {
        //math
        CONVERTER_MAP.put(BigInteger.class, new BigIntegerValueConverter());
        CONVERTER_MAP.put(BigDecimal.class, new BigDecimalValueConverter());
        //util
        CONVERTER_MAP.put(Date.class, new DateValueConverter());
        CONVERTER_MAP.put(Currency.class, new CurrencyValueConverter());

        //8 大基本类型
        CONVERTER_MAP.put(Boolean.class, new BooleanValueConverter());
        CONVERTER_MAP.put(Byte.class, new ByteValueConverter());
        CONVERTER_MAP.put(Character.class, new CharacterValueConverter());
        CONVERTER_MAP.put(Short.class, new ShortValueConverter());
        CONVERTER_MAP.put(Integer.class, new IntegerValueConverter());
        CONVERTER_MAP.put(Long.class, new LongValueConverter());
        CONVERTER_MAP.put(Float.class, new FloatValueConverter());
        CONVERTER_MAP.put(Double.class, new DoubleValueConverter());
        CONVERTER_MAP.put(String.class, new StringValueConverter());
        CONVERTER_MAP.put(Enum.class, new EnumValueConverter());

        // 数组
        CONVERTER_MAP.put(int[].class, new IntArrayValueConverter());
        CONVERTER_MAP.put(boolean[].class, new BooleanArrayValueConverter());
        CONVERTER_MAP.put(short[].class, new ShortArrayValueConverter());
        CONVERTER_MAP.put(byte[].class, new ByteArrayValueConverter());
        CONVERTER_MAP.put(long[].class, new LongArrayValueConverter());
        CONVERTER_MAP.put(char[].class, new CharArrayValueConverter());
        CONVERTER_MAP.put(float[].class, new FloatArrayValueConverter());
        CONVERTER_MAP.put(double[].class, new DoubleArrayValueConverter());
        CONVERTER_MAP.put(Array.class, new StringArrayValueConverter());

        // 集合
        CONVERTER_MAP.put(Collection.class, new StringCollectionValueConverter());
    }

    @Override
    public Object fieldValue(String value, IFieldValueContext context) {
        Class type = context.field().getType();
        return getValueConverter(type).fieldValue(value, context);
    }

    @Override
    public String propertyValue(Object value, IPropertyValueContext context) {
        Class type = context.field().getType();

        return getValueConverter(type).propertyValue(value, context);
    }

    /**
     * 获取值转换类型
     * @param type 类型
     * @return 结果
     * @since 0.0.3
     */
    private IValueConverter getValueConverter(final Class type) {
        ArgUtil.notNull(type, "type");

        IValueConverter valueConverter = CONVERTER_MAP.get(type);
        if(ObjectUtil.isNotNull(valueConverter)) {
            return valueConverter;
        }
        if(ClassTypeUtil.isPrimitive(type)) {
            Class actualType = PrimitiveUtil.getReferenceType(type);
            return CONVERTER_MAP.get(actualType);
        }
        if(ClassTypeUtil.isArray(type)) {
            return CONVERTER_MAP.get(Array.class);
        }
        if(ClassTypeUtil.isCollection(type)) {
            return CONVERTER_MAP.get(Collection.class);
        }
        if(ClassTypeUtil.isEnum(type)) {
            return CONVERTER_MAP.get(Enum.class);
        }

        throw new UnsupportedOperationException("Unsupported class type " + type.getName());
    }

}
