package info.bowkett.ddt;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.*;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * Created by jbowkett on 27/04/15.
 */
public class PreparedStatementSpy implements PreparedStatement {

  private final PreparedStatement pstmt;
  private final String sql;
  private Map<Integer, Param> sqlIndexToParamMap = new ConcurrentHashMap<>();


  public PreparedStatementSpy(PreparedStatement pstmt, String sql) {
    this.pstmt = pstmt;
    this.sql = sql;
  }

  /**
   * Executes the SQL query in this <code>PreparedStatement</code> object
   * and returns the <code>ResultSet</code> object generated by the query.
   *
   * @return a <code>ResultSet</code> object that contains the data produced by the
   * query; never <code>null</code>
   * @throws java.sql.SQLException        if a database access error occurs;
   *                                      this method is called on a closed  <code>PreparedStatement</code> or the SQL
   *                                      statement does not return a <code>ResultSet</code> object
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   */
  @Override
  public ResultSet executeQuery() throws SQLException {
    return pstmt.executeQuery();
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object,
   * which must be an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   * <code>DELETE</code>; or an SQL statement that returns nothing,
   * such as a DDL statement.
   *
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException        if a database access error occurs;
   *                                      this method is called on a closed  <code>PreparedStatement</code>
   *                                      or the SQL statement returns a <code>ResultSet</code> object
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   */
  @Override
  public int executeUpdate() throws SQLException {
    return pstmt.executeUpdate();
  }

  /**
   * Sets the designated parameter to SQL <code>NULL</code>.
   * <p>
   * <P><B>Note:</B> You must specify the parameter's SQL type.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param sqlType        the SQL type code defined in <code>java.sql.Types</code>
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if <code>sqlType</code> is
   *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
   *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
   *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
   *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
   *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
   *                                                  this data type
   */
  @Override
  public void setNull(int parameterIndex, int sqlType) throws SQLException {
    pstmt.setNull(parameterIndex, sqlType);
    spy(parameterIndex, new NullParam());
  }

  /**
   * Sets the designated parameter to the given Java <code>boolean</code> value.
   * The driver converts this
   * to an SQL <code>BIT</code> or <code>BOOLEAN</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement;
   *                               if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setBoolean(int parameterIndex, boolean x) throws SQLException {
    pstmt.setBoolean(parameterIndex, x);
    spy(parameterIndex, new BooleanParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>byte</code> value.
   * The driver converts this
   * to an SQL <code>TINYINT</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setByte(int parameterIndex, byte x) throws SQLException {
    pstmt.setByte(parameterIndex, x);
    spy(parameterIndex, new ByteParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>short</code> value.
   * The driver converts this
   * to an SQL <code>SMALLINT</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setShort(int parameterIndex, short x) throws SQLException {
    pstmt.setShort(parameterIndex, x);
    spy(parameterIndex, new LongParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>int</code> value.
   * The driver converts this
   * to an SQL <code>INTEGER</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setInt(int parameterIndex, int x) throws SQLException {
    pstmt.setInt(parameterIndex, x);
    spy(parameterIndex, new LongParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>long</code> value.
   * The driver converts this
   * to an SQL <code>BIGINT</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setLong(int parameterIndex, long x) throws SQLException {
    pstmt.setLong(parameterIndex, x);
    spy(parameterIndex, new LongParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>float</code> value.
   * The driver converts this
   * to an SQL <code>REAL</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setFloat(int parameterIndex, float x) throws SQLException {
    pstmt.setFloat(parameterIndex, x);
    spy(parameterIndex, new DoubleParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>double</code> value.
   * The driver converts this
   * to an SQL <code>DOUBLE</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setDouble(int parameterIndex, double x) throws SQLException {
    pstmt.setDouble(parameterIndex, x);
    spy(parameterIndex, new DoubleParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.math.BigDecimal</code> value.
   * The driver converts this to an SQL <code>NUMERIC</code> value when
   * it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
    pstmt.setBigDecimal(parameterIndex, x);
    spy(parameterIndex, new BigDecimalParam(x));
  }

  /**
   * Sets the designated parameter to the given Java <code>String</code> value.
   * The driver converts this
   * to an SQL <code>VARCHAR</code> or <code>LONGVARCHAR</code> value
   * (depending on the argument's
   * size relative to the driver's limits on <code>VARCHAR</code> values)
   * when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setString(int parameterIndex, String x) throws SQLException {
    pstmt.setString(parameterIndex, x);
    spy(parameterIndex, new StringParam(x));
  }

  /**
   * Sets the designated parameter to the given Java array of bytes.  The driver converts
   * this to an SQL <code>VARBINARY</code> or <code>LONGVARBINARY</code>
   * (depending on the argument's size relative to the driver's limits on
   * <code>VARBINARY</code> values) when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setBytes(int parameterIndex, byte[] x) throws SQLException {
    pstmt.setBytes(parameterIndex, x);
    spy(parameterIndex, new ByteArrayParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Date</code> value
   * using the default time zone of the virtual machine that is running
   * the application.
   * The driver converts this
   * to an SQL <code>DATE</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setDate(int parameterIndex, Date x) throws SQLException {
    pstmt.setDate(parameterIndex, x);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Time</code> value.
   * The driver converts this
   * to an SQL <code>TIME</code> value when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setTime(int parameterIndex, Time x) throws SQLException {
    pstmt.setTime(parameterIndex, x);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Timestamp</code> value.
   * The driver
   * converts this to an SQL <code>TIMESTAMP</code> value when it sends it to the
   * database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
    pstmt.setTimestamp(parameterIndex, x);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes.
   * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code>. Data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from ASCII to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the Java input stream that contains the ASCII parameter value
   * @param length         the number of bytes in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
    pstmt.setAsciiStream(parameterIndex, x, length);
    spy(parameterIndex, new AsciiStreamParam());
  }

  /**
   * Sets the designated parameter to the given input stream, which
   * will have the specified number of bytes.
   * <p>
   * When a very large Unicode value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code> object. The data will be read from the
   * stream as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from Unicode to the database char format.
   * <p>
   * The byte format of the Unicode stream must be a Java UTF-8, as defined in the
   * Java Virtual Machine Specification.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              a <code>java.io.InputStream</code> object that contains the
   *                       Unicode parameter value
   * @param length         the number of bytes in the stream
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method
   * @deprecated Use {@code setCharacterStream}
   */
  @Override
  public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
    pstmt.setUnicodeStream(parameterIndex, x, length);
    spy(parameterIndex, new UnicodeParam());
  }

  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes.
   * When a very large binary value is input to a <code>LONGVARBINARY</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code> object. The data will be read from the
   * stream as needed until end-of-file is reached.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the java input stream which contains the binary parameter value
   * @param length         the number of bytes in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
    pstmt.setBinaryStream(parameterIndex, x, length);
    spy(parameterIndex, new InputStreamParam());
  }

  /**
   * Clears the current parameter values immediately.
   * <P>In general, parameter values remain in force for repeated use of a
   * statement. Setting a parameter value automatically clears its
   * previous value.  However, in some cases it is useful to immediately
   * release the resources used by the current parameter values; this can
   * be done by calling the method <code>clearParameters</code>.
   *
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   */
  @Override
  public void clearParameters() throws SQLException {
    pstmt.clearParameters();
    this.sqlIndexToParamMap.clear();
  }

  /**
   * Sets the value of the designated parameter with the given object.
   * <p>
   * This method is similar to {@link #setObject(int parameterIndex,
   * Object x, int targetSqlType, int scaleOrLength)},
   * except that it assumes a scale of zero.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the object containing the input parameter value
   * @param targetSqlType  the SQL type (as defined in java.sql.Types) to be
   *                       sent to the database
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or this
   *                                                  method is called on a closed PreparedStatement
   * @throws java.sql.SQLFeatureNotSupportedException if
   *                                                  the JDBC driver does not support the specified targetSqlType
   * @see java.sql.Types
   */
  @Override
  public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
    pstmt.setObject(parameterIndex, x, targetSqlType);
    spy(parameterIndex, new ObjectParam(x));
  }

  /**
   * <p>Sets the value of the designated parameter using the given object.
   * <p>
   * <p>The JDBC specification specifies a standard mapping from
   * Java <code>Object</code> types to SQL types.  The given argument
   * will be converted to the corresponding SQL type before being
   * sent to the database.
   * <p>
   * <p>Note that this method may be used to pass datatabase-
   * specific abstract data types, by using a driver-specific Java
   * type.
   * <p>
   * If the object is of a class implementing the interface <code>SQLData</code>,
   * the JDBC driver should call the method <code>SQLData.writeSQL</code>
   * to write it to the SQL data stream.
   * If, on the other hand, the object is of a class implementing
   * <code>Ref</code>, <code>Blob</code>, <code>Clob</code>,  <code>NClob</code>,
   * <code>Struct</code>, <code>java.net.URL</code>, <code>RowId</code>, <code>SQLXML</code>
   * or <code>Array</code>, the driver should pass it to the database as a
   * value of the corresponding SQL type.
   * <p>
   * <b>Note:</b> Not all databases allow for a non-typed Null to be sent to
   * the backend. For maximum portability, the <code>setNull</code> or the
   * <code>setObject(int parameterIndex, Object x, int sqlType)</code>
   * method should be used
   * instead of <code>setObject(int parameterIndex, Object x)</code>.
   * <p>
   * <b>Note:</b> This method throws an exception if there is an ambiguity, for example, if the
   * object is of a class implementing more than one of the interfaces named above.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the object containing the input parameter value
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs;
   *                               this method is called on a closed <code>PreparedStatement</code>
   *                               or the type of the given object is ambiguous
   */
  @Override
  public void setObject(int parameterIndex, Object x) throws SQLException {
    pstmt.setObject(parameterIndex, x);
    spy(parameterIndex, new ObjectParam(x));
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object,
   * which may be any kind of SQL statement.
   * Some prepared statements return multiple results; the <code>execute</code>
   * method handles these complex statements as well as the simpler
   * form of statements handled by the methods <code>executeQuery</code>
   * and <code>executeUpdate</code>.
   * <p>
   * The <code>execute</code> method returns a <code>boolean</code> to
   * indicate the form of the first result.  You must call either the method
   * <code>getResultSet</code> or <code>getUpdateCount</code>
   * to retrieve the result; you must call <code>getMoreResults</code> to
   * move to any subsequent result(s).
   *
   * @return <code>true</code> if the first result is a <code>ResultSet</code>
   * object; <code>false</code> if the first result is an update
   * count or there is no result
   * @throws java.sql.SQLException        if a database access error occurs;
   *                                      this method is called on a closed <code>PreparedStatement</code>
   *                                      or an argument is supplied to this method
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @see java.sql.Statement#execute
   * @see java.sql.Statement#getResultSet
   * @see java.sql.Statement#getUpdateCount
   * @see java.sql.Statement#getMoreResults
   */
  @Override
  public boolean execute() throws SQLException {
    boolean result = false;
    try {
      result = pstmt.execute();
      this.sqlIndexToParamMap.clear();
    }
    finally {
      return result;
    }
  }

  /**
   * Adds a set of parameters to this <code>PreparedStatement</code>
   * object's batch of commands.
   *
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @see java.sql.Statement#addBatch
   * @since 1.2
   */
  @Override
  public void addBatch() throws SQLException {
    pstmt.addBatch();
  }

  /**
   * Sets the designated parameter to the given <code>Reader</code>
   * object, which is the given number of characters long.
   * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.Reader</code> object. The data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from UNICODE to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param reader         the <code>java.io.Reader</code> object that contains the
   *                       Unicode data
   * @param length         the number of characters in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.2
   */
  @Override
  public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
    pstmt.setCharacterStream(parameterIndex, reader, length);
    spy(parameterIndex, new CharstreamParam());
  }

  /**
   * Sets the designated parameter to the given
   * <code>REF(&lt;structured-type&gt;)</code> value.
   * The driver converts this to an SQL <code>REF</code> value when it
   * sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              an SQL <code>REF</code> value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public void setRef(int parameterIndex, Ref x) throws SQLException {
    pstmt.setRef(parameterIndex, x);
    spy(parameterIndex, new RefParam());
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Blob</code> object.
   * The driver converts this to an SQL <code>BLOB</code> value when it
   * sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              a <code>Blob</code> object that maps an SQL <code>BLOB</code> value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public void setBlob(int parameterIndex, Blob x) throws SQLException {
    pstmt.setBlob(parameterIndex, x);
    spy(parameterIndex, new BlobParam());
  }

  @Override
  public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
    pstmt.setBlob(parameterIndex, inputStream);
    spy(parameterIndex, new BlobParam());
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Clob</code> object.
   * The driver converts this to an SQL <code>CLOB</code> value when it
   * sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              a <code>Clob</code> object that maps an SQL <code>CLOB</code> value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public void setClob(int parameterIndex, Clob x) throws SQLException {
    pstmt.setClob(parameterIndex, x);
    spy(parameterIndex, new ClobParam());
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Array</code> object.
   * The driver converts this to an SQL <code>ARRAY</code> value when it
   * sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              an <code>Array</code> object that maps an SQL <code>ARRAY</code> value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public void setArray(int parameterIndex, Array x) throws SQLException {
    pstmt.setArray(parameterIndex, x);
    spy(parameterIndex, new ArrayParam(x));
  }

  /**
   * Retrieves a <code>ResultSetMetaData</code> object that contains
   * information about the columns of the <code>ResultSet</code> object
   * that will be returned when this <code>PreparedStatement</code> object
   * is executed.
   * <p>
   * Because a <code>PreparedStatement</code> object is precompiled, it is
   * possible to know about the <code>ResultSet</code> object that it will
   * return without having to execute it.  Consequently, it is possible
   * to invoke the method <code>getMetaData</code> on a
   * <code>PreparedStatement</code> object rather than waiting to execute
   * it and then invoking the <code>ResultSet.getMetaData</code> method
   * on the <code>ResultSet</code> object that is returned.
   * <p>
   * <B>NOTE:</B> Using this method may be expensive for some drivers due
   * to the lack of underlying DBMS support.
   *
   * @return the description of a <code>ResultSet</code> object's columns or
   * <code>null</code> if the driver cannot return a
   * <code>ResultSetMetaData</code> object
   * @throws java.sql.SQLException                    if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method
   * @since 1.2
   */
  @Override
  public ResultSetMetaData getMetaData() throws SQLException {
    return pstmt.getMetaData();
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Date</code> value,
   * using the given <code>Calendar</code> object.  The driver uses
   * the <code>Calendar</code> object to construct an SQL <code>DATE</code> value,
   * which the driver then sends to the database.  With
   * a <code>Calendar</code> object, the driver can calculate the date
   * taking into account a custom timezone.  If no
   * <code>Calendar</code> object is specified, the driver uses the default
   * timezone, which is that of the virtual machine running the application.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @param cal            the <code>Calendar</code> object the driver will use
   *                       to construct the date
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.2
   */
  @Override
  public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
    pstmt.setDate(parameterIndex, x, cal);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Time</code> value,
   * using the given <code>Calendar</code> object.  The driver uses
   * the <code>Calendar</code> object to construct an SQL <code>TIME</code> value,
   * which the driver then sends to the database.  With
   * a <code>Calendar</code> object, the driver can calculate the time
   * taking into account a custom timezone.  If no
   * <code>Calendar</code> object is specified, the driver uses the default
   * timezone, which is that of the virtual machine running the application.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @param cal            the <code>Calendar</code> object the driver will use
   *                       to construct the time
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.2
   */
  @Override
  public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
    pstmt.setTime(parameterIndex, x, cal);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.Timestamp</code> value,
   * using the given <code>Calendar</code> object.  The driver uses
   * the <code>Calendar</code> object to construct an SQL <code>TIMESTAMP</code> value,
   * which the driver then sends to the database.  With a
   * <code>Calendar</code> object, the driver can calculate the timestamp
   * taking into account a custom timezone.  If no
   * <code>Calendar</code> object is specified, the driver uses the default
   * timezone, which is that of the virtual machine running the application.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @param cal            the <code>Calendar</code> object the driver will use
   *                       to construct the timestamp
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.2
   */
  @Override
  public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
    pstmt.setTimestamp(parameterIndex, x, cal);
    spy(parameterIndex, new DateParam(x));
  }

  /**
   * Sets the designated parameter to SQL <code>NULL</code>.
   * This version of the method <code>setNull</code> should
   * be used for user-defined types and REF type parameters.  Examples
   * of user-defined types include: STRUCT, DISTINCT, JAVA_OBJECT, and
   * named array types.
   * <p>
   * <P><B>Note:</B> To be portable, applications must give the
   * SQL type code and the fully-qualified SQL type name when specifying
   * a NULL user-defined or REF parameter.  In the case of a user-defined type
   * the name is the type name of the parameter itself.  For a REF
   * parameter, the name is the type name of the referenced type.  If
   * a JDBC driver does not need the type code or type name information,
   * it may ignore it.
   * <p>
   * Although it is intended for user-defined and Ref parameters,
   * this method may be used to set a null parameter of any JDBC type.
   * If the parameter does not have a user-defined or REF type, the given
   * typeName is ignored.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param sqlType        a value from <code>java.sql.Types</code>
   * @param typeName       the fully-qualified name of an SQL user-defined type;
   *                       ignored if the parameter is not a user-defined type or REF
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if <code>sqlType</code> is
   *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
   *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
   *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
   *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
   *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
   *                                                  this data type or if the JDBC driver does not support this method
   * @since 1.2
   */
  @Override
  public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
    pstmt.setNull(parameterIndex, sqlType, typeName);
    spy(parameterIndex, new NullParam());
  }

  /**
   * Sets the designated parameter to the given <code>java.net.URL</code> value.
   * The driver converts this to an SQL <code>DATALINK</code> value
   * when it sends it to the database.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the <code>java.net.URL</code> object to be set
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.4
   */
  @Override
  public void setURL(int parameterIndex, URL x) throws SQLException {
    pstmt.setURL(parameterIndex, x);
    spy(parameterIndex, new URLParam(x));
  }

  /**
   * Retrieves the number, types and properties of this
   * <code>PreparedStatement</code> object's parameters.
   *
   * @return a <code>ParameterMetaData</code> object that contains information
   * about the number, types and properties for each
   * parameter marker of this <code>PreparedStatement</code> object
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @see java.sql.ParameterMetaData
   * @since 1.4
   */
  @Override
  public ParameterMetaData getParameterMetaData() throws SQLException {
    return pstmt.getParameterMetaData();
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.RowId</code> object. The
   * driver converts this to a SQL <code>ROWID</code> value when it sends it
   * to the database
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setRowId(int parameterIndex, RowId x) throws SQLException {
    pstmt.setRowId(parameterIndex, x);
    spy(parameterIndex, new RowIdParam());
  }

  /**
   * Sets the designated parameter to the given <code>String</code> object.
   * The driver converts this to a SQL <code>NCHAR</code> or
   * <code>NVARCHAR</code> or <code>LONGNVARCHAR</code> value
   * (depending on the argument's
   * size relative to the driver's limits on <code>NVARCHAR</code> values)
   * when it sends it to the database.
   *
   * @param parameterIndex of the first parameter is 1, the second is 2, ...
   * @param value          the parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if the driver does not support national
   *                                                  character sets;  if the driver can detect that a data conversion
   *                                                  error could occur; if a database access error occurs; or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setNString(int parameterIndex, String value) throws SQLException {
    pstmt.setNString(parameterIndex, value);
    spy(parameterIndex, new StringParam(value));
  }

  /**
   * Sets the designated parameter to a <code>Reader</code> object. The
   * <code>Reader</code> reads the data till end-of-file is reached. The
   * driver does the necessary conversion from Java character format to
   * the national character set in the database.
   *
   * @param parameterIndex of the first parameter is 1, the second is 2, ...
   * @param value          the parameter value
   * @param length         the number of characters in the parameter data.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if the driver does not support national
   *                                                  character sets;  if the driver can detect that a data conversion
   *                                                  error could occur; if a database access error occurs; or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException {
    pstmt.setNCharacterStream(parameterIndex, value, length);
    spy(parameterIndex, new NCharstreamParam());
  }

  /**
   * Sets the designated parameter to a <code>java.sql.NClob</code> object. The driver converts this to a
   * SQL <code>NCLOB</code> value when it sends it to the database.
   *
   * @param parameterIndex of the first parameter is 1, the second is 2, ...
   * @param value          the parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if the driver does not support national
   *                                                  character sets;  if the driver can detect that a data conversion
   *                                                  error could occur; if a database access error occurs; or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setNClob(int parameterIndex, NClob value) throws SQLException {
    pstmt.setNClob(parameterIndex, value);
    spy(parameterIndex, new NClobParam());
  }

  /**
   * Sets the designated parameter to a <code>Reader</code> object.  The reader must contain  the number
   * of characters specified by length otherwise a <code>SQLException</code> will be
   * generated when the <code>PreparedStatement</code> is executed.
   * This method differs from the <code>setCharacterStream (int, Reader, int)</code> method
   * because it informs the driver that the parameter value should be sent to
   * the server as a <code>CLOB</code>.  When the <code>setCharacterStream</code> method is used, the
   * driver may have to do extra work to determine whether the parameter
   * data should be sent to the server as a <code>LONGVARCHAR</code> or a <code>CLOB</code>
   *
   * @param parameterIndex index of the first parameter is 1, the second is 2, ...
   * @param reader         An object that contains the data to set the parameter value to.
   * @param length         the number of characters in the parameter data.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs; this method is called on
   *                                                  a closed <code>PreparedStatement</code> or if the length specified is less than zero.
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
    pstmt.setClob(parameterIndex, reader, length);
    spy(parameterIndex, new ClobParam());
  }

  @Override
  public void setClob(int parameterIndex, Reader reader) throws SQLException {
    pstmt.setClob(parameterIndex, reader);
    spy(parameterIndex, new ClobParam());
  }

  /**
   * Sets the designated parameter to a <code>InputStream</code> object.  The inputstream must contain  the number
   * of characters specified by length otherwise a <code>SQLException</code> will be
   * generated when the <code>PreparedStatement</code> is executed.
   * This method differs from the <code>setBinaryStream (int, InputStream, int)</code>
   * method because it informs the driver that the parameter value should be
   * sent to the server as a <code>BLOB</code>.  When the <code>setBinaryStream</code> method is used,
   * the driver may have to do extra work to determine whether the parameter
   * data should be sent to the server as a <code>LONGVARBINARY</code> or a <code>BLOB</code>
   *
   * @param parameterIndex index of the first parameter is 1,
   *                       the second is 2, ...
   * @param inputStream    An object that contains the data to set the parameter
   *                       value to.
   * @param length         the number of bytes in the parameter data.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs;
   *                                                  this method is called on a closed <code>PreparedStatement</code>;
   *                                                  if the length specified
   *                                                  is less than zero or if the number of bytes in the inputstream does not match
   *                                                  the specified length.
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
    pstmt.setBlob(parameterIndex, inputStream, length);
    spy(parameterIndex, new BlobParam());
  }

  /**
   * Sets the designated parameter to a <code>Reader</code> object.  The reader must contain  the number
   * of characters specified by length otherwise a <code>SQLException</code> will be
   * generated when the <code>PreparedStatement</code> is executed.
   * This method differs from the <code>setCharacterStream (int, Reader, int)</code> method
   * because it informs the driver that the parameter value should be sent to
   * the server as a <code>NCLOB</code>.  When the <code>setCharacterStream</code> method is used, the
   * driver may have to do extra work to determine whether the parameter
   * data should be sent to the server as a <code>LONGNVARCHAR</code> or a <code>NCLOB</code>
   *
   * @param parameterIndex index of the first parameter is 1, the second is 2, ...
   * @param reader         An object that contains the data to set the parameter value to.
   * @param length         the number of characters in the parameter data.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if the length specified is less than zero;
   *                                                  if the driver does not support national character sets;
   *                                                  if the driver can detect that a data conversion
   *                                                  error could occur;  if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
    pstmt.setNClob(parameterIndex, reader, length);
    spy(parameterIndex, new NClobParam());
  }

  @Override
  public void setNClob(int parameterIndex, Reader reader) throws SQLException {
    pstmt.setNClob(parameterIndex, reader);
    spy(parameterIndex, new NClobParam());
  }

  /**
   * Sets the designated parameter to the given <code>java.sql.SQLXML</code> object.
   * The driver converts this to an
   * SQL <code>XML</code> value when it sends it to the database.
   * <p>
   *
   * @param parameterIndex index of the first parameter is 1, the second is 2, ...
   * @param xmlObject      a <code>SQLXML</code> object that maps an SQL <code>XML</code> value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs;
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   *                                                  or the <code>java.xml.transform.Result</code>,
   *                                                  <code>Writer</code> or <code>OutputStream</code> has not been closed for
   *                                                  the <code>SQLXML</code> object
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
    pstmt.setSQLXML(parameterIndex, xmlObject);
    spy(parameterIndex, new SQLXMLParam());
  }

  /**
   * <p>Sets the value of the designated parameter with the given object.
   * <p>
   * If the second argument is an <code>InputStream</code> then the stream must contain
   * the number of bytes specified by scaleOrLength.  If the second argument is a
   * <code>Reader</code> then the reader must contain the number of characters specified
   * by scaleOrLength. If these conditions are not true the driver will generate a
   * <code>SQLException</code> when the prepared statement is executed.
   * <p>
   * <p>The given Java object will be converted to the given targetSqlType
   * before being sent to the database.
   * <p>
   * If the object has a custom mapping (is of a class implementing the
   * interface <code>SQLData</code>),
   * the JDBC driver should call the method <code>SQLData.writeSQL</code> to
   * write it to the SQL data stream.
   * If, on the other hand, the object is of a class implementing
   * <code>Ref</code>, <code>Blob</code>, <code>Clob</code>,  <code>NClob</code>,
   * <code>Struct</code>, <code>java.net.URL</code>,
   * or <code>Array</code>, the driver should pass it to the database as a
   * value of the corresponding SQL type.
   * <p>
   * <p>Note that this method may be used to pass database-specific
   * abstract data types.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the object containing the input parameter value
   * @param targetSqlType  the SQL type (as defined in java.sql.Types) to be
   *                       sent to the database. The scale argument may further qualify this type.
   * @param scaleOrLength  for <code>java.sql.Types.DECIMAL</code>
   *                       or <code>java.sql.Types.NUMERIC types</code>,
   *                       this is the number of digits after the decimal point. For
   *                       Java Object types <code>InputStream</code> and <code>Reader</code>,
   *                       this is the length
   *                       of the data in the stream or reader.  For all other types,
   *                       this value will be ignored.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs;
   *                                                  this method is called on a closed <code>PreparedStatement</code> or
   *                                                  if the Java Object specified by x is an InputStream
   *                                                  or Reader object and the value of the scale parameter is less
   *                                                  than zero
   * @throws java.sql.SQLFeatureNotSupportedException if
   *                                                  the JDBC driver does not support the specified targetSqlType
   * @see java.sql.Types
   */
  @Override
  public void setObject(int parameterIndex, Object x, int targetSqlType, int scaleOrLength) throws SQLException {
    pstmt.setObject(parameterIndex, x, targetSqlType, scaleOrLength);
    spy(parameterIndex, new ObjectParam(x));
  }

  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes.
   * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code>. Data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from ASCII to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the Java input stream that contains the ASCII parameter value
   * @param length         the number of bytes in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.6
   */
  @Override
  public void setAsciiStream(int parameterIndex, InputStream x, long length) throws SQLException {
    pstmt.setAsciiStream(parameterIndex, x, length);
    spy(parameterIndex, new AsciiStreamParam());
  }

  /**
   * Sets the designated parameter to the given input stream, which will have
   * the specified number of bytes.
   * When a very large binary value is input to a <code>LONGVARBINARY</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code> object. The data will be read from the
   * stream as needed until end-of-file is reached.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the java input stream which contains the binary parameter value
   * @param length         the number of bytes in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.6
   */
  @Override
  public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException {
    pstmt.setBinaryStream(parameterIndex, x, length);
    spy(parameterIndex, new InputStreamParam());
  }

  /**
   * Sets the designated parameter to the given <code>Reader</code>
   * object, which is the given number of characters long.
   * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.Reader</code> object. The data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from UNICODE to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param reader         the <code>java.io.Reader</code> object that contains the
   *                       Unicode data
   * @param length         the number of characters in the stream
   * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
   *                               marker in the SQL statement; if a database access error occurs or
   *                               this method is called on a closed <code>PreparedStatement</code>
   * @since 1.6
   */
  @Override
  public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
    pstmt.setCharacterStream(parameterIndex, reader, length);
    spy(parameterIndex, new CharstreamParam());
  }

  /**
   * Sets the designated parameter to the given input stream.
   * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code>. Data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from ASCII to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
   * it might be more efficient to use a version of
   * <code>setAsciiStream</code> which takes a length parameter.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the Java input stream that contains the ASCII parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setAsciiStream(int parameterIndex, InputStream x) throws SQLException {
    pstmt.setAsciiStream(parameterIndex, x);
    spy(parameterIndex, new AsciiStreamParam());
  }

  /**
   * Sets the designated parameter to the given input stream.
   * When a very large binary value is input to a <code>LONGVARBINARY</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.InputStream</code> object. The data will be read from the
   * stream as needed until end-of-file is reached.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
   * it might be more efficient to use a version of
   * <code>setBinaryStream</code> which takes a length parameter.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the java input stream which contains the binary parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException {
    this.pstmt.setBinaryStream(parameterIndex, x);
    spy(parameterIndex, new InputStreamParam());
  }

  /**
   * Sets the designated parameter to the given <code>Reader</code>
   * object.
   * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
   * parameter, it may be more practical to send it via a
   * <code>java.io.Reader</code> object. The data will be read from the stream
   * as needed until end-of-file is reached.  The JDBC driver will
   * do any necessary conversion from UNICODE to the database char format.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
   * it might be more efficient to use a version of
   * <code>setCharacterStream</code> which takes a length parameter.
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param reader         the <code>java.io.Reader</code> object that contains the
   *                       Unicode data
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if a database access error occurs or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
    pstmt.setCharacterStream(parameterIndex, reader);
    spy(parameterIndex, new CharstreamParam());
  }

  /**
   * Sets the designated parameter to a <code>Reader</code> object. The
   * <code>Reader</code> reads the data till end-of-file is reached. The
   * driver does the necessary conversion from Java character format to
   * the national character set in the database.
   * <p>
   * <P><B>Note:</B> This stream object can either be a standard
   * Java stream object or your own subclass that implements the
   * standard interface.
   * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
   * it might be more efficient to use a version of
   * <code>setNCharacterStream</code> which takes a length parameter.
   *
   * @param parameterIndex of the first parameter is 1, the second is 2, ...
   * @param value          the parameter value
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
   *                                                  marker in the SQL statement; if the driver does not support national
   *                                                  character sets;  if the driver can detect that a data conversion
   *                                                  error could occur; if a database access error occurs; or
   *                                                  this method is called on a closed <code>PreparedStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.6
   */
  @Override
  public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException {
    pstmt.setNCharacterStream(parameterIndex, value);
    spy(parameterIndex, new NCharstreamParam());
  }

  /**
   * <p>Sets the value of the designated parameter with the given object.
   * <p>
   * If the second argument is an {@code InputStream} then the stream
   * must contain the number of bytes specified by scaleOrLength.
   * If the second argument is a {@code Reader} then the reader must
   * contain the number of characters specified by scaleOrLength. If these
   * conditions are not true the driver will generate a
   * {@code SQLException} when the prepared statement is executed.
   * <p>
   * <p>The given Java object will be converted to the given targetSqlType
   * before being sent to the database.
   * <p>
   * If the object has a custom mapping (is of a class implementing the
   * interface {@code SQLData}),
   * the JDBC driver should call the method {@code SQLData.writeSQL} to
   * write it to the SQL data stream.
   * If, on the other hand, the object is of a class implementing
   * {@code Ref}, {@code Blob}, {@code Clob},  {@code NClob},
   * {@code Struct}, {@code java.net.URL},
   * or {@code Array}, the driver should pass it to the database as a
   * value of the corresponding SQL type.
   * <p>
   * <p>Note that this method may be used to pass database-specific
   * abstract data types.
   * <p>
   * The default implementation will throw {@code SQLFeatureNotSupportedException}
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the object containing the input parameter value
   * @param targetSqlType  the SQL type to be sent to the database. The
   *                       scale argument may further qualify this type.
   * @param scaleOrLength  for {@code java.sql.JDBCType.DECIMAL}
   *                       or {@code java.sql.JDBCType.NUMERIC types},
   *                       this is the number of digits after the decimal point. For
   *                       Java Object types {@code InputStream} and {@code Reader},
   *                       this is the length
   *                       of the data in the stream or reader.  For all other types,
   *                       this value will be ignored.
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a
   *                                                  parameter marker in the SQL statement; if a database access error occurs
   *                                                  or this method is called on a closed {@code PreparedStatement}  or
   *                                                  if the Java Object specified by x is an InputStream
   *                                                  or Reader object and the value of the scale parameter is less
   *                                                  than zero
   * @throws java.sql.SQLFeatureNotSupportedException if
   *                                                  the JDBC driver does not support the specified targetSqlType
   * @see java.sql.JDBCType
   * @see java.sql.SQLType
   * @since 1.8
   */
  @Override
  public void setObject(int parameterIndex, Object x, SQLType targetSqlType, int scaleOrLength) throws SQLException {
    pstmt.setObject(parameterIndex, x, targetSqlType, scaleOrLength);
    spy(parameterIndex, new ObjectParam(x));
  }

  /**
   * Sets the value of the designated parameter with the given object.
   * <p>
   * This method is similar to {@link #setObject(int parameterIndex,
   * Object x, java.sql.SQLType targetSqlType, int scaleOrLength)},
   * except that it assumes a scale of zero.
   * <p>
   * The default implementation will throw {@code SQLFeatureNotSupportedException}
   *
   * @param parameterIndex the first parameter is 1, the second is 2, ...
   * @param x              the object containing the input parameter value
   * @param targetSqlType  the SQL type to be sent to the database
   * @throws java.sql.SQLException                    if parameterIndex does not correspond to a
   *                                                  parameter marker in the SQL statement; if a database access error occurs
   *                                                  or this method is called on a closed {@code PreparedStatement}
   * @throws java.sql.SQLFeatureNotSupportedException if
   *                                                  the JDBC driver does not support the specified targetSqlType
   * @see java.sql.JDBCType
   * @see java.sql.SQLType
   * @since 1.8
   */
  @Override
  public void setObject(int parameterIndex, Object x, SQLType targetSqlType) throws SQLException {
    pstmt.setObject(parameterIndex, x, targetSqlType);
    spy(parameterIndex, new ObjectParam(x));
  }

  /**
   * Executes the SQL statement in this <code>PreparedStatement</code> object,
   * which must be an SQL Data Manipulation Language (DML) statement,
   * such as <code>INSERT</code>, <code>UPDATE</code> or
   * <code>DELETE</code>; or an SQL statement that returns nothing,
   * such as a DDL statement.
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * The default implementation will throw {@code UnsupportedOperationException}
   *
   * @return either (1) the row count for SQL Data Manipulation Language
   * (DML) statements or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException        if a database access error occurs;
   *                                      this method is called on a closed  <code>PreparedStatement</code>
   *                                      or the SQL statement returns a <code>ResultSet</code> object
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate() throws SQLException {
    return pstmt.executeLargeUpdate();
  }

  /**
   * Returns an object that implements the given interface to allow access to
   * non-standard methods, or standard methods not exposed by the proxy.
   * <p>
   * If the receiver implements the interface then the result is the receiver
   * or a proxy for the receiver. If the receiver is a wrapper
   * and the wrapped object implements the interface then the result is the
   * wrapped object or a proxy for the wrapped object. Otherwise return the
   * the result of calling <code>unwrap</code> recursively on the wrapped object
   * or a proxy for that result. If the receiver is not a
   * wrapper and does not implement the interface, then an <code>SQLException</code> is thrown.
   *
   * @param iface A Class defining an interface that the result must implement.
   * @return an object that implements the interface. May be a proxy for the actual implementing object.
   * @throws java.sql.SQLException If no object found that implements the interface
   * @since 1.6
   */
  @Override
  public <T> T unwrap(Class<T> iface) throws SQLException {
    return pstmt.unwrap(iface);
  }

  /**
   * Returns true if this either implements the interface argument or is directly or indirectly a wrapper
   * for an object that does. Returns false otherwise. If this implements the interface then return true,
   * else if this is a wrapper then return the result of recursively calling <code>isWrapperFor</code> on the wrapped
   * object. If this does not implement the interface and is not a wrapper, return false.
   * This method should be implemented as a low-cost operation compared to <code>unwrap</code> so that
   * callers can use this method to avoid expensive <code>unwrap</code> calls that may fail. If this method
   * returns true then calling <code>unwrap</code> with the same argument should succeed.
   *
   * @param iface a Class defining an interface.
   * @return true if this implements the interface or directly or indirectly wraps an object that does.
   * @throws java.sql.SQLException if an error occurs while determining whether this is a wrapper
   *                               for an object with the given interface.
   * @since 1.6
   */
  @Override
  public boolean isWrapperFor(Class<?> iface) throws SQLException {
    return pstmt.isWrapperFor(iface);
  }

  /**
   * Executes the given SQL statement, which returns a single
   * <code>ResultSet</code> object.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql an SQL statement to be sent to the database, typically a
   *            static SQL <code>SELECT</code> statement
   * @return a <code>ResultSet</code> object that contains the data produced
   * by the given query; never <code>null</code>
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code>, the given
   *                                      SQL statement produces anything other than a single
   *                                      <code>ResultSet</code> object, the method is called on a
   *                                      <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   */
  @Override
  public ResultSet executeQuery(String sql) throws SQLException {
    return pstmt.executeQuery(sql);
  }

  /**
   * Executes the given SQL statement, which may be an <code>INSERT</code>,
   * <code>UPDATE</code>, or <code>DELETE</code> statement or an
   * SQL statement that returns nothing, such as an SQL DDL statement.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   *            <code>DELETE</code>; or an SQL statement that returns nothing,
   *            such as a DDL statement.
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code>, the given
   *                                      SQL statement produces a <code>ResultSet</code> object, the method is called on a
   *                                      <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   */
  @Override
  public int executeUpdate(String sql) throws SQLException {
    return pstmt.executeUpdate(sql);
  }

  /**
   * Releases this <code>Statement</code> object's database
   * and JDBC resources immediately instead of waiting for
   * this to happen when it is automatically closed.
   * It is generally good practice to release resources as soon as
   * you are finished with them to avoid tying up database
   * resources.
   * <p>
   * Calling the method <code>close</code> on a <code>Statement</code>
   * object that is already closed has no effect.
   * <p>
   * <B>Note:</B>When a <code>Statement</code> object is
   * closed, its current <code>ResultSet</code> object, if one exists, is
   * also closed.
   *
   * @throws java.sql.SQLException if a database access error occurs
   */
  @Override
  public void close() throws SQLException {
    pstmt.close();
  }

  /**
   * Retrieves the maximum number of bytes that can be
   * returned for character and binary column values in a <code>ResultSet</code>
   * object produced by this <code>Statement</code> object.
   * This limit applies only to  <code>BINARY</code>, <code>VARBINARY</code>,
   * <code>LONGVARBINARY</code>, <code>CHAR</code>, <code>VARCHAR</code>,
   * <code>NCHAR</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>
   * and <code>LONGVARCHAR</code> columns.  If the limit is exceeded, the
   * excess data is silently discarded.
   *
   * @return the current column size limit for columns storing character and
   * binary values; zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setMaxFieldSize
   */
  @Override
  public int getMaxFieldSize() throws SQLException {
    return pstmt.getMaxFieldSize();
  }

  /**
   * Sets the limit for the maximum number of bytes that can be returned for
   * character and binary column values in a <code>ResultSet</code>
   * object produced by this <code>Statement</code> object.
   * <p>
   * This limit applies
   * only to <code>BINARY</code>, <code>VARBINARY</code>,
   * <code>LONGVARBINARY</code>, <code>CHAR</code>, <code>VARCHAR</code>,
   * <code>NCHAR</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code> and
   * <code>LONGVARCHAR</code> fields.  If the limit is exceeded, the excess data
   * is silently discarded. For maximum portability, use values
   * greater than 256.
   *
   * @param max the new column size limit in bytes; zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>
   *                               or the condition {@code max >= 0} is not satisfied
   * @see #getMaxFieldSize
   */
  @Override
  public void setMaxFieldSize(int max) throws SQLException {
    pstmt.setMaxFieldSize(max);
  }

  /**
   * Retrieves the maximum number of rows that a
   * <code>ResultSet</code> object produced by this
   * <code>Statement</code> object can contain.  If this limit is exceeded,
   * the excess rows are silently dropped.
   *
   * @return the current maximum number of rows for a <code>ResultSet</code>
   * object produced by this <code>Statement</code> object;
   * zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setMaxRows
   */
  @Override
  public int getMaxRows() throws SQLException {
    return pstmt.getMaxRows();
  }

  /**
   * Sets the limit for the maximum number of rows that any
   * <code>ResultSet</code> object  generated by this <code>Statement</code>
   * object can contain to the given number.
   * If the limit is exceeded, the excess
   * rows are silently dropped.
   *
   * @param max the new max rows limit; zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>
   *                               or the condition {@code max >= 0} is not satisfied
   * @see #getMaxRows
   */
  @Override
  public void setMaxRows(int max) throws SQLException {
    pstmt.setMaxRows(max);
  }

  /**
   * Sets escape processing on or off.
   * If escape scanning is on (the default), the driver will do
   * escape substitution before sending the SQL statement to the database.
   * <p>
   * The {@code Connection} and {@code DataSource} property
   * {@code escapeProcessing} may be used to change the default escape processing
   * behavior.  A value of true (the default) enables escape Processing for
   * all {@code Statement} objects. A value of false disables escape processing
   * for all {@code Statement} objects.  The {@code setEscapeProcessing}
   * method may be used to specify the escape processing behavior for an
   * individual {@code Statement} object.
   * <p>
   * Note: Since prepared statements have usually been parsed prior
   * to making this call, disabling escape processing for
   * <code>PreparedStatements</code> objects will have no effect.
   *
   * @param enable <code>true</code> to enable escape processing;
   *               <code>false</code> to disable it
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   */
  @Override
  public void setEscapeProcessing(boolean enable) throws SQLException {
    pstmt.setEscapeProcessing(enable);
  }

  /**
   * Retrieves the number of seconds the driver will
   * wait for a <code>Statement</code> object to execute.
   * If the limit is exceeded, a
   * <code>SQLException</code> is thrown.
   *
   * @return the current query timeout limit in seconds; zero means there is
   * no limit
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setQueryTimeout
   */
  @Override
  public int getQueryTimeout() throws SQLException {
    return pstmt.getQueryTimeout();
  }

  /**
   * Sets the number of seconds the driver will wait for a
   * <code>Statement</code> object to execute to the given number of seconds.
   * By default there is no limit on the amount of time allowed for a running
   * statement to complete. If the limit is exceeded, an
   * <code>SQLTimeoutException</code> is thrown.
   * A JDBC driver must apply this limit to the <code>execute</code>,
   * <code>executeQuery</code> and <code>executeUpdate</code> methods.
   * <p>
   * <strong>Note:</strong> JDBC driver implementations may also apply this
   * limit to {@code ResultSet} methods
   * (consult your driver vendor documentation for details).
   * <p>
   * <strong>Note:</strong> In the case of {@code Statement} batching, it is
   * implementation defined as to whether the time-out is applied to
   * individual SQL commands added via the {@code addBatch} method or to
   * the entire batch of SQL commands invoked by the {@code executeBatch}
   * method (consult your driver vendor documentation for details).
   *
   * @param seconds the new query timeout limit in seconds; zero means
   *                there is no limit
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>
   *                               or the condition {@code seconds >= 0} is not satisfied
   * @see #getQueryTimeout
   */
  @Override
  public void setQueryTimeout(int seconds) throws SQLException {
    pstmt.setQueryTimeout(seconds);
  }

  /**
   * Cancels this <code>Statement</code> object if both the DBMS and
   * driver support aborting an SQL statement.
   * This method can be used by one thread to cancel a statement that
   * is being executed by another thread.
   *
   * @throws java.sql.SQLException                    if a database access error occurs or
   *                                                  this method is called on a closed <code>Statement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method
   */
  @Override
  public void cancel() throws SQLException {
    pstmt.cancel();
  }

  /**
   * Retrieves the first warning reported by calls on this <code>Statement</code> object.
   * Subsequent <code>Statement</code> object warnings will be chained to this
   * <code>SQLWarning</code> object.
   * <p>
   * <p>The warning chain is automatically cleared each time
   * a statement is (re)executed. This method may not be called on a closed
   * <code>Statement</code> object; doing so will cause an <code>SQLException</code>
   * to be thrown.
   * <p>
   * <P><B>Note:</B> If you are processing a <code>ResultSet</code> object, any
   * warnings associated with reads on that <code>ResultSet</code> object
   * will be chained on it rather than on the <code>Statement</code>
   * object that produced it.
   *
   * @return the first <code>SQLWarning</code> object or <code>null</code>
   * if there are no warnings
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   */
  @Override
  public SQLWarning getWarnings() throws SQLException {
    return pstmt.getWarnings();
  }

  /**
   * Clears all the warnings reported on this <code>Statement</code>
   * object. After a call to this method,
   * the method <code>getWarnings</code> will return
   * <code>null</code> until a new warning is reported for this
   * <code>Statement</code> object.
   *
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   */
  @Override
  public void clearWarnings() throws SQLException {
    pstmt.clearWarnings();
  }

  /**
   * Sets the SQL cursor name to the given <code>String</code>, which
   * will be used by subsequent <code>Statement</code> object
   * <code>execute</code> methods. This name can then be
   * used in SQL positioned update or delete statements to identify the
   * current row in the <code>ResultSet</code> object generated by this
   * statement.  If the database does not support positioned update/delete,
   * this method is a noop.  To insure that a cursor has the proper isolation
   * level to support updates, the cursor's <code>SELECT</code> statement
   * should have the form <code>SELECT FOR UPDATE</code>.  If
   * <code>FOR UPDATE</code> is not present, positioned updates may fail.
   * <p>
   * <P><B>Note:</B> By definition, the execution of positioned updates and
   * deletes must be done by a different <code>Statement</code> object than
   * the one that generated the <code>ResultSet</code> object being used for
   * positioning. Also, cursor names must be unique within a connection.
   *
   * @param name the new cursor name, which must be unique within
   *             a connection
   * @throws java.sql.SQLException                    if a database access error occurs or
   *                                                  this method is called on a closed <code>Statement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   */
  @Override
  public void setCursorName(String name) throws SQLException {
    pstmt.setCursorName(name);
  }

  /**
   * Executes the given SQL statement, which may return multiple results.
   * In some (uncommon) situations, a single SQL statement may return
   * multiple result sets and/or update counts.  Normally you can ignore
   * this unless you are (1) executing a stored procedure that you know may
   * return multiple results or (2) you are dynamically executing an
   * unknown SQL string.
   * <p>
   * The <code>execute</code> method executes an SQL statement and indicates the
   * form of the first result.  You must then use the methods
   * <code>getResultSet</code> or <code>getUpdateCount</code>
   * to retrieve the result, and <code>getMoreResults</code> to
   * move to any subsequent result(s).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql any SQL statement
   * @return <code>true</code> if the first result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there are
   * no results
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code>,
   *                                      the method is called on a
   *                                      <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @see #getResultSet
   * @see #getUpdateCount
   * @see #getMoreResults
   */
  @Override
  public boolean execute(String sql) throws SQLException {
    return pstmt.execute(sql);
  }

  /**
   * Retrieves the current result as a <code>ResultSet</code> object.
   * This method should be called only once per result.
   *
   * @return the current result as a <code>ResultSet</code> object or
   * <code>null</code> if the result is an update count or there are no more results
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #execute
   */
  @Override
  public ResultSet getResultSet() throws SQLException {
    return pstmt.getResultSet();
  }

  /**
   * Retrieves the current result as an update count;
   * if the result is a <code>ResultSet</code> object or there are no more results, -1
   * is returned. This method should be called only once per result.
   *
   * @return the current result as an update count; -1 if the current result is a
   * <code>ResultSet</code> object or there are no more results
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #execute
   */
  @Override
  public int getUpdateCount() throws SQLException {
    return pstmt.getUpdateCount();
  }

  /**
   * Moves to this <code>Statement</code> object's next result, returns
   * <code>true</code> if it is a <code>ResultSet</code> object, and
   * implicitly closes any current <code>ResultSet</code>
   * object(s) obtained with the method <code>getResultSet</code>.
   * <p>
   * <P>There are no more results when the following is true:
   * <PRE>{@code
   * // stmt is a Statement object
   * ((stmt.getMoreResults() == false) && (stmt.getUpdateCount() == -1))
   * }</PRE>
   *
   * @return <code>true</code> if the next result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there are
   * no more results
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #execute
   */
  @Override
  public boolean getMoreResults() throws SQLException {
    return pstmt.getMoreResults();
  }

  /**
   * Retrieves the direction for fetching rows from
   * database tables that is the default for result sets
   * generated from this <code>Statement</code> object.
   * If this <code>Statement</code> object has not set
   * a fetch direction by calling the method <code>setFetchDirection</code>,
   * the return value is implementation-specific.
   *
   * @return the default fetch direction for result sets generated
   * from this <code>Statement</code> object
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setFetchDirection
   * @since 1.2
   */
  @Override
  public int getFetchDirection() throws SQLException {
    return pstmt.getFetchDirection();
  }

  /**
   * Gives the driver a hint as to the direction in which
   * rows will be processed in <code>ResultSet</code>
   * objects created using this <code>Statement</code> object.  The
   * default value is <code>ResultSet.FETCH_FORWARD</code>.
   * <p>
   * Note that this method sets the default fetch direction for
   * result sets generated by this <code>Statement</code> object.
   * Each result set has its own methods for getting and setting
   * its own fetch direction.
   *
   * @param direction the initial direction for processing rows
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>
   *                               or the given direction
   *                               is not one of <code>ResultSet.FETCH_FORWARD</code>,
   *                               <code>ResultSet.FETCH_REVERSE</code>, or <code>ResultSet.FETCH_UNKNOWN</code>
   * @see #getFetchDirection
   * @since 1.2
   */
  @Override
  public void setFetchDirection(int direction) throws SQLException {
    pstmt.setFetchDirection(direction);
  }

  /**
   * Retrieves the number of result set rows that is the default
   * fetch size for <code>ResultSet</code> objects
   * generated from this <code>Statement</code> object.
   * If this <code>Statement</code> object has not set
   * a fetch size by calling the method <code>setFetchSize</code>,
   * the return value is implementation-specific.
   *
   * @return the default fetch size for result sets generated
   * from this <code>Statement</code> object
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setFetchSize
   * @since 1.2
   */
  @Override
  public int getFetchSize() throws SQLException {
    return pstmt.getFetchSize();
  }

  /**
   * Gives the JDBC driver a hint as to the number of rows that should
   * be fetched from the database when more rows are needed for
   * <code>ResultSet</code> objects generated by this <code>Statement</code>.
   * If the value specified is zero, then the hint is ignored.
   * The default value is zero.
   *
   * @param rows the number of rows to fetch
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code> or the
   *                               condition {@code rows >= 0} is not satisfied.
   * @see #getFetchSize
   * @since 1.2
   */
  @Override
  public void setFetchSize(int rows) throws SQLException {
    pstmt.setFetchSize(rows);
  }

  /**
   * Retrieves the result set concurrency for <code>ResultSet</code> objects
   * generated by this <code>Statement</code> object.
   *
   * @return either <code>ResultSet.CONCUR_READ_ONLY</code> or
   * <code>ResultSet.CONCUR_UPDATABLE</code>
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @since 1.2
   */
  @Override
  public int getResultSetConcurrency() throws SQLException {
    return pstmt.getResultSetConcurrency();
  }

  /**
   * Retrieves the result set type for <code>ResultSet</code> objects
   * generated by this <code>Statement</code> object.
   *
   * @return one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
   * <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or
   * <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @since 1.2
   */
  @Override
  public int getResultSetType() throws SQLException {
    return pstmt.getResultSetType();
  }

  /**
   * Adds the given SQL command to the current list of commands for this
   * <code>Statement</code> object. The commands in this list can be
   * executed as a batch by calling the method <code>executeBatch</code>.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql typically this is a SQL <code>INSERT</code> or
   *            <code>UPDATE</code> statement
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>, the
   *                               driver does not support batch updates, the method is called on a
   *                               <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @see #executeBatch
   * @see java.sql.DatabaseMetaData#supportsBatchUpdates
   * @since 1.2
   */
  @Override
  public void addBatch(String sql) throws SQLException {
    pstmt.addBatch(sql);
  }

  /**
   * Empties this <code>Statement</code> object's current list of
   * SQL commands.
   * <p>
   *
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code> or the
   *                               driver does not support batch updates
   * @see #addBatch
   * @see java.sql.DatabaseMetaData#supportsBatchUpdates
   * @since 1.2
   */
  @Override
  public void clearBatch() throws SQLException {
    pstmt.clearBatch();
  }

  /**
   * Submits a batch of commands to the database for execution and
   * if all commands execute successfully, returns an array of update counts.
   * The <code>int</code> elements of the array that is returned are ordered
   * to correspond to the commands in the batch, which are ordered
   * according to the order in which they were added to the batch.
   * The elements in the array returned by the method <code>executeBatch</code>
   * may be one of the following:
   * <OL>
   * <LI>A number greater than or equal to zero -- indicates that the
   * command was processed successfully and is an update count giving the
   * number of rows in the database that were affected by the command's
   * execution
   * <LI>A value of <code>SUCCESS_NO_INFO</code> -- indicates that the command was
   * processed successfully but that the number of rows affected is
   * unknown
   * <p>
   * If one of the commands in a batch update fails to execute properly,
   * this method throws a <code>BatchUpdateException</code>, and a JDBC
   * driver may or may not continue to process the remaining commands in
   * the batch.  However, the driver's behavior must be consistent with a
   * particular DBMS, either always continuing to process commands or never
   * continuing to process commands.  If the driver continues processing
   * after a failure, the array returned by the method
   * <code>BatchUpdateException.getUpdateCounts</code>
   * will contain as many elements as there are commands in the batch, and
   * at least one of the elements will be the following:
   * <p>
   * <LI>A value of <code>EXECUTE_FAILED</code> -- indicates that the command failed
   * to execute successfully and occurs only if a driver continues to
   * process commands after a command fails
   * </OL>
   * <p>
   * The possible implementations and return values have been modified in
   * the Java 2 SDK, Standard Edition, version 1.3 to
   * accommodate the option of continuing to process commands in a batch
   * update after a <code>BatchUpdateException</code> object has been thrown.
   *
   * @return an array of update counts containing one element for each
   * command in the batch.  The elements of the array are ordered according
   * to the order in which commands were added to the batch.
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code> or the
   *                                      driver does not support batch statements. Throws {@link java.sql.BatchUpdateException}
   *                                      (a subclass of <code>SQLException</code>) if one of the commands sent to the
   *                                      database fails to execute properly or attempts to return a result set.
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @see #addBatch
   * @see java.sql.DatabaseMetaData#supportsBatchUpdates
   * @since 1.2
   */
  @Override
  public int[] executeBatch() throws SQLException {
    return pstmt.executeBatch();
  }

  /**
   * Retrieves the <code>Connection</code> object
   * that produced this <code>Statement</code> object.
   *
   * @return the connection that produced this statement
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @since 1.2
   */
  @Override
  public Connection getConnection() throws SQLException {
    return pstmt.getConnection();
  }

  /**
   * Moves to this <code>Statement</code> object's next result, deals with
   * any current <code>ResultSet</code> object(s) according  to the instructions
   * specified by the given flag, and returns
   * <code>true</code> if the next result is a <code>ResultSet</code> object.
   * <p>
   * <P>There are no more results when the following is true:
   * <PRE>{@code
   * // stmt is a Statement object
   * ((stmt.getMoreResults(current) == false) && (stmt.getUpdateCount() == -1))
   * }</PRE>
   *
   * @param current one of the following <code>Statement</code>
   *                constants indicating what should happen to current
   *                <code>ResultSet</code> objects obtained using the method
   *                <code>getResultSet</code>:
   *                <code>Statement.CLOSE_CURRENT_RESULT</code>,
   *                <code>Statement.KEEP_CURRENT_RESULT</code>, or
   *                <code>Statement.CLOSE_ALL_RESULTS</code>
   * @return <code>true</code> if the next result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there are no
   * more results
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code> or the argument
   *                                                  supplied is not one of the following:
   *                                                  <code>Statement.CLOSE_CURRENT_RESULT</code>,
   *                                                  <code>Statement.KEEP_CURRENT_RESULT</code> or
   *                                                  <code>Statement.CLOSE_ALL_RESULTS</code>
   * @throws java.sql.SQLFeatureNotSupportedException if
   *                                                  <code>DatabaseMetaData.supportsMultipleOpenResults</code> returns
   *                                                  <code>false</code> and either
   *                                                  <code>Statement.KEEP_CURRENT_RESULT</code> or
   *                                                  <code>Statement.CLOSE_ALL_RESULTS</code> are supplied as
   *                                                  the argument.
   * @see #execute
   * @since 1.4
   */
  @Override
  public boolean getMoreResults(int current) throws SQLException {
    return pstmt.getMoreResults(current);
  }

  /**
   * Retrieves any auto-generated keys created as a result of executing this
   * <code>Statement</code> object. If this <code>Statement</code> object did
   * not generate any keys, an empty <code>ResultSet</code>
   * object is returned.
   * <p>
   * <p><B>Note:</B>If the columns which represent the auto-generated keys were not specified,
   * the JDBC driver implementation will determine the columns which best represent the auto-generated keys.
   *
   * @return a <code>ResultSet</code> object containing the auto-generated key(s)
   * generated by the execution of this <code>Statement</code> object
   * @throws java.sql.SQLException                    if a database access error occurs or
   *                                                  this method is called on a closed <code>Statement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @since 1.4
   */
  @Override
  public ResultSet getGeneratedKeys() throws SQLException {
    return pstmt.getGeneratedKeys();
  }

  /**
   * Executes the given SQL statement and signals the driver with the
   * given flag about whether the
   * auto-generated keys produced by this <code>Statement</code> object
   * should be made available for retrieval.  The driver will ignore the
   * flag if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql               an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   *                          <code>DELETE</code>; or an SQL statement that returns nothing,
   *                          such as a DDL statement.
   * @param autoGeneratedKeys a flag indicating whether auto-generated keys
   *                          should be made available for retrieval;
   *                          one of the following constants:
   *                          <code>Statement.RETURN_GENERATED_KEYS</code>
   *                          <code>Statement.NO_GENERATED_KEYS</code>
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the given
   *                                                  SQL statement returns a <code>ResultSet</code> object,
   *                                                  the given constant is not one of those allowed, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method with a constant of Statement.RETURN_GENERATED_KEYS
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.4
   */
  @Override
  public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    return pstmt.executeUpdate(sql, autoGeneratedKeys);
  }

  /**
   * Executes the given SQL statement and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval.   This array contains the indexes of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available. The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql           an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   *                      <code>DELETE</code>; or an SQL statement that returns nothing,
   *                      such as a DDL statement.
   * @param columnIndexes an array of column indexes indicating the columns
   *                      that should be returned from the inserted row
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the SQL
   *                                                  statement returns a <code>ResultSet</code> object,the second argument
   *                                                  supplied to this method is not an
   *                                                  <code>int</code> array whose elements are valid column indexes, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.4
   */
  @Override
  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
    return pstmt.executeUpdate(sql, columnIndexes);
  }

  /**
   * Executes the given SQL statement and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval.   This array contains the names of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available. The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql         an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
   *                    <code>DELETE</code>; or an SQL statement that returns nothing,
   *                    such as a DDL statement.
   * @param columnNames an array of the names of the columns that should be
   *                    returned from the inserted row
   * @return either the row count for <code>INSERT</code>, <code>UPDATE</code>,
   * or <code>DELETE</code> statements, or 0 for SQL statements
   * that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the SQL
   *                                                  statement returns a <code>ResultSet</code> object, the
   *                                                  second argument supplied to this method is not a <code>String</code> array
   *                                                  whose elements are valid column names, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.4
   */
  @Override
  public int executeUpdate(String sql, String[] columnNames) throws SQLException {
    return pstmt.executeUpdate(sql, columnNames);
  }

  /**
   * Executes the given SQL statement, which may return multiple results,
   * and signals the driver that any
   * auto-generated keys should be made available
   * for retrieval.  The driver will ignore this signal if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * In some (uncommon) situations, a single SQL statement may return
   * multiple result sets and/or update counts.  Normally you can ignore
   * this unless you are (1) executing a stored procedure that you know may
   * return multiple results or (2) you are dynamically executing an
   * unknown SQL string.
   * <p>
   * The <code>execute</code> method executes an SQL statement and indicates the
   * form of the first result.  You must then use the methods
   * <code>getResultSet</code> or <code>getUpdateCount</code>
   * to retrieve the result, and <code>getMoreResults</code> to
   * move to any subsequent result(s).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql               any SQL statement
   * @param autoGeneratedKeys a constant indicating whether auto-generated
   *                          keys should be made available for retrieval using the method
   *                          <code>getGeneratedKeys</code>; one of the following constants:
   *                          <code>Statement.RETURN_GENERATED_KEYS</code> or
   *                          <code>Statement.NO_GENERATED_KEYS</code>
   * @return <code>true</code> if the first result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there are
   * no results
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the second
   *                                                  parameter supplied to this method is not
   *                                                  <code>Statement.RETURN_GENERATED_KEYS</code> or
   *                                                  <code>Statement.NO_GENERATED_KEYS</code>,
   *                                                  the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method with a constant of Statement.RETURN_GENERATED_KEYS
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @see #getResultSet
   * @see #getUpdateCount
   * @see #getMoreResults
   * @see #getGeneratedKeys
   * @since 1.4
   */
  @Override
  public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
    return pstmt.execute(sql, autoGeneratedKeys);
  }

  /**
   * Executes the given SQL statement, which may return multiple results,
   * and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval.  This array contains the indexes of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available.  The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * Under some (uncommon) situations, a single SQL statement may return
   * multiple result sets and/or update counts.  Normally you can ignore
   * this unless you are (1) executing a stored procedure that you know may
   * return multiple results or (2) you are dynamically executing an
   * unknown SQL string.
   * <p>
   * The <code>execute</code> method executes an SQL statement and indicates the
   * form of the first result.  You must then use the methods
   * <code>getResultSet</code> or <code>getUpdateCount</code>
   * to retrieve the result, and <code>getMoreResults</code> to
   * move to any subsequent result(s).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql           any SQL statement
   * @param columnIndexes an array of the indexes of the columns in the
   *                      inserted row that should be  made available for retrieval by a
   *                      call to the method <code>getGeneratedKeys</code>
   * @return <code>true</code> if the first result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there
   * are no results
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the
   *                                                  elements in the <code>int</code> array passed to this method
   *                                                  are not valid column indexes, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @see #getResultSet
   * @see #getUpdateCount
   * @see #getMoreResults
   * @since 1.4
   */
  @Override
  public boolean execute(String sql, int[] columnIndexes) throws SQLException {
    return pstmt.execute(sql, columnIndexes);
  }

  /**
   * Executes the given SQL statement, which may return multiple results,
   * and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval. This array contains the names of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available.  The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * In some (uncommon) situations, a single SQL statement may return
   * multiple result sets and/or update counts.  Normally you can ignore
   * this unless you are (1) executing a stored procedure that you know may
   * return multiple results or (2) you are dynamically executing an
   * unknown SQL string.
   * <p>
   * The <code>execute</code> method executes an SQL statement and indicates the
   * form of the first result.  You must then use the methods
   * <code>getResultSet</code> or <code>getUpdateCount</code>
   * to retrieve the result, and <code>getMoreResults</code> to
   * move to any subsequent result(s).
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   *
   * @param sql         any SQL statement
   * @param columnNames an array of the names of the columns in the inserted
   *                    row that should be made available for retrieval by a call to the
   *                    method <code>getGeneratedKeys</code>
   * @return <code>true</code> if the next result is a <code>ResultSet</code>
   * object; <code>false</code> if it is an update count or there
   * are no more results
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>,the
   *                                                  elements of the <code>String</code> array passed to this
   *                                                  method are not valid column names, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @see #getResultSet
   * @see #getUpdateCount
   * @see #getMoreResults
   * @see #getGeneratedKeys
   * @since 1.4
   */
  @Override
  public boolean execute(String sql, String[] columnNames) throws SQLException {
    return pstmt.execute(sql, columnNames);
  }

  /**
   * Retrieves the result set holdability for <code>ResultSet</code> objects
   * generated by this <code>Statement</code> object.
   *
   * @return either <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> or
   * <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code>
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @since 1.4
   */
  @Override
  public int getResultSetHoldability() throws SQLException {
    return pstmt.getResultSetHoldability();
  }

  /**
   * Retrieves whether this <code>Statement</code> object has been closed. A <code>Statement</code> is closed if the
   * method close has been called on it, or if it is automatically closed.
   *
   * @return true if this <code>Statement</code> object is closed; false if it is still open
   * @throws java.sql.SQLException if a database access error occurs
   * @since 1.6
   */
  @Override
  public boolean isClosed() throws SQLException {
    return pstmt.isClosed();
  }

  /**
   * Returns a  value indicating whether the <code>Statement</code>
   * is poolable or not.
   * <p>
   *
   * @return <code>true</code> if the <code>Statement</code>
   * is poolable; <code>false</code> otherwise
   * <p>
   * @throws java.sql.SQLException if this method is called on a closed
   *                               <code>Statement</code>
   *                               <p>
   * @see java.sql.Statement#setPoolable(boolean) setPoolable(boolean)
   * @since 1.6
   * <p>
   */
  @Override
  public boolean isPoolable() throws SQLException {
    return pstmt.isPoolable();
  }

  /**
   * Requests that a <code>Statement</code> be pooled or not pooled.  The value
   * specified is a hint to the statement pool implementation indicating
   * whether the application wants the statement to be pooled.  It is up to
   * the statement pool manager as to whether the hint is used.
   * <p>
   * The poolable value of a statement is applicable to both internal
   * statement caches implemented by the driver and external statement caches
   * implemented by application servers and other applications.
   * <p>
   * By default, a <code>Statement</code> is not poolable when created, and
   * a <code>PreparedStatement</code> and <code>CallableStatement</code>
   * are poolable when created.
   * <p>
   *
   * @param poolable requests that the statement be pooled if true and
   *                 that the statement not be pooled if false
   *                 <p>
   * @throws java.sql.SQLException if this method is called on a closed
   *                               <code>Statement</code>
   *                               <p>
   * @since 1.6
   */
  @Override
  public void setPoolable(boolean poolable) throws SQLException {
    pstmt.setPoolable(poolable);
  }

  /**
   * Specifies that this {@code Statement} will be closed when all its
   * dependent result sets are closed. If execution of the {@code Statement}
   * does not produce any result sets, this method has no effect.
   * <p>
   * <strong>Note:</strong> Multiple calls to {@code closeOnCompletion} do
   * not toggle the effect on this {@code Statement}. However, a call to
   * {@code closeOnCompletion} does effect both the subsequent execution of
   * statements, and statements that currently have open, dependent,
   * result sets.
   *
   * @throws java.sql.SQLException if this method is called on a closed
   *                               {@code Statement}
   * @since 1.7
   */
  @Override
  public void closeOnCompletion() throws SQLException {
    pstmt.closeOnCompletion();
  }

  /**
   * Returns a value indicating whether this {@code Statement} will be
   * closed when all its dependent result sets are closed.
   *
   * @return {@code true} if the {@code Statement} will be closed when all
   * of its dependent result sets are closed; {@code false} otherwise
   * @throws java.sql.SQLException if this method is called on a closed
   *                               {@code Statement}
   * @since 1.7
   */
  @Override
  public boolean isCloseOnCompletion() throws SQLException {
    return pstmt.isCloseOnCompletion();
  }

  /**
   * Retrieves the current result as an update count; if the result
   * is a <code>ResultSet</code> object or there are no more results, -1
   * is returned. This method should be called only once per result.
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * The default implementation will throw {@code UnsupportedOperationException}
   *
   * @return the current result as an update count; -1 if the current result
   * is a <code>ResultSet</code> object or there are no more results
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #execute
   * @since 1.8
   */
  @Override
  public long getLargeUpdateCount() throws SQLException {
    return pstmt.getLargeUpdateCount();
  }

  /**
   * Retrieves the maximum number of rows that a
   * <code>ResultSet</code> object produced by this
   * <code>Statement</code> object can contain.  If this limit is exceeded,
   * the excess rows are silently dropped.
   * <p>
   * This method should be used when the returned row limit may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * The default implementation will return {@code 0}
   *
   * @return the current maximum number of rows for a <code>ResultSet</code>
   * object produced by this <code>Statement</code> object;
   * zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs or
   *                               this method is called on a closed <code>Statement</code>
   * @see #setMaxRows
   * @since 1.8
   */
  @Override
  public long getLargeMaxRows() throws SQLException {
    return pstmt.getLargeMaxRows();
  }

  /**
   * Sets the limit for the maximum number of rows that any
   * <code>ResultSet</code> object  generated by this <code>Statement</code>
   * object can contain to the given number.
   * If the limit is exceeded, the excess
   * rows are silently dropped.
   * <p>
   * This method should be used when the row limit may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * The default implementation will throw {@code UnsupportedOperationException}
   *
   * @param max the new max rows limit; zero means there is no limit
   * @throws java.sql.SQLException if a database access error occurs,
   *                               this method is called on a closed <code>Statement</code>
   *                               or the condition {@code max >= 0} is not satisfied
   * @see #getMaxRows
   * @since 1.8
   */
  @Override
  public void setLargeMaxRows(long max) throws SQLException {
    pstmt.setLargeMaxRows(max);
  }

  /**
   * Submits a batch of commands to the database for execution and
   * if all commands execute successfully, returns an array of update counts.
   * The <code>long</code> elements of the array that is returned are ordered
   * to correspond to the commands in the batch, which are ordered
   * according to the order in which they were added to the batch.
   * The elements in the array returned by the method {@code executeLargeBatch}
   * may be one of the following:
   * <OL>
   * <LI>A number greater than or equal to zero -- indicates that the
   * command was processed successfully and is an update count giving the
   * number of rows in the database that were affected by the command's
   * execution
   * <LI>A value of <code>SUCCESS_NO_INFO</code> -- indicates that the command was
   * processed successfully but that the number of rows affected is
   * unknown
   * <p>
   * If one of the commands in a batch update fails to execute properly,
   * this method throws a <code>BatchUpdateException</code>, and a JDBC
   * driver may or may not continue to process the remaining commands in
   * the batch.  However, the driver's behavior must be consistent with a
   * particular DBMS, either always continuing to process commands or never
   * continuing to process commands.  If the driver continues processing
   * after a failure, the array returned by the method
   * <code>BatchUpdateException.getLargeUpdateCounts</code>
   * will contain as many elements as there are commands in the batch, and
   * at least one of the elements will be the following:
   * <p>
   * <LI>A value of <code>EXECUTE_FAILED</code> -- indicates that the command failed
   * to execute successfully and occurs only if a driver continues to
   * process commands after a command fails
   * </OL>
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * The default implementation will throw {@code UnsupportedOperationException}
   *
   * @return an array of update counts containing one element for each
   * command in the batch.  The elements of the array are ordered according
   * to the order in which commands were added to the batch.
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code> or the
   *                                      driver does not support batch statements. Throws {@link java.sql.BatchUpdateException}
   *                                      (a subclass of <code>SQLException</code>) if one of the commands sent to the
   *                                      database fails to execute properly or attempts to return a result set.
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @see #addBatch
   * @see java.sql.DatabaseMetaData#supportsBatchUpdates
   * @since 1.8
   */
  @Override
  public long[] executeLargeBatch() throws SQLException {
    return pstmt.executeLargeBatch();
  }

  /**
   * Executes the given SQL statement, which may be an <code>INSERT</code>,
   * <code>UPDATE</code>, or <code>DELETE</code> statement or an
   * SQL statement that returns nothing, such as an SQL DDL statement.
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   * <p>
   * The default implementation will throw {@code UnsupportedOperationException}
   *
   * @param sql an SQL Data Manipulation Language (DML) statement,
   *            such as <code>INSERT</code>, <code>UPDATE</code> or
   *            <code>DELETE</code>; or an SQL statement that returns nothing,
   *            such as a DDL statement.
   * @return either (1) the row count for SQL Data Manipulation Language
   * (DML) statements or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException        if a database access error occurs,
   *                                      this method is called on a closed <code>Statement</code>, the given
   *                                      SQL statement produces a <code>ResultSet</code> object, the method is called on a
   *                                      <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLTimeoutException when the driver has determined that the
   *                                      timeout value that was specified by the {@code setQueryTimeout}
   *                                      method has been exceeded and has at least attempted to cancel
   *                                      the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate(String sql) throws SQLException {
    return pstmt.executeLargeUpdate(sql);
  }

  /**
   * Executes the given SQL statement and signals the driver with the
   * given flag about whether the
   * auto-generated keys produced by this <code>Statement</code> object
   * should be made available for retrieval.  The driver will ignore the
   * flag if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   * <p>
   * The default implementation will throw {@code SQLFeatureNotSupportedException}
   *
   * @param sql               an SQL Data Manipulation Language (DML) statement,
   *                          such as <code>INSERT</code>, <code>UPDATE</code> or
   *                          <code>DELETE</code>; or an SQL statement that returns nothing,
   *                          such as a DDL statement.
   * @param autoGeneratedKeys a flag indicating whether auto-generated keys
   *                          should be made available for retrieval;
   *                          one of the following constants:
   *                          <code>Statement.RETURN_GENERATED_KEYS</code>
   *                          <code>Statement.NO_GENERATED_KEYS</code>
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the given
   *                                                  SQL statement returns a <code>ResultSet</code> object,
   *                                                  the given constant is not one of those allowed, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
   *                                                  this method with a constant of Statement.RETURN_GENERATED_KEYS
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
    return pstmt.executeLargeUpdate(sql, autoGeneratedKeys);
  }

  /**
   * Executes the given SQL statement and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval.   This array contains the indexes of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available. The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   * <p>
   * The default implementation will throw {@code SQLFeatureNotSupportedException}
   *
   * @param sql           an SQL Data Manipulation Language (DML) statement,
   *                      such as <code>INSERT</code>, <code>UPDATE</code> or
   *                      <code>DELETE</code>; or an SQL statement that returns nothing,
   *                      such as a DDL statement.
   * @param columnIndexes an array of column indexes indicating the columns
   *                      that should be returned from the inserted row
   * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
   * or (2) 0 for SQL statements that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the SQL
   *                                                  statement returns a <code>ResultSet</code> object,the second argument
   *                                                  supplied to this method is not an
   *                                                  <code>int</code> array whose elements are valid column indexes, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
    return pstmt.executeLargeUpdate(sql, columnIndexes);
  }

  /**
   * Executes the given SQL statement and signals the driver that the
   * auto-generated keys indicated in the given array should be made available
   * for retrieval.   This array contains the names of the columns in the
   * target table that contain the auto-generated keys that should be made
   * available. The driver will ignore the array if the SQL statement
   * is not an <code>INSERT</code> statement, or an SQL statement able to return
   * auto-generated keys (the list of such statements is vendor-specific).
   * <p>
   * This method should be used when the returned row count may exceed
   * {@link Integer#MAX_VALUE}.
   * <p>
   * <strong>Note:</strong>This method cannot be called on a
   * <code>PreparedStatement</code> or <code>CallableStatement</code>.
   * <p>
   * The default implementation will throw {@code SQLFeatureNotSupportedException}
   *
   * @param sql         an SQL Data Manipulation Language (DML) statement,
   *                    such as <code>INSERT</code>, <code>UPDATE</code> or
   *                    <code>DELETE</code>; or an SQL statement that returns nothing,
   *                    such as a DDL statement.
   * @param columnNames an array of the names of the columns that should be
   *                    returned from the inserted row
   * @return either the row count for <code>INSERT</code>, <code>UPDATE</code>,
   * or <code>DELETE</code> statements, or 0 for SQL statements
   * that return nothing
   * @throws java.sql.SQLException                    if a database access error occurs,
   *                                                  this method is called on a closed <code>Statement</code>, the SQL
   *                                                  statement returns a <code>ResultSet</code> object, the
   *                                                  second argument supplied to this method is not a <code>String</code> array
   *                                                  whose elements are valid column names, the method is called on a
   *                                                  <code>PreparedStatement</code> or <code>CallableStatement</code>
   * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
   * @throws java.sql.SQLTimeoutException             when the driver has determined that the
   *                                                  timeout value that was specified by the {@code setQueryTimeout}
   *                                                  method has been exceeded and has at least attempted to cancel
   *                                                  the currently running {@code Statement}
   * @since 1.8
   */
  @Override
  public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
    return pstmt.executeLargeUpdate(sql, columnNames);
  }

  public String toString() {
    final StringBuilder buffer = new StringBuilder();
    final char[] chars = sql.toCharArray();
    int sqlParamIndex = 1;
    for (char theChar : chars) {
      if (theChar == '?') {
        final Param param = getParamAt(sqlParamIndex);
        buffer.append(param.toSqlValue());
        sqlParamIndex++;
      }
      else {
        buffer.append(theChar);
      }
    }
    return buffer.toString();
  }

  private Param getParamAt(int sqlParamIndex) {
    return sqlIndexToParamMap.getOrDefault(sqlParamIndex, new UnsetParam());
  }

  private void spy(int sqlIndex, Param param) {
    sqlIndexToParamMap.put(sqlIndex, param);
  }


  private static interface Param {
    public String toSqlValue();
  }

  private static class UnsetParam implements Param {

    @Override
    public String toSqlValue() {
      return "?";
    }
  }

  private static class StringParam implements Param {
    private final String value;

    public StringParam(String value) {
      this.value = value;
    }

    @Override
    public String toSqlValue() {
      return "\'" + value + "\'";
    }
  }

  private static class AsciiStreamParam implements Param {

    @Override
    public String toSqlValue() {
      return "<_Ascii_>";
    }
  }

  private static class InputStreamParam implements Param {

    @Override
    public String toSqlValue() {
      return "<_Binary_>";
    }
  }

  private static class ArrayParam implements Param {

    private final Array value;

    public ArrayParam(Array value) {
      this.value = value;
    }

    @Override
    public String toSqlValue() {
      try {
        final Object[] array = (Object[]) value.getArray();
        return '[' + 
                 Arrays.stream(array)
                .map(ObjectParam::new)
                .map(ObjectParam::toSqlValue)
                .collect(Collectors.joining(", ")) + ']';
      }
      catch (SQLException e) {
        e.printStackTrace();
        return "[]";
      }
    }
  }

  private static class BigDecimalParam implements Param {
    private final BigDecimal param;

    public BigDecimalParam(BigDecimal param) {
      this.param = param;
    }

    @Override
    public String toSqlValue() {
      return param.toString();
    }
  }

  private static class BlobParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Blob_>";
    }
  }

  private static class BooleanParam implements Param {
    private final boolean value;

    public BooleanParam(boolean x) {
      value = x;
    }

    @Override
    public String toSqlValue() {
      return value+"";
    }
  }

  private static class ByteParam implements Param {

    private final Byte x;

    public ByteParam(Byte x){
      this.x = x;
    }
    @Override
    public String toSqlValue() {
      return "0x"+Integer.toHexString(x);
    }
  }

  private static class ByteArrayParam implements Param {
    private final Byte[] x;

    public ByteArrayParam(byte[] x) {
      this.x = new Byte[x.length];
      for (int i = 0; i < x.length; i++) {
        this.x[i] = x[i];
      }
    }

    @Override
    public String toSqlValue() {
      return '['+
          Arrays.stream(x)
          .map(ByteParam::new)
          .map(ByteParam::toSqlValue)
          .collect(Collectors.joining(", ")) +']';
    }
  }

  private static class CharstreamParam implements Param {

    @Override
    public String toSqlValue() {
      return "<_Char_>";
    }
  }
  
  private static class ClobParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Clob_>";
    }
  }

  private static class DateParam implements Param {
    private final java.util.Date x;

    public DateParam(java.util.Date x) {
      this.x = x;
    }

    @Override
    public String toSqlValue() {
      return String.format("'%1$tY-%1$tm-%1$te %1$tH:%1tM:%1$tS'", x);
    }
  }

  private static class DoubleParam implements Param {
    private final double x;

    public DoubleParam(double x) {
      this.x = x;
    }

    @Override
    public String toSqlValue() {
      return String.format("%1$.5f", x);
    }
  }

  private static class LongParam implements Param {
    private final long param;

    public LongParam(long param) {
      this.param = param;
    }

    @Override
    public String toSqlValue() {
      return String.format("%1$d", param);
    }
  }

  private static class NCharstreamParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Nchar_>";
    }
  }

  private static class NClobParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Nclob_>";
    }
  }

  private static class NullParam implements Param {
    @Override
    public String toSqlValue() {
      return "NULL";
    }
  }

  private static class RefParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Ref_>";
    }
  }

  private static class RowIdParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_RowId_>";
    }
  }

  private static class SQLXMLParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_SQLXML_>";
    }
  }

  private static class URLParam implements Param {
    private final URL x;

    public URLParam(URL x) {
      this.x = x;
    }

    @Override
    public String toSqlValue() {
      return '\''+x.toString()+'\'';
    }
  }

  private static class UnicodeParam implements Param {
    @Override
    public String toSqlValue() {
      return "<_Unicode_>";
    }
  }

  private static class ObjectParam implements Param {
    private final Object param;

    public ObjectParam(Object param) {
      this.param = param;
    }

    @Override
    public String toSqlValue() {
      return '\''+param.toString()+'\'';
    }
  }
}
