/*
 * Decompiled with CFR 0.152.
 */
package com.github.jcustenborder.vertica;

import com.github.jcustenborder.vertica.Constants;
import com.github.jcustenborder.vertica.VerticaColumnType;
import com.github.jcustenborder.vertica.binary.Encoder;
import com.github.jcustenborder.vertica.binary.Encoders;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.sql.Date;
import java.util.Calendar;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VerticaColumnInfo {
    private static final Logger log = LoggerFactory.getLogger(VerticaColumnInfo.class);
    final String name;
    final VerticaColumnType type;
    final int size;
    final int precision;
    final int scale;
    final Calendar calendar;
    final Encoders encoders = new Encoders();

    public String name() {
        return this.name;
    }

    public VerticaColumnType type() {
        return this.type;
    }

    public int size() {
        return this.size;
    }

    public int precision() {
        return this.precision;
    }

    public int scale() {
        return this.scale;
    }

    VerticaColumnInfo(String name, VerticaColumnType type, int size, int precision, int scale) {
        Preconditions.checkNotNull((Object)name, (Object)"name cannot be null.");
        this.name = name;
        this.type = type;
        if (VerticaColumnType.NUMERIC == type) {
            Preconditions.checkState((precision > 0 ? 1 : 0) != 0, (Object)"precision must be greater than zero.");
            Preconditions.checkState((scale > -1 ? 1 : 0) != 0, (Object)"scale must be greater than -1.");
            this.size = VerticaColumnInfo.numericSize(precision);
        } else {
            this.size = size;
        }
        this.precision = precision;
        this.scale = scale;
        this.calendar = Calendar.getInstance(Constants.UTC_TIMEZONE);
    }

    VerticaColumnInfo(String name, VerticaColumnType type) {
        this(name, type, VerticaColumnInfo.sizeForType(type), -1, -1);
    }

    VerticaColumnInfo(String name, VerticaColumnType type, int size) {
        this(name, type, size, -1, -1);
    }

    VerticaColumnInfo(String name, VerticaColumnType type, int precision, int scale) {
        this(name, type, VerticaColumnInfo.numericSize(precision), precision, scale);
    }

    static final int sizeForType(VerticaColumnType type) {
        int size;
        switch (type) {
            case BOOLEAN: {
                size = 1;
                break;
            }
            case FLOAT: {
                size = 8;
                break;
            }
            case DATE: {
                size = 8;
                break;
            }
            case TIME: {
                size = 8;
                break;
            }
            case TIMETZ: {
                size = 8;
                break;
            }
            case TIMESTAMP: {
                size = 8;
                break;
            }
            case TIMESTAMPTZ: {
                size = 8;
                break;
            }
            case INTERVAL: {
                size = 8;
                break;
            }
            case VARBINARY: 
            case VARCHAR: {
                size = -1;
                break;
            }
            default: {
                throw new IllegalStateException(String.format("Size must be specified for type '%s'", new Object[]{type}));
            }
        }
        return size;
    }

    void writeFloat(ByteBuffer buffer, Object value) {
        log.trace("writeFloat() - value = {}", value);
        Number number = (Number)value;
        buffer.putDouble(number.doubleValue());
    }

    void writeBoolean(ByteBuffer buffer, Object value) {
        log.trace("writeBoolean() - value = {}", value);
        boolean bool = (Boolean)value;
        buffer.put(bool ? (byte)1 : 0);
    }

    void writeChar(ByteBuffer buffer, Object value) {
        log.trace("writeChar() - value = {}", value);
        ByteBuffer valueBuffer = Charsets.UTF_8.encode(value.toString());
        Preconditions.checkState((this.size >= valueBuffer.remaining() ? 1 : 0) != 0, (String)"Encoded value for '%s' is %s byte(s) but the column is only %s byte(s).", (Object[])new Object[]{this.name, valueBuffer.remaining(), this.size});
        buffer.put(valueBuffer);
        int padding = this.size - valueBuffer.capacity();
        log.trace("writeChar() - padding value by {} byte(s).");
        for (int i = 0; i < padding; ++i) {
            buffer.put((byte)0);
        }
    }

    void writeBinary(ByteBuffer buffer, Object value) {
        log.trace("writeBinary() - value = {}", value);
        byte[] valueBuffer = (byte[])value;
        Preconditions.checkState((this.size >= valueBuffer.length ? 1 : 0) != 0, (String)"Encoded value for '%s' is %s byte(s) but the column is only %s byte(s).", (Object[])new Object[]{this.name, valueBuffer.length, this.size});
        buffer.put(valueBuffer);
        int padding = this.size - valueBuffer.length;
        log.trace("writeBinary() - padding value by {} byte(s).");
        for (int i = 0; i < padding; ++i) {
            buffer.put((byte)0);
        }
    }

    void writeVarchar(ByteBuffer buffer, Object value) {
        log.trace("writeVarchar() - value = {}", value);
        ByteBuffer valueBuffer = Charsets.UTF_8.encode(value.toString());
        log.trace("writeVarchar() - writing {} byte(s).", (Object)valueBuffer.remaining());
        buffer.putInt(valueBuffer.remaining());
        buffer.put(valueBuffer);
    }

    void writeVarbinary(ByteBuffer buffer, Object value) {
        log.trace("writeVarbinary() - value = {}", value);
        byte[] valueBuffer = (byte[])value;
        log.trace("writeVarbinary() - writing {} byte(s).", (Object)valueBuffer.length);
        buffer.putInt(valueBuffer.length);
        buffer.put(valueBuffer);
    }

    <T> T checkedCast(Object value, Class<T> cls) {
        try {
            return cls.cast(value);
        }
        catch (ClassCastException ex) {
            throw new IllegalStateException(String.format("Could not cast '%s' to '%s' for column '%s'.", value.getClass().getName(), cls.getName(), this.name), ex);
        }
    }

    void writeInteger(ByteBuffer buffer, Object value) {
        log.trace("writeInteger() - value = {}", value);
        Number number = (Number)value;
        switch (this.size) {
            case 1: {
                buffer.put(number.byteValue());
                break;
            }
            case 2: {
                buffer.putShort(number.shortValue());
                break;
            }
            case 4: {
                buffer.putInt(number.intValue());
                break;
            }
            case 8: {
                buffer.putLong(number.longValue());
                break;
            }
            default: {
                throw new UnsupportedOperationException(String.format("An integer of %s bytes is not supported", this.size));
            }
        }
    }

    void writeDate(ByteBuffer buffer, Object value) {
        log.trace("writeDate() - value = {}", value);
        long input = this.toDateStorage(value);
        long storage = (input - 946684800000L) / 86400000L;
        log.trace("writeDate() - storage = {}", (Object)storage);
        buffer.putLong(storage);
    }

    void writeTimestamp(ByteBuffer buffer, Object value) {
        log.trace("writeTimestamp() - value = {}", value);
        long input = this.toDateStorage(value);
        long storage = input * 1000L - 946684800000000L;
        log.trace("writeTimestamp() - storage = {}", (Object)storage);
        buffer.putLong(storage);
    }

    void writeTimestampTZ(ByteBuffer buffer, Object value) {
        log.trace("writeTimestampTZ() - value = {}", value);
        long input = this.toDateStorage(value);
        long storage = input * 1000L - 946684800000000L;
        log.trace("writeTimestampTZ() - storage = {}", (Object)storage);
        buffer.putLong(storage);
    }

    void writeTime(ByteBuffer buffer, Object value) {
        log.trace("writeTime() - value = {}", value);
        long input = this.toDateStorage(value);
        this.calendar.setTimeInMillis(input);
        this.calendar.set(2000, 0, 1);
        long storage = this.calendar.getTimeInMillis() * 1000L - 946684800000000L;
        log.trace("writeTime() - storage = {}", (Object)storage);
        buffer.putLong(storage);
    }

    void writeTimeTZ(ByteBuffer buffer, Object value) {
        buffer.put(BaseEncoding.base16().decode((CharSequence)"D0970180F079F010"));
    }

    static int numericSize(int precision) {
        return (int)Math.ceil(((double)precision / 19.0 + 1.0) * 8.0);
    }

    void writeNumeric(ByteBuffer buffer, Object value) {
        int i;
        log.trace("writeNumeric() - value = {}", value);
        BigDecimal decimal = (BigDecimal)value;
        Preconditions.checkState((this.scale == decimal.scale() ? 1 : 0) != 0, (String)"Scale for '%s' is mismatched. Value(%s) does not match definition of %s.", (Object[])new Object[]{decimal.scale(), this.scale});
        BigInteger unscaled = decimal.unscaledValue();
        byte[] unscaledBuffer = unscaled.toByteArray();
        log.trace("writeNumeric() - bufferSize:{}", (Object)this.size);
        ByteBuffer byteBuffer = ByteBuffer.allocate(this.size).order(ByteOrder.LITTLE_ENDIAN);
        int bufferMinusScale = this.size - 5;
        int paddingNeeded = bufferMinusScale - unscaledBuffer.length;
        log.trace("writeNumeric() - Padding with {} byte(s).", (Object)paddingNeeded);
        for (i = 0; i < paddingNeeded; ++i) {
            byteBuffer.put((byte)0);
        }
        for (i = unscaledBuffer.length - 1; i >= 0; --i) {
            byteBuffer.put(unscaledBuffer[i]);
        }
        byteBuffer.put((byte)0);
        byteBuffer.putInt(this.scale);
        byteBuffer.flip();
        buffer.put(byteBuffer);
    }

    private long toDateStorage(Object value) {
        long input;
        if (value instanceof java.util.Date) {
            input = ((java.util.Date)value).getTime();
        } else if (value instanceof Date) {
            input = ((Date)value).getTime();
        } else {
            throw new UnsupportedOperationException(String.format("Type '%s' is not supported.", value.getClass().getName()));
        }
        return input;
    }

    void writeInterval(ByteBuffer buffer, Object value) {
        Number number = (Number)value;
        buffer.putLong(number.longValue());
    }

    void encode(ByteBuffer buffer, Object value) {
        Preconditions.checkNotNull((Object)buffer, (Object)"buffer cannot be null.");
        Preconditions.checkState((ByteOrder.LITTLE_ENDIAN == buffer.order() ? 1 : 0) != 0, (Object)"buffer.order() must be LITTLE_ENDIAN.");
        if (null == value) {
            log.trace("encode() - Skipping due to null value.");
            return;
        }
        Encoder encoder = this.encoders.get(this.type, value);
        if (null == encoder) {
            throw new UnsupportedOperationException(String.format("Encoder for %s:%s was found", new Object[]{this.type, value.getClass().getName()}));
        }
        encoder.encode(buffer, value, this.name, this.size, this.scale);
    }
}

