
/*
 * //  Copyright (c) 2015 Couchbase, Inc.
 * //  Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * //  except in compliance with the License. You may obtain a copy of the License at
 * //    http://www.apache.org/licenses/LICENSE-2.0
 * //  Unless required by applicable law or agreed to in writing, software distributed under the
 * //  License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * //  either express or implied. See the License for the specific language governing permissions
 * //  and limitations under the License.
 */

package com.couchbase;

import com.couchbase.jdbc.CBParameterMetaData;
import com.couchbase.jdbc.CBPreparedResult;
import com.couchbase.jdbc.Protocol;
import com.couchbase.jdbc.core.CouchResponse;
import com.couchbase.jdbc.util.SqlParser;
import com.couchbase.jdbc.util.TimestampUtils;
import com.couchbase.json.SQLJSON;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.net.URL;
import java.sql.*;
import java.sql.Date;
import java.util.*;

/**
 * Created by davec on 2015-02-20.
 */
public class CBPreparedStatement extends CBStatement implements java.sql.PreparedStatement
{
    private final static Logger logger = LoggerFactory.getLogger(CBPreparedStatement.class);

    private static final String PREPARED="prepared";
    private static final String ARGS="args";

    final String sql;

    final CBPreparedResult preparedStatement;
    final SqlParser parser;

    final Object []fields;
    Map <String,Object> parameters = new HashMap<String,Object>();
    TimestampUtils timestampUtils = new TimestampUtils();
    String []returning = null;


    public CBPreparedStatement(Connection con, Protocol protocol, String sql) throws SQLException
    {
        super(con, protocol);
        parser = new SqlParser(sql);
        parser.parse();
        fields = new Object[parser.getNumFields()];
        this.sql = parser.replaceProcessing(sql, escapeProcessing);

        logger.trace("Prepare statement {}", parser.toString() );

        CouchResponse ret = protocol.prepareStatement(parser.toString(),returning);

        // we have to put the result into $1 for raw results
        preparedStatement = new CBPreparedResult((Map)ret.getFirstResult().get("$1"));
    }

    public CBPreparedStatement(Connection con, Protocol protocol, String sql, String []returning) throws SQLException
    {
        super(con, protocol);
        parser = new SqlParser(sql);
        parser.parse();
        fields = new Object[parser.getNumFields()];
        this.sql = sql;
        this.returning = returning;

        logger.trace("Prepare statement {}", parser.toString() );
        CouchResponse ret = protocol.prepareStatement(parser.toString(), returning);


        // we have to put the result into $1 for raw results
        preparedStatement = new CBPreparedResult((Map)ret.getFirstResult().get("$1"));
    }
    /**
     * Executes the SQL query in this <code>PreparedStatement</code> object
     * and returns the <code>ResultSet</code> object generated by the query.
     *
     * @return a <code>ResultSet</code> object that contains the data produced by the
     * query; never <code>null</code>
     * @throws java.sql.SQLException if a database access error occurs;
     *                               this method is called on a closed  <code>PreparedStatement</code> or the SQL
     *                               statement does not return a <code>ResultSet</code> object
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     */
    @Override
    public ResultSet executeQuery() throws SQLException
    {

        checkClosed();

        // get the query identifier
        parameters.put(PREPARED, preparedStatement.getName());
        // add the encoded plan as well
        parameters.put("encoded_plan", preparedStatement.getEncodedPlan() );

        if (fields!=null && fields.length >0)
        {
            parameters.put(ARGS, fields);
        }

        CouchResponse couchResponse = protocol.doQuery(sql, parameters);

        return new CBResultSet(this, couchResponse);
    }

    /**
     * Executes the SQL statement in this <code>PreparedStatement</code> object,
     * which must be an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
     * <code>DELETE</code>; or an SQL statement that returns nothing,
     * such as a DDL statement.
     *
     * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
     * or (2) 0 for SQL statements that return nothing
     * @throws java.sql.SQLException if a database access error occurs;
     *                               this method is called on a closed  <code>PreparedStatement</code>
     *                               or the SQL statement returns a <code>ResultSet</code> object
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     */
    @Override

    public int executeUpdate() throws SQLException
    {
        checkClosed();
        parameters.clear();

        logger.trace( "Using {}\n Encoded Plan {}", preparedStatement.getName(),preparedStatement.getEncodedPlan());

        parameters.put(PREPARED,  preparedStatement.getName() );
        // add the encoded plan as well
        parameters.put("encoded_plan", preparedStatement.getEncodedPlan() );

        if (fields!=null && fields.length>0)
        {
            parameters.put(ARGS,fields);
        }

        CouchResponse couchResponse = protocol.doQuery(sql, parameters);
        return (int)couchResponse.getMetrics().getMutationCount();
    }

    /**
     * Sets the designated parameter to SQL <code>NULL</code>.
     * 
     * <P><B>Note:</B> You must specify the parameter's SQL type.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param sqlType        the SQL type code defined in <code>java.sql.Types</code>
     * @throws java.sql.SQLException           if parameterIndex does not correspond to a parameter
     *                                         marker in the SQL statement; if a database access error occurs or
     *                                         this method is called on a closed <code>PreparedStatement</code>
     * @throws SQLFeatureNotSupportedException if <code>sqlType</code> is
     *                                         a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
     *                                         <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
     *                                         <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
     *                                         <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
     *                                         or  <code>STRUCT</code> data type and the JDBC driver does not support
     *                                         this data type
     */
    @Override
    public void setNull(int parameterIndex, int sqlType) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);
        fields[parameterIndex-1]=null;
    }

    /**
     * Sets the designated parameter to the given Java <code>boolean</code> value.
     * The driver converts this
     * to an SQL <code>BIT</code> or <code>BOOLEAN</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement;
     *                               if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setBoolean(int parameterIndex, boolean x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);
        fields[parameterIndex-1] = Boolean.toString(x);
    }

    /**
     * Sets the designated parameter to the given Java <code>byte</code> value.
     * The driver converts this
     * to an SQL <code>TINYINT</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setByte(int parameterIndex, byte x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java <code>short</code> value.
     * The driver converts this
     * to an SQL <code>SMALLINT</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setShort(int parameterIndex, short x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java <code>int</code> value.
     * The driver converts this
     * to an SQL <code>INTEGER</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setInt(int parameterIndex, int x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java <code>long</code> value.
     * The driver converts this
     * to an SQL <code>BIGINT</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setLong(int parameterIndex, long x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = Long.toString(x);
    }

    /**
     * Sets the designated parameter to the given Java <code>float</code> value.
     * The driver converts this
     * to an SQL <code>REAL</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setFloat(int parameterIndex, float x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java <code>double</code> value.
     * The driver converts this
     * to an SQL <code>DOUBLE</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setDouble(int parameterIndex, double x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given <code>java.math.BigDecimal</code> value.
     * The driver converts this to an SQL <code>NUMERIC</code> value when
     * it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java <code>String</code> value.
     * The driver converts this
     * to an SQL <code>VARCHAR</code> or <code>LONGVARCHAR</code> value
     * (depending on the argument's
     * size relative to the driver's limits on <code>VARCHAR</code> values)
     * when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setString(int parameterIndex, String x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x;
    }

    /**
     * Sets the designated parameter to the given Java array of bytes.  The driver converts
     * this to an SQL <code>VARBINARY</code> or <code>LONGVARBINARY</code>
     * (depending on the argument's size relative to the driver's limits on
     * <code>VARBINARY</code> values) when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setBytes(int parameterIndex, byte[] x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = new String(x);
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Date</code> value
     * using the default time zone of the virtual machine that is running
     * the application.
     * The driver converts this
     * to an SQL <code>DATE</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setDate(int parameterIndex, Date x) throws SQLException
    {
        setDate(parameterIndex,x, null);

    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Time</code> value.
     * The driver converts this
     * to an SQL <code>TIME</code> value when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setTime(int parameterIndex, Time x) throws SQLException
    {
        setTime(parameterIndex, x, null);
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Timestamp</code> value.
     * The driver
     * converts this to an SQL <code>TIMESTAMP</code> value when it sends it to the
     * database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x.toString();
    }

    /**
     * Sets the designated parameter to the given input stream, which will have
     * the specified number of bytes.
     * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code>. Data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from ASCII to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the Java input stream that contains the ASCII parameter value
     * @param length         the number of bytes in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        setCharacterStream(parameterIndex, x, length, "ASCII");
    }

    /**
     * Sets the designated parameter to the given input stream, which
     * will have the specified number of bytes.
     * 
     * When a very large Unicode value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code> object. The data will be read from the
     * stream as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from Unicode to the database char format.
     * 
     * The byte format of the Unicode stream must be a Java UTF-8, as defined in the
     * Java Virtual Machine Specification.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              a <code>java.io.InputStream</code> object that contains the
     *                       Unicode parameter value
     * @param length         the number of bytes in the stream
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
     *                                                  this method
     * @deprecated
     */
    @Override
    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        setCharacterStream(parameterIndex, x, length, "UTF-8");
    }

    /**
     * Sets the designated parameter to the given input stream, which will have
     * the specified number of bytes.
     * When a very large binary value is input to a <code>LONGVARBINARY</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code> object. The data will be read from the
     * stream as needed until end-of-file is reached.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the java input stream which contains the binary parameter value
     * @param length         the number of bytes in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);
        //todo encode this//
        fields[parameterIndex-1] =  x.toString();
    }

    /**
     * Clears the current parameter values immediately.
     * <P>In general, parameter values remain in force for repeated use of a
     * statement. Setting a parameter value automatically clears its
     * previous value.  However, in some cases it is useful to immediately
     * release the resources used by the current parameter values; this can
     * be done by calling the method <code>clearParameters</code>.
     *
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     */
    @Override
    public void clearParameters() throws SQLException
    {
        checkClosed();
        for (int i = 0;i < fields.length;i++)
        {
            fields[i] = null;
        }
        parameters.clear();
    }

    /**
     * Sets the value of the designated parameter with the given object.
     * This method is like the method <code>setObject</code>
     * above, except that it assumes a scale of zero.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the object containing the input parameter value
     * @param targetSqlType  the SQL type (as defined in java.sql.Types) to be
     *                       sent to the database
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if <code>targetSqlType</code> is
     *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
     *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
     *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
     *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
     *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
     *                                                  this data type
     * @see java.sql.Types
     */
    @Override
    public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = x.toString();
    }
    /**
     * Sets the value of the designated parameter with the given object.
     * This method is like the method <code>setObject</code>
     * above, except that it assumes a scale of zero.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param sqljson        sqlJson to update
     *                       sent to the database
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if <code>targetSqlType</code> is
     *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
     *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
     *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
     *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
     *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
     *                                                  this data type
     * @see java.sql.Types
     */
    public void setSQLJSON(int parameterIndex,SQLJSON sqljson) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        fields[parameterIndex-1] = sqljson.parameterValue();
    }
    /**
     * <p>Sets the value of the designated parameter using the given object.
     * The second parameter must be of type <code>Object</code>; therefore, the
     * <code>java.lang</code> equivalent objects should be used for built-in types.
     * 
     * <p>The JDBC specification specifies a standard mapping from
     * Java <code>Object</code> types to SQL types.  The given argument
     * will be converted to the corresponding SQL type before being
     * sent to the database.
     * 
     * <p>Note that this method may be used to pass datatabase-
     * specific abstract data types, by using a driver-specific Java
     * type.
     * 
     * If the object is of a class implementing the interface <code>SQLData</code>,
     * the JDBC driver should call the method <code>SQLData.writeSQL</code>
     * to write it to the SQL data stream.
     * If, on the other hand, the object is of a class implementing
     * <code>Ref</code>, <code>Blob</code>, <code>Clob</code>,  <code>NClob</code>,
     * <code>Struct</code>, <code>java.net.URL</code>, <code>RowId</code>, <code>SQLXML</code>
     * or <code>Array</code>, the driver should pass it to the database as a
     * value of the corresponding SQL type.
     * 
     * <b>Note:</b> Not all databases allow for a non-typed Null to be sent to
     * the backend. For maximum portability, the <code>setNull</code> or the
     * <code>setObject(int parameterIndex, Object x, int sqlType)</code>
     * method should be used
     * instead of <code>setObject(int parameterIndex, Object x)</code>.
     * 
     * <b>Note:</b> This method throws an exception if there is an ambiguity, for example, if the
     * object is of a class implementing more than one of the interfaces named above.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the object containing the input parameter value
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs;
     *                               this method is called on a closed <code>PreparedStatement</code>
     *                               or the type of the given object is ambiguous
     */
    @Override
    public void setObject(int parameterIndex, Object x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (x == null)
            setNull(parameterIndex, Types.JAVA_OBJECT);
        else if (x instanceof String)
            setString(parameterIndex, (String)x);
        else if (x instanceof BigDecimal)
            setBigDecimal(parameterIndex, (BigDecimal)x);
        else if (x instanceof Short)
            setShort(parameterIndex, ((Short)x).shortValue());
        else if (x instanceof Integer)
            setInt(parameterIndex, ((Integer)x).intValue());
        else if (x instanceof Long)
            setLong(parameterIndex, ((Long)x).longValue());
        else if (x instanceof Float)
            setFloat(parameterIndex, ((Float)x).floatValue());
        else if (x instanceof Double)
            setDouble(parameterIndex, ((Double)x).doubleValue());
        else if (x instanceof byte[])
            setBytes(parameterIndex, (byte[])x);
        else if (x instanceof java.sql.Date)
            setDate(parameterIndex, (java.sql.Date)x);
        else if (x instanceof Time)
            setTime(parameterIndex, (Time)x);
        else if (x instanceof Timestamp)
            setTimestamp(parameterIndex, (Timestamp)x);
        else if (x instanceof Boolean)
            setBoolean(parameterIndex, ((Boolean)x).booleanValue());
        else if (x instanceof Byte)
            setByte(parameterIndex, ((Byte)x).byteValue());
        else if (x instanceof Blob)
            setBlob(parameterIndex, (Blob)x);
        else if (x instanceof Clob)
            setClob(parameterIndex, (Clob)x);
        else if (x instanceof Array)
            setArray(parameterIndex, (Array)x);
        else if (x instanceof SQLJSON)
            setSQLJSON(parameterIndex, (SQLJSON) x);
        else if (x instanceof Character)
            setString(parameterIndex, x.toString());
        else if (x instanceof Map)
            setMap(parameterIndex, (Map)x);
        else
        {
            // Can't infer a type.
            throw new SQLException("Can''t infer the SQL type to use for an instance of " + x + ". Use setObject() with an explicit Types value to specify the type to use.");
        }
    }

    /**
     * Executes the SQL statement in this <code>PreparedStatement</code> object,
     * which may be any kind of SQL statement.
     * Some prepared statements return multiple results; the <code>execute</code>
     * method handles these complex statements as well as the simpler
     * form of statements handled by the methods <code>executeQuery</code>
     * and <code>executeUpdate</code>.
     * 
     * The <code>execute</code> method returns a <code>boolean</code> to
     * indicate the form of the first result.  You must call either the method
     * <code>getResultSet</code> or <code>getUpdateCount</code>
     * to retrieve the result; you must call <code>getMoreResults</code> to
     * move to any subsequent result(s).
     *
     * @return <code>true</code> if the first result is a <code>ResultSet</code>
     * object; <code>false</code> if the first result is an update
     * count or there is no result
     * @throws java.sql.SQLException        if a database access error occurs;
     *                                      this method is called on a closed <code>PreparedStatement</code>
     *                                      or an argument is supplied to this method
     * @throws java.sql.SQLTimeoutException when the driver has determined that the
     *                                      timeout value that was specified by the {@code setQueryTimeout}
     *                                      method has been exceeded and has at least attempted to cancel
     *                                      the currently running {@code Statement}
     * @see java.sql.Statement#execute
     * @see java.sql.Statement#getResultSet
     * @see java.sql.Statement#getUpdateCount
     * @see java.sql.Statement#getMoreResults
     */

    @Override
    public boolean execute() throws SQLException
    {
        parameters.clear();

        parameters.put(PREPARED, preparedStatement.getName());

        // add the encoded plan as well
        parameters.put("encoded_plan", preparedStatement.getEncodedPlan() );

        if (fields!=null && fields.length>0)
        {
            parameters.put(ARGS, fields);
        }

        CouchResponse couchResponse = protocol.doQuery(sql, parameters);

        updateCount = (int)couchResponse.getMetrics().getMutationCount();

        if ((int)couchResponse.getMetrics().getResultCount() > 0 )
        {
            resultSet = new CBResultSet(this, couchResponse);
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * Adds a set of parameters to this <code>PreparedStatement</code>
     * object's batch of commands.
     *
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @see java.sql.Statement#addBatch
     * @since 1.2
     */
    @Override
    public void addBatch() throws SQLException
    {
        checkClosed();
    }

    /**
     * Sets the designated parameter to the given <code>Reader</code>
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.Reader</code> object. The data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param reader         the <code>java.io.Reader</code> object that contains the
     *                       Unicode data
     * @param length         the number of characters in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.2
     */
    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (reader == null )
        {
            setNull(parameterIndex, Types.VARCHAR);
        }
        else
        {
            char[] l_chars = new char[length];
            int l_charsRead = 0;
            try
            {
                while (true)
                {
                    int n = reader.read(l_chars, l_charsRead, length - l_charsRead);
                    if (n == -1)
                        break;

                    l_charsRead += n;

                    if (l_charsRead == length)
                        break;
                }
            }
            catch (IOException l_ioe)
            {
                throw new SQLException("Provided Reader failed.");
            }
            setString(parameterIndex, new String(l_chars, 0, l_charsRead));
        }

    }

    /**
     * Sets the designated parameter to the given
     * <code>REF(&lt;structured-type&gt;)</code> value.
     * The driver converts this to an SQL <code>REF</code> value when it
     * sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              an SQL <code>REF</code> value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.2
     */
    @Override
    public void setRef(int parameterIndex, Ref x) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class,"setRef");
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Blob</code> object.
     * The driver converts this to an SQL <code>BLOB</code> value when it
     * sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              a <code>Blob</code> object that maps an SQL <code>BLOB</code> value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.2
     */
    @Override
    public void setBlob(int parameterIndex, Blob x) throws SQLException
    {
      throw CBDriver.notImplemented(CBPreparedStatement.class,"setBlob");
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Clob</code> object.
     * The driver converts this to an SQL <code>CLOB</code> value when it
     * sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              a <code>Clob</code> object that maps an SQL <code>CLOB</code> value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.2
     */
    @Override
    public void setClob(int parameterIndex, Clob x) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class,"setClob");
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Array</code> object.
     * The driver converts this to an SQL <code>ARRAY</code> value when it
     * sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              an <code>Array</code> object that maps an SQL <code>ARRAY</code> value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.2
     */
    @Override
    public void setArray(int parameterIndex, Array x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (x == null)
        {
            setNull(parameterIndex, Types.ARRAY);
        }
        else
        {
            fields[parameterIndex - 1] = x.getArray();
        }
    }

    /**
     * Retrieves a <code>ResultSetMetaData</code> object that contains
     * information about the columns of the <code>ResultSet</code> object
     * that will be returned when this <code>PreparedStatement</code> object
     * is executed.
     * 
     * Because a <code>PreparedStatement</code> object is precompiled, it is
     * possible to know about the <code>ResultSet</code> object that it will
     * return without having to execute it.  Consequently, it is possible
     * to invoke the method <code>getMetaData</code> on a
     * <code>PreparedStatement</code> object rather than waiting to execute
     * it and then invoking the <code>ResultSet.getMetaData</code> method
     * on the <code>ResultSet</code> object that is returned.
     * 
     * <B>NOTE:</B> Using this method may be expensive for some drivers due
     * to the lack of underlying DBMS support.
     *
     * @return the description of a <code>ResultSet</code> object's columns or
     * <code>null</code> if the driver cannot return a
     * <code>ResultSetMetaData</code> object
     * @throws java.sql.SQLException                    if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support
     *                                                  this method
     * @since 1.2
     */
    @Override
    public ResultSetMetaData getMetaData() throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "getMetaData");
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Date</code> value,
     * using the given <code>Calendar</code> object.  The driver uses
     * the <code>Calendar</code> object to construct an SQL <code>DATE</code> value,
     * which the driver then sends to the database.  With
     * a <code>Calendar</code> object, the driver can calculate the date
     * taking into account a custom timezone.  If no
     * <code>Calendar</code> object is specified, the driver uses the default
     * timezone, which is that of the virtual machine running the application.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @param cal            the <code>Calendar</code> object the driver will use
     *                       to construct the date
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.2
     */
    @Override
    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (x == null)
        {
            setNull(parameterIndex, Types.DATE);
        }
        else
        {
            if (cal != null)
            {
                cal = (Calendar) cal.clone();
            }

            fields[parameterIndex - 1] =timestampUtils.toString(cal, x) ;
        }
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Time</code> value,
     * using the given <code>Calendar</code> object.  The driver uses
     * the <code>Calendar</code> object to construct an SQL <code>TIME</code> value,
     * which the driver then sends to the database.  With
     * a <code>Calendar</code> object, the driver can calculate the time
     * taking into account a custom timezone.  If no
     * <code>Calendar</code> object is specified, the driver uses the default
     * timezone, which is that of the virtual machine running the application.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @param cal            the <code>Calendar</code> object the driver will use
     *                       to construct the time
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.2
     */
    @Override
    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (x == null)
        {
            setNull(parameterIndex, Types.TIME);
            return;
        }
        {
            if (cal != null)
            {
                cal = (Calendar) cal.clone();
            }

            fields[parameterIndex - 1] = timestampUtils.toString(cal, x);
        }
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.Timestamp</code> value,
     * using the given <code>Calendar</code> object.  The driver uses
     * the <code>Calendar</code> object to construct an SQL <code>TIMESTAMP</code> value,
     * which the driver then sends to the database.  With a
     * <code>Calendar</code> object, the driver can calculate the timestamp
     * taking into account a custom timezone.  If no
     * <code>Calendar</code> object is specified, the driver uses the default
     * timezone, which is that of the virtual machine running the application.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @param cal            the <code>Calendar</code> object the driver will use
     *                       to construct the timestamp
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.2
     */
    @Override
    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if (x == null)
        {
            setNull(parameterIndex, Types.TIME);
        }
        else
        {
            if (cal != null)
            {
                cal = (Calendar) cal.clone();
            }

            fields[parameterIndex - 1] = timestampUtils.toString(cal, x);
        }
    }

    /**
     * Sets the designated parameter to SQL <code>NULL</code>.
     * This version of the method <code>setNull</code> should
     * be used for user-defined types and REF type parameters.  Examples
     * of user-defined types include: STRUCT, DISTINCT, JAVA_OBJECT, and
     * named array types.
     * 
     * <P><B>Note:</B> To be portable, applications must give the
     * SQL type code and the fully-qualified SQL type name when specifying
     * a NULL user-defined or REF parameter.  In the case of a user-defined type
     * the name is the type name of the parameter itself.  For a REF
     * parameter, the name is the type name of the referenced type.  If
     * a JDBC driver does not need the type code or type name information,
     * it may ignore it.
     * 
     * Although it is intended for user-defined and Ref parameters,
     * this method may be used to set a null parameter of any JDBC type.
     * If the parameter does not have a user-defined or REF type, the given
     * typeName is ignored.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param sqlType        a value from <code>java.sql.Types</code>
     * @param typeName       the fully-qualified name of an SQL user-defined type;
     *                       ignored if the parameter is not a user-defined type or REF
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if <code>sqlType</code> is
     *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
     *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
     *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
     *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
     *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
     *                                                  this data type or if the JDBC driver does not support this method
     * @since 1.2
     */
    @Override
    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException
    {
        setNull(parameterIndex, sqlType);
    }

    /**
     * Sets the designated parameter to the given <code>java.net.URL</code> value.
     * The driver converts this to an SQL <code>DATALINK</code> value
     * when it sends it to the database.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the <code>java.net.URL</code> object to be set
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.4
     */
    @Override
    public void setURL(int parameterIndex, URL x) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);

        if ( x == null)
        {
            setNull(parameterIndex,Types.VARCHAR);
        }
        else
        {
            fields[parameterIndex-1] = x.toString() ;
        }
    }

    /**
     * Retrieves the number, types and properties of this
     * <code>PreparedStatement</code> object's parameters.
     *
     * @return a <code>ParameterMetaData</code> object that contains information
     * about the number, types and properties for each
     * parameter marker of this <code>PreparedStatement</code> object
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @see java.sql.ParameterMetaData
     * @since 1.4
     */
    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException
    {
        //todo test
        checkClosed();
        return new CBParameterMetaData(preparedStatement);
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.RowId</code> object. The
     * driver converts this to a SQL <code>ROWID</code> value when it sends it
     * to the database
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setRowId(int parameterIndex, RowId x) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setRowId");
    }

    /**
     * Sets the designated paramter to the given <code>String</code> object.
     * The driver converts this to a SQL <code>NCHAR</code> or
     * <code>NVARCHAR</code> or <code>LONGNVARCHAR</code> value
     * (depending on the argument's
     * size relative to the driver's limits on <code>NVARCHAR</code> values)
     * when it sends it to the database.
     *
     * @param parameterIndex of the first parameter is 1, the second is 2, ...
     * @param value          the parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if the driver does not support national
     *                                                  character sets;  if the driver can detect that a data conversion
     *                                                  error could occur; if a database access error occurs; or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNString(int parameterIndex, String value) throws SQLException
    {
        checkClosed();
        checkFields(parameterIndex);
        if ( value == null)
        {
            setNull(parameterIndex,Types.VARCHAR);
        }
        else
        {
            fields[parameterIndex-1] =  value ;
        }

    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object. The
     * <code>Reader</code> reads the data till end-of-file is reached. The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     *
     * @param parameterIndex of the first parameter is 1, the second is 2, ...
     * @param value          the parameter value
     * @param length         the number of characters in the parameter data.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if the driver does not support national
     *                                                  character sets;  if the driver can detect that a data conversion
     *                                                  error could occur; if a database access error occurs; or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setNCharacterStream");
    }

    /**
     * Sets the designated parameter to a <code>java.sql.NClob</code> object. The driver converts this to a
     * SQL <code>NCLOB</code> value when it sends it to the database.
     *
     * @param parameterIndex of the first parameter is 1, the second is 2, ...
     * @param value          the parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if the driver does not support national
     *                                                  character sets;  if the driver can detect that a data conversion
     *                                                  error could occur; if a database access error occurs; or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNClob(int parameterIndex, NClob value) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setNClob");
    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object.  The reader must contain  the number
     * of characters specified by length otherwise a <code>SQLException</code> will be
     * generated when the <code>PreparedStatement</code> is executed.
     * This method differs from the <code>setCharacterStream (int, Reader, int)</code> method
     * because it informs the driver that the parameter value should be sent to
     * the server as a <code>CLOB</code>.  When the <code>setCharacterStream</code> method is used, the
     * driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGVARCHAR</code> or a <code>CLOB</code>
     *
     * @param parameterIndex index of the first parameter is 1, the second is 2, ...
     * @param reader         An object that contains the data to set the parameter value to.
     * @param length         the number of characters in the parameter data.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs; this method is called on
     *                                                  a closed <code>PreparedStatement</code> or if the length specified is less than zero.
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setClob(int parameterIndex, Reader reader, long length) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setClob");
    }

    /**
     * Sets the designated parameter to a <code>InputStream</code> object.  The inputstream must contain  the number
     * of characters specified by length otherwise a <code>SQLException</code> will be
     * generated when the <code>PreparedStatement</code> is executed.
     * This method differs from the <code>setBinaryStream (int, InputStream, int)</code>
     * method because it informs the driver that the parameter value should be
     * sent to the server as a <code>BLOB</code>.  When the <code>setBinaryStream</code> method is used,
     * the driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGVARBINARY</code> or a <code>BLOB</code>
     *
     * @param parameterIndex index of the first parameter is 1,
     *                       the second is 2, ...
     * @param inputStream    An object that contains the data to set the parameter
     *                       value to.
     * @param length         the number of bytes in the parameter data.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs;
     *                                                  this method is called on a closed <code>PreparedStatement</code>;
     *                                                  if the length specified
     *                                                  is less than zero or if the number of bytes in the inputstream does not match
     *                                                  the specfied length.
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setBlob");
    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object.  The reader must contain  the number
     * of characters specified by length otherwise a <code>SQLException</code> will be
     * generated when the <code>PreparedStatement</code> is executed.
     * This method differs from the <code>setCharacterStream (int, Reader, int)</code> method
     * because it informs the driver that the parameter value should be sent to
     * the server as a <code>NCLOB</code>.  When the <code>setCharacterStream</code> method is used, the
     * driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGNVARCHAR</code> or a <code>NCLOB</code>
     *
     * @param parameterIndex index of the first parameter is 1, the second is 2, ...
     * @param reader         An object that contains the data to set the parameter value to.
     * @param length         the number of characters in the parameter data.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if the length specified is less than zero;
     *                                                  if the driver does not support national character sets;
     *                                                  if the driver can detect that a data conversion
     *                                                  error could occur;  if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setNClob");
    }

    /**
     * Sets the designated parameter to the given <code>java.sql.SQLXML</code> object.
     * The driver converts this to an
     * SQL <code>XML</code> value when it sends it to the database.
     * 
     *
     * @param parameterIndex index of the first parameter is 1, the second is 2, ...
     * @param xmlObject      a <code>SQLXML</code> object that maps an SQL <code>XML</code> value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs;
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     *                                                  or the <code>java.xml.transform.Result</code>,
     *                                                  <code>Writer</code> or <code>OutputStream</code> has not been closed for
     *                                                  the <code>SQLXML</code> object
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException
    {

    }

    /**
     * <p>Sets the value of the designated parameter with the given object. The second
     * argument must be an object type; for integral values, the
     * <code>java.lang</code> equivalent objects should be used.
     * 
     * If the second argument is an <code>InputStream</code> then the stream must contain
     * the number of bytes specified by scaleOrLength.  If the second argument is a
     * <code>Reader</code> then the reader must contain the number of characters specified
     * by scaleOrLength. If these conditions are not true the driver will generate a
     * <code>SQLException</code> when the prepared statement is executed.
     * 
     * <p>The given Java object will be converted to the given targetSqlType
     * before being sent to the database.
     * 
     * If the object has a custom mapping (is of a class implementing the
     * interface <code>SQLData</code>),
     * the JDBC driver should call the method <code>SQLData.writeSQL</code> to
     * write it to the SQL data stream.
     * If, on the other hand, the object is of a class implementing
     * <code>Ref</code>, <code>Blob</code>, <code>Clob</code>,  <code>NClob</code>,
     * <code>Struct</code>, <code>java.net.URL</code>,
     * or <code>Array</code>, the driver should pass it to the database as a
     * value of the corresponding SQL type.
     * 
     * <p>Note that this method may be used to pass database-specific
     * abstract data types.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param in              the object containing the input parameter value
     * @param targetSqlType  the SQL type (as defined in java.sql.Types) to be
     *                       sent to the database. The scale argument may further qualify this type.
     * @param scaleOrLength  for <code>java.sql.Types.DECIMAL</code>
     *                       or <code>java.sql.Types.NUMERIC types</code>,
     *                       this is the number of digits after the decimal point. For
     *                       Java Object types <code>InputStream</code> and <code>Reader</code>,
     *                       this is the length
     *                       of the data in the stream or reader.  For all other types,
     *                       this value will be ignored.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs;
     *                                                  this method is called on a closed <code>PreparedStatement</code> or
     *                                                  if the Java Object specified by x is an InputStream
     *                                                  or Reader object and the value of the scale parameter is less
     *                                                  than zero
     * @throws java.sql.SQLFeatureNotSupportedException if <code>targetSqlType</code> is
     *                                                  a <code>ARRAY</code>, <code>BLOB</code>, <code>CLOB</code>,
     *                                                  <code>DATALINK</code>, <code>JAVA_OBJECT</code>, <code>NCHAR</code>,
     *                                                  <code>NCLOB</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>,
     *                                                  <code>REF</code>, <code>ROWID</code>, <code>SQLXML</code>
     *                                                  or  <code>STRUCT</code> data type and the JDBC driver does not support
     *                                                  this data type
     * @see java.sql.Types
     * @since 1.6
     */
    @Override
    public void setObject(int parameterIndex, Object in, int targetSqlType, int scaleOrLength) throws SQLException
    {

        checkClosed();

        if (in == null)
        {
            setNull(parameterIndex, targetSqlType);
            return ;
        }

        switch (targetSqlType)
        {
            case Types.INTEGER:
                setInt(parameterIndex, castToInt(in));
                break;
            case Types.TINYINT:
            case Types.SMALLINT:
                setShort(parameterIndex, castToShort(in));
                break;
            case Types.BIGINT:
                setLong(parameterIndex, castToLong(in));
                break;
            case Types.REAL:
                setFloat(parameterIndex, castToFloat(in));
                break;
            case Types.DOUBLE:
            case Types.FLOAT:
                setDouble(parameterIndex, castToDouble(in));
                break;
            case Types.DECIMAL:
            case Types.NUMERIC:
                setBigDecimal(parameterIndex, castToBigDecimal(in, scaleOrLength));
                break;
            case Types.CHAR:
                setString(parameterIndex, castToString(in));
                break;
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
                setString(parameterIndex, castToString(in));
                break;
            case Types.DATE:
                if (in instanceof java.sql.Date)
                    setDate(parameterIndex, (java.sql.Date)in);
                else
                {
                    java.sql.Date tmpd =null;
                    if (in instanceof java.util.Date) {
                        tmpd = new java.sql.Date(((java.util.Date)in).getTime());
                    }
                    //todo parse string
                    setDate(parameterIndex, tmpd);
                }
                break;
            case Types.TIME:
                if (in instanceof java.sql.Time)
                    setTime(parameterIndex, (java.sql.Time)in);
                else
                {
                    java.sql.Time tmpt=null;
                    if (in instanceof java.util.Date) {
                        tmpt = new java.sql.Time(((java.util.Date)in).getTime());
                    }
                    //todo parse string
                    setTime(parameterIndex, tmpt);
                }
                break;
            case Types.TIMESTAMP:
                if (in instanceof java.sql.Timestamp)
                    setTimestamp(parameterIndex , (java.sql.Timestamp)in);
                else
                {
                    java.sql.Timestamp tmpts = null;
                    if (in instanceof java.util.Date)
                    {
                        tmpts = new java.sql.Timestamp(((java.util.Date) in).getTime());
                    }
                    //todo parse string
                    setTimestamp(parameterIndex, tmpts);
                }
                break;
            case Types.BIT:
                setBoolean(parameterIndex, castToBoolean(in));
                break;
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY:
                setObject(parameterIndex, in);
                break;
            case Types.BLOB:
                if (in instanceof Blob)
                {
                    setBlob(parameterIndex, (Blob)in);
                }
                else
                {
                    throw new SQLException("Cannot cast an instance of " + in.getClass().getName() + " to type " + Types.BLOB);
                }
                break;
            case Types.CLOB:
                if (in instanceof Clob)
                    setClob(parameterIndex, (Clob)in);
                else
                    throw new SQLException("Cannot cast an instance of " + in.getClass().getName() + " to type " + Types.CLOB);
                break;
            case Types.ARRAY:
                if (in instanceof Array)
                    setArray(parameterIndex, (Array)in);
                else
                    throw new SQLException("Cannot cast an instance of " + in.getClass().getName() + " to type " + Types.ARRAY);
                break;
            case Types.DISTINCT:
                setString(parameterIndex, in.toString());
                break;
            case Types.JAVA_OBJECT:
                    setString(parameterIndex, in.toString());
                break;
            default:
                throw new SQLException("Unsupported Types value: " +targetSqlType );

        }
    }
    private static String asString(final Clob in) throws SQLException {
        return in.getSubString(1, (int) in.length());
    }
    private static int castToInt(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return Integer.parseInt((String) in);
            if (in instanceof Number)
                return ((Number) in).intValue();
            if (in instanceof java.util.Date)
                return (int) ((java.util.Date) in).getTime();
            if (in instanceof Boolean)
                return (Boolean) in ? 1 : 0;
            if (in instanceof Clob)
                return Integer.parseInt(asString((Clob) in));
            if (in instanceof Character)
                return Integer.parseInt(in.toString());
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "int", e);
        }
        throw cannotCastException(in.getClass().getName(), "int");
    }

    private static short castToShort(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return Short.parseShort((String) in);
            if (in instanceof Number)
                return ((Number) in).shortValue();
            if (in instanceof java.util.Date)
                return (short) ((java.util.Date) in).getTime();
            if (in instanceof Boolean)
                return (Boolean) in ? (short) 1 : (short) 0;
            if (in instanceof Clob)
                return Short.parseShort(asString((Clob) in));
            if (in instanceof Character)
                return Short.parseShort(in.toString());
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "short", e);
        }
        throw cannotCastException(in.getClass().getName(), "short");
    }

    private static long castToLong(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return Long.parseLong((String) in);
            if (in instanceof Number)
                return ((Number) in).longValue();
            if (in instanceof java.util.Date)
                return ((java.util.Date) in).getTime();
            if (in instanceof Boolean)
                return (Boolean) in ? 1L : 0L;
            if (in instanceof Clob)
                return Long.parseLong(asString((Clob) in));
            if (in instanceof Character)
                return Long.parseLong(in.toString());
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "long", e);
        }
        throw cannotCastException(in.getClass().getName(), "long");
    }

    private static float castToFloat(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return Float.parseFloat((String) in);
            if (in instanceof Number)
                return ((Number) in).floatValue();
            if (in instanceof java.util.Date)
                return ((java.util.Date) in).getTime();
            if (in instanceof Boolean)
                return (Boolean) in ? 1f : 0f;
            if (in instanceof Clob)
                return Float.parseFloat(asString((Clob) in));
            if (in instanceof Character)
                return Float.parseFloat(in.toString());
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "float", e);
        }
        throw cannotCastException(in.getClass().getName(), "float");
    }

    private static double castToDouble(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return Double.parseDouble((String) in);
            if (in instanceof Number)
                return ((Number) in).doubleValue();
            if (in instanceof java.util.Date)
                return ((java.util.Date) in).getTime();
            if (in instanceof Boolean)
                return (Boolean) in ? 1d : 0d;
            if (in instanceof Clob)
                return Double.parseDouble(asString((Clob) in));
            if (in instanceof Character)
                return Double.parseDouble(in.toString());
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "double", e);
        }
        throw cannotCastException(in.getClass().getName(), "double");
    }

    private static BigDecimal castToBigDecimal(final Object in, final int scale) throws SQLException {
        try {
            if (in instanceof String)
                return new BigDecimal((String) in).setScale(scale, RoundingMode.HALF_UP);
            if (in instanceof BigDecimal)
                return ((BigDecimal) in).setScale(scale, RoundingMode.HALF_UP);
            if (in instanceof BigInteger)
                return new BigDecimal((BigInteger) in, scale);
            if (in instanceof Long || in instanceof Integer || in instanceof Short || in instanceof Byte)
                return BigDecimal.valueOf(((Number) in).longValue(), scale);
            if (in instanceof Double || in instanceof Float)
                return BigDecimal.valueOf(((Number) in).doubleValue()).setScale(scale, RoundingMode.HALF_UP);
            if (in instanceof java.util.Date)
                return BigDecimal.valueOf(((java.util.Date) in).getTime(), scale);
            if (in instanceof Boolean)
                return (Boolean) in ? BigDecimal.ONE : BigDecimal.ZERO;
            if (in instanceof Clob)
                return new BigDecimal(asString((Clob) in));
            if (in instanceof Character)
                return new BigDecimal(new char[] {(Character) in}).setScale(scale, RoundingMode.HALF_UP);
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "BigDecimal", e);
        }
        throw cannotCastException(in.getClass().getName(), "BigDecimal");
    }

    private static boolean castToBoolean(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return ((String) in).equalsIgnoreCase("true") || in.equals("1") || ((String) in).equalsIgnoreCase("t");
            if (in instanceof BigDecimal)
                return ((BigDecimal) in).signum() != 0;
            if (in instanceof Number)
                return ((Number) in).longValue() != 0L;
            if (in instanceof java.util.Date)
                return ((java.util.Date) in).getTime() != 0L;
            if (in instanceof Boolean)
                return (Boolean) in;
            if (in instanceof Clob) {
                final String asString = asString((Clob) in);
                return asString.equalsIgnoreCase("true") || asString.equals("1") || asString.equalsIgnoreCase("t");
            }
            if (in instanceof Character)
                return (Character) in == '1' || (Character) in == 't' || (Character) in == 'T';
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "boolean", e);
        }
        throw cannotCastException(in.getClass().getName(), "boolean");
    }

    private static String castToString(final Object in) throws SQLException {
        try {
            if (in instanceof String)
                return (String) in;
            if (in instanceof Number || in instanceof Boolean || in instanceof Character || in instanceof java.util.Date)
                return in.toString();
            if (in instanceof Clob)
                return asString((Clob) in);
        } catch (final Exception e) {
            throw cannotCastException(in.getClass().getName(), "String", e);
        }
        throw cannotCastException(in.getClass().getName(), "String");
    }

    private static SQLException cannotCastException(final String fromType, final String toType) {
        return cannotCastException(fromType, toType, null);
    }

    private static SQLException cannotCastException(final String fromType, final String toType, final Exception cause) {
        return new SQLException("Cannot convert an instance of " + fromType + " to type " + toType,  cause);

    }
    /**
     * Sets the designated parameter to the given input stream, which will have
     * the specified number of bytes.
     * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code>. Data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from ASCII to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the Java input stream that contains the ASCII parameter value
     * @param length         the number of bytes in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.6
     */
    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, long length) throws SQLException
    {

        //todo test
        checkClosed();
        checkFields(parameterIndex);
        if ( length > Integer.MAX_VALUE )
        {
            throw new SQLException("Length greater than " + Integer.MAX_VALUE );

        }
        setCharacterStream(parameterIndex,x, (int)length,"ASCII");

    }

    /**
     * Sets the designated parameter to the given input stream, which will have
     * the specified number of bytes.
     * When a very large binary value is input to a <code>LONGVARBINARY</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code> object. The data will be read from the
     * stream as needed until end-of-file is reached.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the java input stream which contains the binary parameter value
     * @param length         the number of bytes in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.6
     */
    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException
    {
        //todo test
        checkClosed();
        checkFields(parameterIndex);
        if ( length > Integer.MAX_VALUE )
        {
            throw new SQLException("Length greater than " + Integer.MAX_VALUE );

        }
        //todo  need to encode this
        setCharacterStream(parameterIndex,x, (int)length,"ASCII");
    }

    /**
     * Sets the designated parameter to the given <code>Reader</code>
     * object, which is the given number of characters long.
     * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.Reader</code> object. The data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param reader         the <code>java.io.Reader</code> object that contains the
     *                       Unicode data
     * @param length         the number of characters in the stream
     * @throws java.sql.SQLException if parameterIndex does not correspond to a parameter
     *                               marker in the SQL statement; if a database access error occurs or
     *                               this method is called on a closed <code>PreparedStatement</code>
     * @since 1.6
     */
    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException
    {
        //todo test
        checkClosed();
        checkFields(parameterIndex);

        if ( length > Integer.MAX_VALUE )
        {
            throw new SQLException("Length greater than " + Integer.MAX_VALUE );

        }
        setCharacterStream(parameterIndex,reader, (int)length);
    }

    /**
     * Sets the designated parameter to the given input stream.
     * When a very large ASCII value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code>. Data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from ASCII to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setAsciiStream</code> which takes a length parameter.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the Java input stream that contains the ASCII parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setAsciiStream(int parameterIndex, InputStream x) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setAsciiStream");
        //todo test
    }

    /**
     * Sets the designated parameter to the given input stream.
     * When a very large binary value is input to a <code>LONGVARBINARY</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.InputStream</code> object. The data will be read from the
     * stream as needed until end-of-file is reached.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setBinaryStream</code> which takes a length parameter.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param x              the java input stream which contains the binary parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setBinaryStream");
    }

    /**
     * Sets the designated parameter to the given <code>Reader</code>
     * object.
     * When a very large UNICODE value is input to a <code>LONGVARCHAR</code>
     * parameter, it may be more practical to send it via a
     * <code>java.io.Reader</code> object. The data will be read from the stream
     * as needed until end-of-file is reached.  The JDBC driver will
     * do any necessary conversion from UNICODE to the database char format.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setCharacterStream</code> which takes a length parameter.
     *
     * @param parameterIndex the first parameter is 1, the second is 2, ...
     * @param reader         the <code>java.io.Reader</code> object that contains the
     *                       Unicode data
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setCharacterStream");

    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object. The
     * <code>Reader</code> reads the data till end-of-file is reached. The
     * driver does the necessary conversion from Java character format to
     * the national character set in the database.
     * 
     * <P><B>Note:</B> This stream object can either be a standard
     * Java stream object or your own subclass that implements the
     * standard interface.
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setNCharacterStream</code> which takes a length parameter.
     *
     * @param parameterIndex of the first parameter is 1, the second is 2, ...
     * @param value          the parameter value
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if the driver does not support national
     *                                                  character sets;  if the driver can detect that a data conversion
     *                                                  error could occur; if a database access error occurs; or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setCharacterNStream");
        //todo test
    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object.
     * This method differs from the <code>setCharacterStream (int, Reader)</code> method
     * because it informs the driver that the parameter value should be sent to
     * the server as a <code>CLOB</code>.  When the <code>setCharacterStream</code> method is used, the
     * driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGVARCHAR</code> or a <code>CLOB</code>
     * 
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setClob</code> which takes a length parameter.
     *
     * @param parameterIndex index of the first parameter is 1, the second is 2, ...
     * @param reader         An object that contains the data to set the parameter value to.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs; this method is called on
     *                                                  a closed <code>PreparedStatement</code>or if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setClob(int parameterIndex, Reader reader) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setClob");
    }

    /**
     * Sets the designated parameter to a <code>InputStream</code> object.
     * This method differs from the <code>setBinaryStream (int, InputStream)</code>
     * method because it informs the driver that the parameter value should be
     * sent to the server as a <code>BLOB</code>.  When the <code>setBinaryStream</code> method is used,
     * the driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGVARBINARY</code> or a <code>BLOB</code>
     * 
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setBlob</code> which takes a length parameter.
     *
     * @param parameterIndex index of the first parameter is 1,
     *                       the second is 2, ...
     * @param inputStream    An object that contains the data to set the parameter
     *                       value to.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement; if a database access error occurs;
     *                                                  this method is called on a closed <code>PreparedStatement</code> or
     *                                                  if parameterIndex does not correspond
     *                                                  to a parameter marker in the SQL statement,
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setBlob");
    }

    /**
     * Sets the designated parameter to a <code>Reader</code> object.
     * This method differs from the <code>setCharacterStream (int, Reader)</code> method
     * because it informs the driver that the parameter value should be sent to
     * the server as a <code>NCLOB</code>.  When the <code>setCharacterStream</code> method is used, the
     * driver may have to do extra work to determine whether the parameter
     * data should be sent to the server as a <code>LONGNVARCHAR</code> or a <code>NCLOB</code>
     * <P><B>Note:</B> Consult your JDBC driver documentation to determine if
     * it might be more efficient to use a version of
     * <code>setNClob</code> which takes a length parameter.
     *
     * @param parameterIndex index of the first parameter is 1, the second is 2, ...
     * @param reader         An object that contains the data to set the parameter value to.
     * @throws java.sql.SQLException                    if parameterIndex does not correspond to a parameter
     *                                                  marker in the SQL statement;
     *                                                  if the driver does not support national character sets;
     *                                                  if the driver can detect that a data conversion
     *                                                  error could occur;  if a database access error occurs or
     *                                                  this method is called on a closed <code>PreparedStatement</code>
     * @throws java.sql.SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.6
     */
    @Override
    public void setNClob(int parameterIndex, Reader reader) throws SQLException
    {
        throw CBDriver.notImplemented(CBPreparedStatement.class, "setNClob");
    }

    private void setCharacterStream(int parameterIndex, InputStream x, int length, String encoding) throws SQLException
    {

        if (x == null)
        {
            setNull(parameterIndex, Types.VARCHAR);
            return;
        }
        if (length < 0)
            throw new SQLException("Invalid stream length " + length);


        try
        {
            InputStreamReader l_inStream = new InputStreamReader(x, encoding);
            char[] l_chars = new char[length];
            int l_charsRead = 0;
            while (true)
            {
                int n = l_inStream.read(l_chars, l_charsRead, length - l_charsRead);
                if (n == -1)
                    break;

                l_charsRead += n;

                if (l_charsRead == length)
                    break;
            }

            setString(parameterIndex, new String(l_chars, 0, l_charsRead));
        }
        catch (UnsupportedEncodingException l_uee)
        {
            throw new SQLException("The JVM claims not to support the " + encoding + " encoding.");
        }
        catch (IOException l_ioe)
        {
            throw new SQLException("Provided InputStream failed.");
        }
    }
    public void setMap(int parameterIndex, Map map ) throws  SQLException
    {
        checkClosed();
        checkFields(parameterIndex);
        fields[parameterIndex-1] = map;
    }
    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException
    {
        checkClosed();
        // ignore as the statement has already been parsed
    }
    private void checkFields(int index) throws SQLException
    {
        if (fields == null) throw new SQLException("fields not initialized");
        if ( index-1 > fields.length ) throw new SQLException("Column number out of bounds");
    }
/*
    private NameValuePair getPositionalParameters()
    {
        StringBuffer parameters = new StringBuffer("[");
        for (String field:fields)
        {
            parameters.append(field).append(',');
        }
        parameters.deleteCharAt(parameters.lastIndexOf(","));
        parameters.append(']');

        String args = JsonFactory.toJson(fields);
        return new BasicNameValuePair("args",args);

    }
*/
}
