
/*
 * //  Copyright (c) 2015 Couchbase, Inc.
 * //  Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * //  except in compliance with the License. You may obtain a copy of the License at
 * //    http://www.apache.org/licenses/LICENSE-2.0
 * //  Unless required by applicable law or agreed to in writing, software distributed under the
 * //  License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * //  either express or implied. See the License for the specific language governing permissions
 * //  and limitations under the License.
 */

package com.couchbase;

import com.couchbase.jdbc.Protocol;
import com.couchbase.jdbc.util.SqlParser;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.SQLTimeoutException;
import java.sql.SQLFeatureNotSupportedException;

import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by davec on 2015-02-20.
 */
public class CBStatement implements java.sql.Statement
{
    Protocol protocol;
    AtomicBoolean closed = new AtomicBoolean(false);
    protected int updateCount=0;
    protected CBResultSet resultSet;
    protected Connection connection;
    protected boolean escapeProcessing = true;
    protected int maxRows = -1;
    protected boolean poolable=false;
    protected boolean closeOnCompletion = false;

    public CBStatement( Connection connection, Protocol protocol )
    {
        this.connection = connection;
        this.protocol = protocol;
    }

    /**
     * Executes the given SQL statement, which returns a single
     * <code>ResultSet</code> object.
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql an SQL statement to be sent to the database, typically a
     *            static SQL <code>SELECT</code> statement
     * @return a <code>ResultSet</code> object that contains the data produced
     * by the given query; never <code>null</code>
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>, the given
     *                               SQL statement produces anything other than a single
     *                               <code>ResultSet</code> object, the method is called on a
     *                               <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     */
    @Override
    public ResultSet executeQuery(String sql) throws SQLException
    {
        checkClosed();
        SqlParser sqlParser = new SqlParser("");
        sql = sqlParser.replaceProcessing(sql, escapeProcessing);
        return protocol.query(this, sql);
    }

    /**
     * Executes the given SQL statement, which may be an <code>INSERT</code>,
     * <code>UPDATE</code>, or <code>DELETE</code> statement or an
     * SQL statement that returns nothing, such as an SQL DDL statement.
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
     *            <code>DELETE</code>; or an SQL statement that returns nothing,
     *            such as a DDL statement.
     * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
     * or (2) 0 for SQL statements that return nothing
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>, the given
     *                               SQL statement produces a <code>ResultSet</code> object, the method is called on a
     *                               <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     */
    @Override
    public int executeUpdate(String sql) throws SQLException
    {
        checkClosed();
        SqlParser sqlParser = new SqlParser("");
        sql = sqlParser.replaceProcessing(sql, escapeProcessing);
        return protocol.executeUpdate(this, sql);
    }

    /**
     * Releases this <code>Statement</code> object's database
     * and JDBC resources immediately instead of waiting for
     * this to happen when it is automatically closed.
     * It is generally good practice to release resources as soon as
     * you are finished with them to avoid tying up database
     * resources.
     * 
     * Calling the method <code>close</code> on a <code>Statement</code>
     * object that is already closed has no effect.
     * 
     * <B>Note:</B>When a <code>Statement</code> object is
     * closed, its current <code>ResultSet</code> object, if one exists, is
     * also closed.
     *
     * @throws java.sql.SQLException if a database access error occurs
     */
    @Override
    public void close() throws SQLException
    {
        closed.getAndSet(true);
    }

    /**
     * Retrieves the maximum number of bytes that can be
     * returned for character and binary column values in a <code>ResultSet</code>
     * object produced by this <code>Statement</code> object.
     * This limit applies only to  <code>BINARY</code>, <code>VARBINARY</code>,
     * <code>LONGVARBINARY</code>, <code>CHAR</code>, <code>VARCHAR</code>,
     * <code>NCHAR</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code>
     * and <code>LONGVARCHAR</code> columns.  If the limit is exceeded, the
     * excess data is silently discarded.
     *
     * @return the current column size limit for columns storing character and
     * binary values; zero means there is no limit
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #setMaxFieldSize
     */
    @Override
    public int getMaxFieldSize() throws SQLException
    {
        checkClosed();
        // 20MB
        return 20*1024*1024;
    }

    /**
     * Sets the limit for the maximum number of bytes that can be returned for
     * character and binary column values in a <code>ResultSet</code>
     * object produced by this <code>Statement</code> object.
     * 
     * This limit applies
     * only to <code>BINARY</code>, <code>VARBINARY</code>,
     * <code>LONGVARBINARY</code>, <code>CHAR</code>, <code>VARCHAR</code>,
     * <code>NCHAR</code>, <code>NVARCHAR</code>, <code>LONGNVARCHAR</code> and
     * <code>LONGVARCHAR</code> fields.  If the limit is exceeded, the excess data
     * is silently discarded. For maximum portability, use values
     * greater than 256.
     *
     * @param max the new column size limit in bytes; zero means there is no limit
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>
     *                               or the condition max &gt;= 0 is not satisfied
     * @see #getMaxFieldSize
     */
    @Override
    public void setMaxFieldSize(int max) throws SQLException
    {
        checkClosed();
    }

    /**
     * Retrieves the maximum number of rows that a
     * <code>ResultSet</code> object produced by this
     * <code>Statement</code> object can contain.  If this limit is exceeded,
     * the excess rows are silently dropped.
     *
     * @return the current maximum number of rows for a <code>ResultSet</code>
     * object produced by this <code>Statement</code> object;
     * zero means there is no limit
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #setMaxRows
     */
    @Override
    public int getMaxRows() throws SQLException
    {
        checkClosed();
        return maxRows;
    }

    /**
     * Sets the limit for the maximum number of rows that any
     * <code>ResultSet</code> object  generated by this <code>Statement</code>
     * object can contain to the given number.
     * If the limit is exceeded, the excess
     * rows are silently dropped.
     *
     * @param max the new max rows limit; zero means there is no limit
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>
     *                               or the condition max &gt;= 0 is not satisfied
     * @see #getMaxRows
     */
    @Override
    public void setMaxRows(int max) throws SQLException
    {
        checkClosed();
        maxRows = max;
    }

    /**
     * Sets escape processing on or off.
     * If escape scanning is on (the default), the driver will do
     * escape substitution before sending the SQL statement to the database.
     * 
     * Note: Since prepared statements have usually been parsed prior
     * to making this call, disabling escape processing for
     * <code>PreparedStatements</code> objects will have no effect.
     *
     * @param enable <code>true</code> to enable escape processing;
     *               <code>false</code> to disable it
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     */
    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException
    {
        checkClosed();
        escapeProcessing = enable;

    }

    /**
     * Retrieves the number of seconds the driver will
     * wait for a <code>Statement</code> object to execute.
     * If the limit is exceeded, a
     * <code>SQLException</code> is thrown.
     *
     * @return the current query timeout limit in seconds; zero means there is
     * no limit
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #setQueryTimeout
     */
    @Override
    public int getQueryTimeout() throws SQLException
    {
        checkClosed();
        return protocol.getQueryTimeout();
    }

    /**
     * Sets the number of seconds the driver will wait for a
     * <code>Statement</code> object to execute to the given number of seconds.
     * By default there is no limit on the amount of time allowed for a running
     * statement to complete. If the limit is exceeded, an
     * <code>SQLTimeoutException</code> is thrown.
     * A JDBC driver must apply this limit to the <code>execute</code>,
     * <code>executeQuery</code> and <code>executeUpdate</code> methods.
     * 
     * <strong>Note:</strong> JDBC driver implementations may also apply this
     * limit to {@code ResultSet} methods
     * (consult your driver vendor documentation for details).
     * 
     * <strong>Note:</strong> In the case of {@code Statement} batching, it is
     * implementation defined as to whether the time-out is applied to
     * individual SQL commands added via the {@code addBatch} method or to
     * the entire batch of SQL commands invoked by the {@code executeBatch}
     * method (consult your driver vendor documentation for details).
     *
     * @param seconds the new query timeout limit in seconds; zero means
     *                there is no limit
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>
     *                               or the condition seconds &gt;= 0 is not satisfied
     * @see #getQueryTimeout
     */
    @Override
    public void setQueryTimeout(int seconds) throws SQLException
    {
        checkClosed();
        protocol.setQueryTimeout(seconds);
        //todo test
    }

    /**
     * Cancels this <code>Statement</code> object if both the DBMS and
     * driver support aborting an SQL statement.
     * This method can be used by one thread to cancel a statement that
     * is being executed by another thread.
     *
     * @throws java.sql.SQLException           if a database access error occurs or
     *                                         this method is called on a closed <code>Statement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     *                                         this method
     */
    @Override
    public void cancel() throws SQLException
    {
        throw CBDriver.notImplemented(CBStatement.class, "cancel");
        //todo test
    }

    /**
     * Retrieves the first warning reported by calls on this <code>Statement</code> object.
     * Subsequent <code>Statement</code> object warnings will be chained to this
     * <code>SQLWarning</code> object.
     * 
     * <p>The warning chain is automatically cleared each time
     * a statement is (re)executed. This method may not be called on a closed
     * <code>Statement</code> object; doing so will cause an <code>SQLException</code>
     * to be thrown.
     * 
     * <P><B>Note:</B> If you are processing a <code>ResultSet</code> object, any
     * warnings associated with reads on that <code>ResultSet</code> object
     * will be chained on it rather than on the <code>Statement</code>
     * object that produced it.
     *
     * @return the first <code>SQLWarning</code> object or <code>null</code>
     * if there are no warnings
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     */
    @Override
    public SQLWarning getWarnings() throws SQLException
    {
        checkClosed();
        //todo implement
        return null;

    }

    /**
     * Clears all the warnings reported on this <code>Statement</code>
     * object. After a call to this method,
     * the method <code>getWarnings</code> will return
     * <code>null</code> until a new warning is reported for this
     * <code>Statement</code> object.
     *
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     */
    @Override
    public void clearWarnings() throws SQLException
    {
        checkClosed();
        //todo implement
    }

    /**
     * Sets the SQL cursor name to the given <code>String</code>, which
     * will be used by subsequent <code>Statement</code> object
     * <code>execute</code> methods. This name can then be
     * used in SQL positioned update or delete statements to identify the
     * current row in the <code>ResultSet</code> object generated by this
     * statement.  If the database does not support positioned update/delete,
     * this method is a noop.  To insure that a cursor has the proper isolation
     * level to support updates, the cursor's <code>SELECT</code> statement
     * should have the form <code>SELECT FOR UPDATE</code>.  If
     * <code>FOR UPDATE</code> is not present, positioned updates may fail.
     * 
     * <P><B>Note:</B> By definition, the execution of positioned updates and
     * deletes must be done by a different <code>Statement</code> object than
     * the one that generated the <code>ResultSet</code> object being used for
     * positioning. Also, cursor names must be unique within a connection.
     *
     * @param name the new cursor name, which must be unique within
     *             a connection
     * @throws java.sql.SQLException           if a database access error occurs or
     *                                         this method is called on a closed <code>Statement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     */
    @Override
    public void setCursorName(String name) throws SQLException
    {
        //todo needs test
        throw CBDriver.notImplemented(CBStatement.class, "setCursorName");
    }

    /**
     * Executes the given SQL statement, which may return multiple results.
     * In some (uncommon) situations, a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.
     * 
     * The <code>execute</code> method executes an SQL statement and indicates the
     * form of the first result.  You must then use the methods
     * <code>getResultSet</code> or <code>getUpdateCount</code>
     * to retrieve the result, and <code>getMoreResults</code> to
     * move to any subsequent result(s).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql any SQL statement
     * @return <code>true</code> if the first result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there are
     * no results
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>,
     *                               the method is called on a
     *                               <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults
     */
    @Override
    public boolean execute(String sql) throws SQLException
    {
        checkClosed();

        // todo this needs to be refactored, no reason to store the result in the protocol layer
        boolean hasResult = protocol.execute(this,sql);
        resultSet =  protocol.getResultSet();
        updateCount = (int)protocol.getUpdateCount();
        return hasResult;
    }

    /**
     * Retrieves the current result as a <code>ResultSet</code> object.
     * This method should be called only once per result.
     *
     * @return the current result as a <code>ResultSet</code> object or
     * <code>null</code> if the result is an update count or there are no more results
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #execute
     */
    @Override
    public ResultSet getResultSet() throws SQLException
    {
        checkClosed();
        return resultSet;
    }

    /**
     * Retrieves the current result as an update count;
     * if the result is a <code>ResultSet</code> object or there are no more results, -1
     * is returned. This method should be called only once per result.
     *
     * @return the current result as an update count; -1 if the current result is a
     * <code>ResultSet</code> object or there are no more results
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #execute
     */
    @Override
    public int getUpdateCount() throws SQLException
    {
        checkClosed();
        return updateCount;
    }

    /**
     * Moves to this <code>Statement</code> object's next result, returns
     * <code>true</code> if it is a <code>ResultSet</code> object, and
     * implicitly closes any current <code>ResultSet</code>
     * object(s) obtained with the method <code>getResultSet</code>.
     * 
     * <P>There are no more results when the following is true:
     * <PRE>
     * // stmt is a Statement object
     * ((stmt.getMoreResults() == false) &amp;&amp; (stmt.getUpdateCount() == -1))
     * </PRE>
     *
     * @return <code>true</code> if the next result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there are
     * no more results
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #execute
     */
    @Override
    public boolean getMoreResults() throws SQLException
    {
        checkClosed();
        return false;
    }

    /**
     * Gives the driver a hint as to the direction in which
     * rows will be processed in <code>ResultSet</code>
     * objects created using this <code>Statement</code> object.  The
     * default value is <code>ResultSet.FETCH_FORWARD</code>.
     * 
     * Note that this method sets the default fetch direction for
     * result sets generated by this <code>Statement</code> object.
     * Each result set has its own methods for getting and setting
     * its own fetch direction.
     *
     * @param direction the initial direction for processing rows
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>
     *                               or the given direction
     *                               is not one of <code>ResultSet.FETCH_FORWARD</code>,
     *                               <code>ResultSet.FETCH_REVERSE</code>, or <code>ResultSet.FETCH_UNKNOWN</code>
     * @see #getFetchDirection
     * @since 1.2
     */
    @Override
    public void setFetchDirection(int direction) throws SQLException
    {
        checkClosed();

    }

    /**
     * Retrieves the direction for fetching rows from
     * database tables that is the default for result sets
     * generated from this <code>Statement</code> object.
     * If this <code>Statement</code> object has not set
     * a fetch direction by calling the method <code>setFetchDirection</code>,
     * the return value is implementation-specific.
     *
     * @return the default fetch direction for result sets generated
     * from this <code>Statement</code> object
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #setFetchDirection
     * @since 1.2
     */
    @Override
    public int getFetchDirection() throws SQLException
    {
        checkClosed();
        return ResultSet.FETCH_FORWARD;
    }

    /**
     * Gives the JDBC driver a hint as to the number of rows that should
     * be fetched from the database when more rows are needed for
     * <code>ResultSet</code> objects genrated by this <code>Statement</code>.
     * If the value specified is zero, then the hint is ignored.
     * The default value is zero.
     *
     * @param rows the number of rows to fetch
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code> or the
     *                               condition  <code>rows &gt;= 0</code> is not satisfied.
     * @see #getFetchSize
     * @since 1.2
     */
    @Override
    public void setFetchSize(int rows) throws SQLException
    {
        checkClosed();
    }

    /**
     * Retrieves the number of result set rows that is the default
     * fetch size for <code>ResultSet</code> objects
     * generated from this <code>Statement</code> object.
     * If this <code>Statement</code> object has not set
     * a fetch size by calling the method <code>setFetchSize</code>,
     * the return value is implementation-specific.
     *
     * @return the default fetch size for result sets generated
     * from this <code>Statement</code> object
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @see #setFetchSize
     * @since 1.2
     */
    @Override
    public int getFetchSize() throws SQLException
    {
        checkClosed();
        return Short.MAX_VALUE;
    }

    /**
     * Retrieves the result set concurrency for <code>ResultSet</code> objects
     * generated by this <code>Statement</code> object.
     *
     * @return either <code>ResultSet.CONCUR_READ_ONLY</code> or
     * <code>ResultSet.CONCUR_UPDATABLE</code>
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @since 1.2
     */
    @Override
    public int getResultSetConcurrency() throws SQLException
    {
        checkClosed();
        return ResultSet.CONCUR_READ_ONLY;
    }

    /**
     * Retrieves the result set type for <code>ResultSet</code> objects
     * generated by this <code>Statement</code> object.
     *
     * @return one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     * <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or
     * <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @since 1.2
     */
    @Override
    public int getResultSetType() throws SQLException
    {
        checkClosed();
        return ResultSet.TYPE_FORWARD_ONLY;
    }

    /**
     * Adds the given SQL command to the current list of commmands for this
     * <code>Statement</code> object. The commands in this list can be
     * executed as a batch by calling the method <code>executeBatch</code>.
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql typically this is a SQL <code>INSERT</code> or
     *            <code>UPDATE</code> statement
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code>, the
     *                               driver does not support batch updates, the method is called on a
     *                               <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @see #executeBatch
     * @see CBDatabaseMetaData#supportsBatchUpdates
     * @since 1.2
     */
    @Override
    public void addBatch(String sql) throws SQLException
    {
        checkClosed();
        protocol.addBatch(sql);
    }

    /**
     * Empties this <code>Statement</code> object's current list of
     * SQL commands.
     * 
     *
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code> or the
     *                               driver does not support batch updates
     * @see #addBatch
     * @see CBDatabaseMetaData#supportsBatchUpdates
     * @since 1.2
     */
    @Override
    public void clearBatch() throws SQLException
    {
        checkClosed();
        protocol.clearBatch();

    }

    /**
     * Submits a batch of commands to the database for execution and
     * if all commands execute successfully, returns an array of update counts.
     * The <code>int</code> elements of the array that is returned are ordered
     * to correspond to the commands in the batch, which are ordered
     * according to the order in which they were added to the batch.
     * The elements in the array returned by the method <code>executeBatch</code>
     * may be one of the following:
     * <OL>
     * <LI>A number greater than or equal to zero -- indicates that the
     * command was processed successfully and is an update count giving the
     * number of rows in the database that were affected by the command's
     * execution
     * <LI>A value of <code>SUCCESS_NO_INFO</code> -- indicates that the command was
     * processed successfully but that the number of rows affected is
     * unknown
     * 
     * If one of the commands in a batch update fails to execute properly,
     * this method throws a <code>BatchUpdateException</code>, and a JDBC
     * driver may or may not continue to process the remaining commands in
     * the batch.  However, the driver's behavior must be consistent with a
     * particular DBMS, either always continuing to process commands or never
     * continuing to process commands.  If the driver continues processing
     * after a failure, the array returned by the method
     * <code>BatchUpdateException.getUpdateCounts</code>
     * will contain as many elements as there are commands in the batch, and
     * at least one of the elements will be the following:
     * 
     * <LI>A value of <code>EXECUTE_FAILED</code> -- indicates that the command failed
     * to execute successfully and occurs only if a driver continues to
     * process commands after a command fails
     * </OL>
     * 
     * The possible implementations and return values have been modified in
     * the Java 2 SDK, Standard Edition, version 1.3 to
     * accommodate the option of continuing to proccess commands in a batch
     * update after a <code>BatchUpdateException</code> obejct has been thrown.
     *
     * @return an array of update counts containing one element for each
     * command in the batch.  The elements of the array are ordered according
     * to the order in which commands were added to the batch.
     * @throws java.sql.SQLException if a database access error occurs,
     *                               this method is called on a closed <code>Statement</code> or the
     *                               driver does not support batch statements. Throws <code>BatchUpdateException</code>
     *                               (a subclass of <code>SQLException</code>) if one of the commands sent to the
     *                               database fails to execute properly or attempts to return a result set.
     * @throws SQLTimeoutException   when the driver has determined that the
     *                               timeout value that was specified by the {@code setQueryTimeout}
     *                               method has been exceeded and has at least attempted to cancel
     *                               the currently running {@code Statement}
     * @see #addBatch
     * @see CBDatabaseMetaData#supportsBatchUpdates
     * @since 1.2
     */
    @Override
    public int[] executeBatch() throws SQLException
    {
        checkClosed();
        return protocol.executeBatch();
    }

    /**
     * Retrieves the <code>Connection</code> object
     * that produced this <code>Statement</code> object.
     *
     * @return the connection that produced this statement
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @since 1.2
     */
    @Override
    public Connection getConnection() throws SQLException
    {

        checkClosed();
        return connection;
    }

    /**
     * Moves to this <code>Statement</code> object's next result, deals with
     * any current <code>ResultSet</code> object(s) according  to the instructions
     * specified by the given flag, and returns
     * <code>true</code> if the next result is a <code>ResultSet</code> object.
     * 
     * <P>There are no more results when the following is true:
     * <PRE>
     * // stmt is a Statement object
     * ((stmt.getMoreResults(current) == false) &amp;&amp; (stmt.getUpdateCount() == -1))
     * </PRE>
     *
     * @param current one of the following <code>Statement</code>
     *                constants indicating what should happen to current
     *                <code>ResultSet</code> objects obtained using the method
     *                <code>getResultSet</code>:
     *                <code>Statement.CLOSE_CURRENT_RESULT</code>,
     *                <code>Statement.KEEP_CURRENT_RESULT</code>, or
     *                <code>Statement.CLOSE_ALL_RESULTS</code>
     * @return <code>true</code> if the next result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there are no
     * more results
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code> or the argument
     *                                         supplied is not one of the following:
     *                                         <code>Statement.CLOSE_CURRENT_RESULT</code>,
     *                                         <code>Statement.KEEP_CURRENT_RESULT</code> or
     *                                         <code>Statement.CLOSE_ALL_RESULTS</code>
     * @throws SQLFeatureNotSupportedException if
     *                                         <code>DatabaseMetaData.supportsMultipleOpenResults</code> returns
     *                                         <code>false</code> and either
     *                                         <code>Statement.KEEP_CURRENT_RESULT</code> or
     *                                         <code>Statement.CLOSE_ALL_RESULTS</code> are supplied as
     *                                         the argument.
     * @see #execute
     * @since 1.4
     */
    @Override
    public boolean getMoreResults(int current) throws SQLException
    {
        throw CBDriver.notImplemented(CBStatement.class, "getMoreResults");
        //todo need to test
    }

    /**
     * Retrieves any auto-generated keys created as a result of executing this
     * <code>Statement</code> object. If this <code>Statement</code> object did
     * not generate any keys, an empty <code>ResultSet</code>
     * object is returned.
     * 
     * <p><B>Note:</B>If the columns which represent the auto-generated keys were not specified,
     * the JDBC driver implementation will determine the columns which best represent the auto-generated keys.
     *
     * @return a <code>ResultSet</code> object containing the auto-generated key(s)
     * generated by the execution of this <code>Statement</code> object
     * @throws java.sql.SQLException           if a database access error occurs or
     *                                         this method is called on a closed <code>Statement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @since 1.4
     */
    @Override
    public ResultSet getGeneratedKeys() throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "getGeneratedKeys");
        //todo needs test
    }

    /**
     * Executes the given SQL statement and signals the driver with the
     * given flag about whether the
     * auto-generated keys produced by this <code>Statement</code> object
     * should be made available for retrieval.  The driver will ignore the
     * flag if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql               an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
     *                          <code>DELETE</code>; or an SQL statement that returns nothing,
     *                          such as a DDL statement.
     * @param autoGeneratedKeys a flag indicating whether auto-generated keys
     *                          should be made available for retrieval;
     *                          one of the following constants:
     *                          <code>Statement.RETURN_GENERATED_KEYS</code>
     *                          <code>Statement.NO_GENERATED_KEYS</code>
     * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
     * or (2) 0 for SQL statements that return nothing
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>, the given
     *                                         SQL statement returns a <code>ResultSet</code> object,
     *                                         the given constant is not one of those allowed, the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     *                                         this method with a constant of Statement.RETURN_GENERATED_KEYS
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @since 1.4
     */
    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Executes the given SQL statement and signals the driver that the
     * auto-generated keys indicated in the given array should be made available
     * for retrieval.   This array contains the indexes of the columns in the
     * target table that contain the auto-generated keys that should be made
     * available. The driver will ignore the array if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql           an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
     *                      <code>DELETE</code>; or an SQL statement that returns nothing,
     *                      such as a DDL statement.
     * @param columnIndexes an array of column indexes indicating the columns
     *                      that should be returned from the inserted row
     * @return either (1) the row count for SQL Data Manipulation Language (DML) statements
     * or (2) 0 for SQL statements that return nothing
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>, the SQL
     *                                         statement returns a <code>ResultSet</code> object,the second argument
     *                                         supplied to this method is not an
     *                                         <code>int</code> array whose elements are valid column indexes, the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @since 1.4
     */
    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Executes the given SQL statement and signals the driver that the
     * auto-generated keys indicated in the given array should be made available
     * for retrieval.   This array contains the names of the columns in the
     * target table that contain the auto-generated keys that should be made
     * available. The driver will ignore the array if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql         an SQL Data Manipulation Language (DML) statement, such as <code>INSERT</code>, <code>UPDATE</code> or
     *                    <code>DELETE</code>; or an SQL statement that returns nothing,
     *                    such as a DDL statement.
     * @param columnNames an array of the names of the columns that should be
     *                    returned from the inserted row
     * @return either the row count for <code>INSERT</code>, <code>UPDATE</code>,
     * or <code>DELETE</code> statements, or 0 for SQL statements
     * that return nothing
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>, the SQL
     *                                         statement returns a <code>ResultSet</code> object, the
     *                                         second argument supplied to this method is not a <code>String</code> array
     *                                         whose elements are valid column names, the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @since 1.4
     */
    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Executes the given SQL statement, which may return multiple results,
     * and signals the driver that any
     * auto-generated keys should be made available
     * for retrieval.  The driver will ignore this signal if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * In some (uncommon) situations, a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.
     * 
     * The <code>execute</code> method executes an SQL statement and indicates the
     * form of the first result.  You must then use the methods
     * <code>getResultSet</code> or <code>getUpdateCount</code>
     * to retrieve the result, and <code>getMoreResults</code> to
     * move to any subsequent result(s).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql               any SQL statement
     * @param autoGeneratedKeys a constant indicating whether auto-generated
     *                          keys should be made available for retrieval using the method
     *                          <code>getGeneratedKeys</code>; one of the following constants:
     *                          <code>Statement.RETURN_GENERATED_KEYS</code> or
     *                          <code>Statement.NO_GENERATED_KEYS</code>
     * @return <code>true</code> if the first result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there are
     * no results
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>, the second
     *                                         parameter supplied to this method is not
     *                                         <code>Statement.RETURN_GENERATED_KEYS</code> or
     *                                         <code>Statement.NO_GENERATED_KEYS</code>,
     *                                         the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support
     *                                         this method with a constant of Statement.RETURN_GENERATED_KEYS
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults
     * @see #getGeneratedKeys
     * @since 1.4
     */
    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Executes the given SQL statement, which may return multiple results,
     * and signals the driver that the
     * auto-generated keys indicated in the given array should be made available
     * for retrieval.  This array contains the indexes of the columns in the
     * target table that contain the auto-generated keys that should be made
     * available.  The driver will ignore the array if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * Under some (uncommon) situations, a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.
     * 
     * The <code>execute</code> method executes an SQL statement and indicates the
     * form of the first result.  You must then use the methods
     * <code>getResultSet</code> or <code>getUpdateCount</code>
     * to retrieve the result, and <code>getMoreResults</code> to
     * move to any subsequent result(s).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql           any SQL statement
     * @param columnIndexes an array of the indexes of the columns in the
     *                      inserted row that should be  made available for retrieval by a
     *                      call to the method <code>getGeneratedKeys</code>
     * @return <code>true</code> if the first result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there
     * are no results
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>, the
     *                                         elements in the <code>int</code> array passed to this method
     *                                         are not valid column indexes, the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults
     * @since 1.4
     */
    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Executes the given SQL statement, which may return multiple results,
     * and signals the driver that the
     * auto-generated keys indicated in the given array should be made available
     * for retrieval. This array contains the names of the columns in the
     * target table that contain the auto-generated keys that should be made
     * available.  The driver will ignore the array if the SQL statement
     * is not an <code>INSERT</code> statement, or an SQL statement able to return
     * auto-generated keys (the list of such statements is vendor-specific).
     * 
     * In some (uncommon) situations, a single SQL statement may return
     * multiple result sets and/or update counts.  Normally you can ignore
     * this unless you are (1) executing a stored procedure that you know may
     * return multiple results or (2) you are dynamically executing an
     * unknown SQL string.
     * 
     * The <code>execute</code> method executes an SQL statement and indicates the
     * form of the first result.  You must then use the methods
     * <code>getResultSet</code> or <code>getUpdateCount</code>
     * to retrieve the result, and <code>getMoreResults</code> to
     * move to any subsequent result(s).
     * 
     * <strong>Note:</strong>This method cannot be called on a
     * <code>PreparedStatement</code> or <code>CallableStatement</code>.
     *
     * @param sql         any SQL statement
     * @param columnNames an array of the names of the columns in the inserted
     *                    row that should be made available for retrieval by a call to the
     *                    method <code>getGeneratedKeys</code>
     * @return <code>true</code> if the next result is a <code>ResultSet</code>
     * object; <code>false</code> if it is an update count or there
     * are no more results
     * @throws java.sql.SQLException           if a database access error occurs,
     *                                         this method is called on a closed <code>Statement</code>,the
     *                                         elements of the <code>String</code> array passed to this
     *                                         method are not valid column names, the method is called on a
     *                                         <code>PreparedStatement</code> or <code>CallableStatement</code>
     * @throws SQLFeatureNotSupportedException if the JDBC driver does not support this method
     * @throws SQLTimeoutException             when the driver has determined that the
     *                                         timeout value that was specified by the {@code setQueryTimeout}
     *                                         method has been exceeded and has at least attempted to cancel
     *                                         the currently running {@code Statement}
     * @see #getResultSet
     * @see #getUpdateCount
     * @see #getMoreResults
     * @see #getGeneratedKeys
     * @since 1.4
     */
    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException
    {
        throw CBDriver.notImplemented(ResultSet.class, "executeUpdate");
        //TODO NEEDS TEST
    }

    /**
     * Retrieves the result set holdability for <code>ResultSet</code> objects
     * generated by this <code>Statement</code> object.
     *
     * @return either <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> or
     * <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code>
     * @throws java.sql.SQLException if a database access error occurs or
     *                               this method is called on a closed <code>Statement</code>
     * @since 1.4
     */
    @Override
    public int getResultSetHoldability() throws SQLException
    {
        checkClosed();
        return ResultSet.CLOSE_CURSORS_AT_COMMIT;
        //todo needs test
    }

    /**
     * Retrieves whether this <code>Statement</code> object has been closed. A <code>Statement</code> is closed if the
     * method close has been called on it, or if it is automatically closed.
     *
     * @return true if this <code>Statement</code> object is closed; false if it is still open
     * @throws java.sql.SQLException if a database access error occurs
     * @since 1.6
     */
    @Override
    public boolean isClosed() throws SQLException
    {
        return closed.get();
    }

    /**
     * Requests that a <code>Statement</code> be pooled or not pooled.  The value
     * specified is a hint to the statement pool implementation indicating
     * whether the applicaiton wants the statement to be pooled.  It is up to
     * the statement pool manager as to whether the hint is used.
     * 
     * The poolable value of a statement is applicable to both internal
     * statement caches implemented by the driver and external statement caches
     * implemented by application servers and other applications.
     * 
     * By default, a <code>Statement</code> is not poolable when created, and
     * a <code>PreparedStatement</code> and <code>CallableStatement</code>
     * are poolable when created.
     * 
     *
     * @param poolable requests that the statement be pooled if true and
     *                 that the statement not be pooled if false
     *                 
     * @throws java.sql.SQLException if this method is called on a closed
     *                               <code>Statement</code>
     *                               
     * @since 1.6
     */
    @Override
    public void setPoolable(boolean poolable) throws SQLException
    {
        checkClosed();
        this.poolable = poolable;
    }

    /**
     * Returns a  value indicating whether the <code>Statement</code>
     * is poolable or not.
     * 
     *
     * @return <code>true</code> if the <code>Statement</code>
     * is poolable; <code>false</code> otherwise
     * 
     * @throws java.sql.SQLException if this method is called on a closed
     *                               <code>Statement</code>
     *                               
     * @see java.sql.Statement#setPoolable(boolean) setPoolable(boolean)
     * @since 1.6
     * 
     */
    @Override
    public boolean isPoolable() throws SQLException
    {
        checkClosed();
        return poolable;
    }

    /**
     * Specifies that this {@code Statement} will be closed when all its
     * dependent result sets are closed. If execution of the {@code Statement}
     * does not produce any result sets, this method has no effect.
     * 
     * <strong>Note:</strong> Multiple calls to {@code closeOnCompletion} do
     * not toggle the effect on this {@code Statement}. However, a call to
     * {@code closeOnCompletion} does effect both the subsequent execution of
     * statements, and statements that currently have open, dependent,
     * result sets.
     *
     * @throws java.sql.SQLException if this method is called on a closed
     *                               {@code Statement}
     * @since 1.7
     */
    @Override
    public void closeOnCompletion() throws SQLException
    {
        checkClosed();
        closeOnCompletion = true;
    }

    /**
     * Returns a value indicating whether this {@code Statement} will be
     * closed when all its dependent result sets are closed.
     *
     * @return {@code true} if the {@code Statement} will be closed when all
     * of its dependent result sets are closed; {@code false} otherwise
     * @throws java.sql.SQLException if this method is called on a closed
     *                               {@code Statement}
     * @since 1.7
     */
    @Override
    public boolean isCloseOnCompletion() throws SQLException
    {
        checkClosed();
        return closeOnCompletion;
    }

    /**
     * Returns an object that implements the given interface to allow access to
     * non-standard methods, or standard methods not exposed by the proxy.
     * 
     * If the receiver implements the interface then the result is the receiver
     * or a proxy for the receiver. If the receiver is a wrapper
     * and the wrapped object implements the interface then the result is the
     * wrapped object or a proxy for the wrapped object. Otherwise return the
     * the result of calling <code>unwrap</code> recursively on the wrapped object
     * or a proxy for that result. If the receiver is not a
     * wrapper and does not implement the interface, then an <code>SQLException</code> is thrown.
     *
     * @param iface A Class defining an interface that the result must implement.
     * @return an object that implements the interface. May be a proxy for the actual implementing object.
     * @throws java.sql.SQLException If no object found that implements the interface
     * @since 1.6
     */
    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException
    {
        checkClosed();
        if (iface.isAssignableFrom(getClass()))
        {
            return iface.cast(this);
        }
        throw new SQLException("Cannot unwrap to " + iface.getName());
    }

    /**
     * Returns true if this either implements the interface argument or is directly or indirectly a wrapper
     * for an object that does. Returns false otherwise. If this implements the interface then return true,
     * else if this is a wrapper then return the result of recursively calling <code>isWrapperFor</code> on the wrapped
     * object. If this does not implement the interface and is not a wrapper, return false.
     * This method should be implemented as a low-cost operation compared to <code>unwrap</code> so that
     * callers can use this method to avoid expensive <code>unwrap</code> calls that may fail. If this method
     * returns true then calling <code>unwrap</code> with the same argument should succeed.
     *
     * @param iface a Class defining an interface.
     * @return true if this implements the interface or directly or indirectly wraps an object that does.
     * @throws java.sql.SQLException if an error occurs while determining whether this is a wrapper
     *                               for an object with the given interface.
     * @since 1.6
     */
    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException
    {
        checkClosed();
        return iface.isAssignableFrom(getClass());
    }
    protected void checkClosed() throws SQLException
    {
        if (isClosed()) throw new SQLException("Statement is closed");
    }
}
