/*
 * Copyright 2015 - Jeandeson O. Merelis
 */
package com.github.jeanmerelis.jeanson.ext.typehandler;

/*
 * #%L
 * jeanson
 * %%
 * Copyright (C) 2015 Jeandeson O. Merelis
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
import com.github.jeanmerelis.jeanson.typehandler.DefaultStringHandler;
import com.github.jeanmerelis.jeanson.typehandler.TypeHandler;
import java.io.IOException;
import java.io.Writer;
import org.joda.time.DateTime;

/**
 *
 * @author Jeandeson O. Merelis
 */
public class JodaDateTimeHandler implements TypeHandler<DateTime> {

    protected String pattern;

    protected static String patternDefault;
    protected static String patternDefaultForDate;
    protected static String patternDefaultForTime;
    protected static String patternDefaultForDateTime;
    protected static boolean usesMills;

    static {
        patternDefault = "yyyy-MM-dd'T'HH:mm:ssZZ";
        patternDefaultForDate = "yyyy-MM-dd";
        patternDefaultForTime = "HH:mm:ss";
        patternDefaultForDateTime = patternDefault;
        usesMills = true;
    }

    public JodaDateTimeHandler() {
        setDefaultValues();
    }

    @Override
    public void write(Writer w, DateTime obj) throws IOException {
        if (obj == null) {
            w.write("null");
            return;
        }
        if (!usesMills) {
            DefaultStringHandler.escapeAndQuote(w, obj.toString(pattern));
        } else {
            w.write(String.valueOf(obj.getMillis()));
        }
    }

    @Override
    public void setConfig(String[] params) {
        if (params == null || params.length == 0) {
            setDefaultValues();
            return;
        }
        for (String s : params) {
            String[] param = s.split("=");
            String key = param[0].trim();
            String value;
            if (param.length > 1) {
                value = param[1];
            } else {
                value = key;
            }
            switch (key) {
                case "date":
                    pattern = patternDefaultForDate;
                    usesMills = false;
                    break;
                case "time":
                    pattern = patternDefaultForTime;
                    usesMills = false;
                    break;
                case "datetime":
                    pattern = patternDefaultForDateTime;
                    usesMills = false;
                    break;
                case "pattern":
                    pattern = value;
                    usesMills = false;
                case "usesMills":
                    usesMills = true;
                    break;
                default:
                    pattern = value;
            }
        }

    }

    private void setDefaultValues() {
        pattern = patternDefault;
        usesMills = true;
    }

    public static String getPatternDefault() {
        return patternDefault;
    }

    public static void setPatternDefault(String patternDefault) {
        JodaDateTimeHandler.patternDefault = patternDefault;
    }

    public static String getPatternDefaultForDate() {
        return patternDefaultForDate;
    }

    public static void setPatternDefaultForDate(String patternDefaultForDate) {
        JodaDateTimeHandler.patternDefaultForDate = patternDefaultForDate;
    }

    public static String getPatternDefaultForTime() {
        return patternDefaultForTime;
    }

    public static void setPatternDefaultForTime(String patternDefaultForTime) {
        JodaDateTimeHandler.patternDefaultForTime = patternDefaultForTime;
    }

    public static String getPatternDefaultForDateTime() {
        return patternDefaultForDateTime;
    }

    public static void setPatternDefaultForDateTime(String patternDefaultForDateTime) {
        JodaDateTimeHandler.patternDefaultForDateTime = patternDefaultForDateTime;
    }

    public static boolean isUsesMills() {
        return usesMills;
    }

    public static void setUsesMills(boolean usesMills) {
        JodaDateTimeHandler.usesMills = usesMills;
    }
}
