/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.Groups;
import org.apache.hadoop.security.ShellBasedUnixGroupsMapping;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.FakeTimer;
import org.apache.hadoop.util.Timer;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestGroupsCaching {
    public static final Log LOG = LogFactory.getLog(TestGroupsCaching.class);
    private static String[] myGroups = new String[]{"grp1", "grp2"};
    private Configuration conf;

    @Before
    public void setup() {
        FakeGroupMapping.resetRequestCount();
        ExceptionalGroupMapping.resetRequestCount();
        this.conf = new Configuration();
        this.conf.setClass("hadoop.security.group.mapping", FakeGroupMapping.class, ShellBasedUnixGroupsMapping.class);
    }

    @Test
    public void testGroupsCaching() throws Exception {
        Groups groups;
        block2: {
            this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
            groups = new Groups(this.conf);
            groups.cacheGroupsAdd(Arrays.asList(myGroups));
            groups.refresh();
            FakeGroupMapping.clearBlackList();
            FakeGroupMapping.addToBlackList("user1");
            Assert.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
            FakeGroupMapping.addToBlackList("me");
            Assert.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
            try {
                LOG.error((Object)("We are not supposed to get here." + groups.getGroups("user1").toString()));
                Assert.fail();
            }
            catch (IOException ioe) {
                if (ioe.getMessage().startsWith("No groups found")) break block2;
                LOG.error((Object)("Got unexpected exception: " + ioe.getMessage()));
                Assert.fail();
            }
        }
        FakeGroupMapping.clearBlackList();
        Assert.assertTrue((groups.getGroups("user1").size() == 2 ? 1 : 0) != 0);
    }

    @Test
    public void testGroupLookupForStaticUsers() throws Exception {
        this.conf.setClass("hadoop.security.group.mapping", FakeunPrivilegedGroupMapping.class, ShellBasedUnixGroupsMapping.class);
        this.conf.set("hadoop.user.group.static.mapping.overrides", "me=;user1=group1;user2=group1,group2");
        Groups groups = new Groups(this.conf);
        List userGroups = groups.getGroups("me");
        Assert.assertTrue((String)"non-empty groups for static user", (boolean)userGroups.isEmpty());
        Assert.assertFalse((String)"group lookup done for static user", (boolean)FakeunPrivilegedGroupMapping.invoked);
        ArrayList<String> expected = new ArrayList<String>();
        expected.add("group1");
        FakeunPrivilegedGroupMapping.invoked = false;
        userGroups = groups.getGroups("user1");
        Assert.assertTrue((String)"groups not correct", (boolean)expected.equals(userGroups));
        Assert.assertFalse((String)"group lookup done for unprivileged user", (boolean)FakeunPrivilegedGroupMapping.invoked);
        expected.add("group2");
        FakeunPrivilegedGroupMapping.invoked = false;
        userGroups = groups.getGroups("user2");
        Assert.assertTrue((String)"groups not correct", (boolean)expected.equals(userGroups));
        Assert.assertFalse((String)"group lookup done for unprivileged user", (boolean)FakeunPrivilegedGroupMapping.invoked);
    }

    @Test
    public void testNegativeGroupCaching() throws Exception {
        String user = "negcache";
        String failMessage = "Did not throw IOException: ";
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 2L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.addToBlackList("negcache");
        try {
            groups.getGroups("negcache");
            Assert.fail((String)"Did not throw IOException: Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        try {
            groups.getGroups("negcache");
            Assert.fail((String)"Did not throw IOException: The user is in the negative cache.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        FakeGroupMapping.clearBlackList();
        try {
            groups.getGroups("negcache");
            Assert.fail((String)"Did not throw IOException: The user is still in the negative cache, even FakeGroupMapping has resumed.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        timer.advance(4000L);
        Assert.assertEquals(Arrays.asList(myGroups), (Object)groups.getGroups("negcache"));
    }

    @Test
    public void testCachePreventsImplRequest() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        Assert.assertEquals((long)0L, (long)FakeGroupMapping.getRequestCount());
        Assert.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
        Assert.assertEquals((long)1L, (long)FakeGroupMapping.getRequestCount());
        Assert.assertTrue((groups.getGroups("me").size() == 2 ? 1 : 0) != 0);
        Assert.assertEquals((long)1L, (long)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testExceptionsFromImplNotCachedInNegativeCache() {
        this.conf.setClass("hadoop.security.group.mapping", ExceptionalGroupMapping.class, ShellBasedUnixGroupsMapping.class);
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 10000L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        Assert.assertEquals((long)0L, (long)ExceptionalGroupMapping.getRequestCount());
        try {
            groups.getGroups("anything");
            Assert.fail((String)"Should have thrown");
        }
        catch (IOException e) {
            // empty catch block
        }
        Assert.assertEquals((long)1L, (long)ExceptionalGroupMapping.getRequestCount());
        try {
            groups.getGroups("anything");
            Assert.fail((String)"Should have thrown");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertEquals((long)2L, (long)ExceptionalGroupMapping.getRequestCount());
    }

    @Test
    public void testOnlyOneRequestWhenNoEntryIsCached() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 0L);
        final Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        ArrayList<1> threads = new ArrayList<1>();
        for (int i = 0; i < 10; ++i) {
            threads.add(new Thread(){

                @Override
                public void run() {
                    try {
                        Assert.assertEquals((long)2L, (long)groups.getGroups("me").size());
                    }
                    catch (IOException e) {
                        Assert.fail((String)"Should not happen");
                    }
                }
            });
        }
        for (Thread thread : threads) {
            thread.start();
        }
        for (Thread thread : threads) {
            thread.join();
        }
        Assert.assertEquals((long)1L, (long)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testOnlyOneRequestWhenExpiredEntryExists() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        FakeTimer timer = new FakeTimer();
        final Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.setGetGroupsDelayMs(100L);
        groups.getGroups("me");
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(400000L);
        Thread.sleep(100L);
        ArrayList<2> threads = new ArrayList<2>();
        for (int i = 0; i < 10; ++i) {
            threads.add(new Thread(){

                @Override
                public void run() {
                    try {
                        Assert.assertEquals((long)2L, (long)groups.getGroups("me").size());
                    }
                    catch (IOException e) {
                        Assert.fail((String)"Should not happen");
                    }
                }
            });
        }
        for (Thread thread : threads) {
            thread.start();
        }
        for (Thread thread : threads) {
            thread.join();
        }
        Assert.assertEquals((long)(startingRequestCount + 1), (long)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testCacheEntriesExpire() throws Exception {
        this.conf.setLong("hadoop.security.groups.cache.secs", 1L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        groups.getGroups("me");
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        timer.advance(20000L);
        groups.getGroups("me");
        Assert.assertEquals((long)(startingRequestCount + 1), (long)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testNegativeCacheClearedOnRefresh() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 100L);
        Groups groups = new Groups(this.conf);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.clearBlackList();
        FakeGroupMapping.addToBlackList("dne");
        try {
            groups.getGroups("dne");
            Assert.fail((String)"Should have failed to find this group");
        }
        catch (IOException e) {
            // empty catch block
        }
        int startingRequestCount = FakeGroupMapping.getRequestCount();
        groups.refresh();
        FakeGroupMapping.addToBlackList("dne");
        try {
            List g = groups.getGroups("dne");
            Assert.fail((String)"Should have failed to find this group");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertEquals((long)(startingRequestCount + 1), (long)FakeGroupMapping.getRequestCount());
    }

    @Test
    public void testNegativeCacheEntriesExpire() throws Exception {
        this.conf.setLong("hadoop.security.groups.negative-cache.secs", 2L);
        FakeTimer timer = new FakeTimer();
        Groups groups = new Groups(this.conf, (Timer)timer);
        groups.cacheGroupsAdd(Arrays.asList(myGroups));
        groups.refresh();
        FakeGroupMapping.addToBlackList("user1");
        FakeGroupMapping.addToBlackList("user2");
        try {
            groups.getGroups("user1");
            Assert.fail((String)"Did not throw IOException : Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        Assert.assertTrue((boolean)groups.getNegativeCache().contains("user1"));
        timer.advance(1000L);
        try {
            groups.getGroups("user2");
            Assert.fail((String)"Did not throw IOException : Failed to obtain groups from FakeGroupMapping.");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains("No groups found for user", e);
        }
        Assert.assertTrue((boolean)groups.getNegativeCache().contains("user2"));
        timer.advance(1100L);
        Assert.assertFalse((boolean)groups.getNegativeCache().contains("user1"));
        Assert.assertTrue((boolean)groups.getNegativeCache().contains("user2"));
        timer.advance(1000L);
        Assert.assertFalse((boolean)groups.getNegativeCache().contains("user2"));
    }

    public static class FakeunPrivilegedGroupMapping
    extends FakeGroupMapping {
        private static boolean invoked = false;

        @Override
        public List<String> getGroups(String user) throws IOException {
            invoked = true;
            return super.getGroups(user);
        }
    }

    public static class ExceptionalGroupMapping
    extends ShellBasedUnixGroupsMapping {
        private static int requestCount = 0;

        public List<String> getGroups(String user) throws IOException {
            ++requestCount;
            throw new IOException("For test");
        }

        public static int getRequestCount() {
            return requestCount;
        }

        public static void resetRequestCount() {
            requestCount = 0;
        }
    }

    public static class FakeGroupMapping
    extends ShellBasedUnixGroupsMapping {
        private static Set<String> allGroups = new HashSet<String>();
        private static Set<String> blackList = new HashSet<String>();
        private static int requestCount = 0;
        private static long getGroupsDelayMs = 0L;

        public List<String> getGroups(String user) throws IOException {
            LOG.info((Object)("Getting groups for " + user));
            ++requestCount;
            this.delayIfNecessary();
            if (blackList.contains(user)) {
                return new LinkedList<String>();
            }
            return new LinkedList<String>(allGroups);
        }

        private void delayIfNecessary() {
            if (getGroupsDelayMs > 0L) {
                try {
                    Thread.sleep(getGroupsDelayMs);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        public void cacheGroupsRefresh() throws IOException {
            LOG.info((Object)"Cache is being refreshed.");
            FakeGroupMapping.clearBlackList();
        }

        public static void clearBlackList() throws IOException {
            LOG.info((Object)"Clearing the blacklist");
            blackList.clear();
        }

        public void cacheGroupsAdd(List<String> groups) throws IOException {
            LOG.info((Object)("Adding " + groups + " to groups."));
            allGroups.addAll(groups);
        }

        public static void addToBlackList(String user) throws IOException {
            LOG.info((Object)("Adding " + user + " to the blacklist"));
            blackList.add(user);
        }

        public static int getRequestCount() {
            return requestCount;
        }

        public static void resetRequestCount() {
            requestCount = 0;
        }

        public static void setGetGroupsDelayMs(long delayMs) {
            getGroupsDelayMs = delayMs;
        }
    }
}

