/*   __    __         _
 *   \ \  / /__ _ __ (_) ___ ___ 
 *    \ \/ / _ \ '_ \| |/ __/ _ \
 *     \  /  __/ | | | | (_|  __/
 *      \/ \___|_| |_|_|\___\___|
 *
 *
 * Copyright 2014-2018 Venice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.jlangch.venice.javainterop;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * Keeps the sandbox rules.
 * 
 * <p>The sandbox keeps whitelist rules for the Java Interop and blacklist rules for the
 * Venice functions.
 * 
 * <p>Java whitelist rules for class/instance accessor follow the schema: 
 * '{package}.{className}:{methodName | fieldName}'. The asterix may be used as a wildcard
 * 
 * <p>
 * E.g: white listing Java Interop
 * <ul>
 *   <li>java.lang.Boolean (allow calling Java methods with arguments or return values of type Boolean)</li>
 *   <li>java.lang.* (allow calling Java methods with arguments or return values of any type in the package 'java.lang')</li>
 *   <li>java.lang.Long:new (allow calling Long constructor)</li>
 *   <li>java.lang.Math:abs (allow calling Math::abs method)</li>
 *   <li>java.lang.Math:* (allow calling all Math constructors/methods/fields)</li>
 *   <li>java.lang.*:*  (allow calling all constructors/methods/fields for classes in the package 'java.lang')</li>
 * </ul>
 * 
 * <p>
 * E.g: white listing Java system properties
 * <ul>
 *   <li>system.property:file.separator</li>
 *   <li>system.property:java.home</li>
 * </ul>
 * 
 * <p>
 * E.g: black listing Venice I/O functions
 * <ul>
 *   <li>blacklist:venice:io/slurp (reject calls to 'io/slurp')</li>
 *   <li>blacklist:venice:*io* (reject all Venice I/O calls like 'io/slurp', 'create-file', ...)</li>
 *   <li>blacklist:venice:. (reject java interop)</li>
 * </ul>
 */
public class SandboxRules {
	
	public SandboxRules() {
	}
	
	/**
	 * Add rules to the sandbox
	 * 
	 * @param rules a collection of rules
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules add(final Collection<String> rules) {
		if (rules != null) {
			this.rules.addAll(rules);
		}
		return this;
	}
	
	/**
	 * Add rules to the sandbox
	 * 
	 * @param rules rules
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules add(final String... rules) {
		if (rules != null) {
			this.rules.addAll(Arrays.asList(rules));
		}
		return this;
	}
	
	/**
	 * Add classes to the sandbox. 
	 * 
	 * <p>Adds a rule "x.y.classname:*" for each class
	 * 
	 * @param classes classes
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules addClasses(final Class<?>... classes) {
		if (classes != null) {
			for(Class<?> clazz : classes) {
				this.rules.add(clazz.getName() + ":*");
			}
		}
		return this;
	}
	
	/**
	 * Add classes to the sandbox. 
	 * 
	 * <p>Adds a rule "x.y.classname:*" for each class
	 * 
	 * @param classes classes
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules addClasses(final Collection<Class<?>> classes) {
		if (classes != null) {
			for(Class<?> clazz : classes) {
				this.rules.add(clazz.getName() + ":*");
			}
		}
		return this;
	}
	
	/**
	 * Reject access to all Venice I/O related functions
	 * 
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules rejectAllVeniceIoFunctions() {
		if (rules != null) {
			this.rules.add("blacklist:venice:*io*");
		}
		return this;
	}
	
	/**
	 * Allow access to all standard Java system properties
	 * 
	 * <p>Standard system properties:
	 * <ul>
	 *   <li>file.separator</li>
	 *   <li>java.home</li>
	 *   <li>java.vendor</li>
	 *   <li>java.vendor.url</li>
	 *   <li>java.version</li>
	 *   <li>line.separator</li>
	 *   <li>os.arch</li>
	 *   <li>os.name</li>
	 *   <li>os.version</li>
	 *   <li>path.separator</li>
	 *   <li>user.dir</li>
	 *   <li>user.home</li>
	 *   <li>user.name</li>
	 * </ul>
	 * 
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules allowAccessToStandardSystemProperties() {
		rules.addAll(DEFAULT_SYSTEM_PROPERTIES
				.stream()
				.map(p -> "system.property:" + p)
				.collect(Collectors.toSet()));		
		return this;
	}
	
	/**
	 * Allow access to all Java system properties
	 * 
	 * @return this <code>SandboxRules</code>
	 */
	public SandboxRules allowAccessToAllSystemProperties() {
		rules.add("system.property:*");
		return this;
	}
		
	/**
	 * Merges this <code>SandboxRules</code> with the passed other 
	 * <code>SandboxRules</code> 
	 * 
	 * @param other the other SandboxRules to merge with
	 * @return the new merged <code>SandboxRules</code>
	 */
	public SandboxRules merge(final SandboxRules other) {
		final SandboxRules merged = new SandboxRules();
		merged.add(this.rules);
		merged.add(other.rules);
		return merged;
	}
	
	/**
	 * @return the rules of this <code>SandboxRules</code>
	 */
	public Set<String> getRules() {
		return Collections.unmodifiableSet(rules);
	}

	@Override
	public String toString() {
		return new ArrayList<String>(rules)
					.stream()
					.sorted()
					.collect(Collectors.joining("\n"));
	}

	
	private static final Set<String> DEFAULT_CLASS_RULES = 
			new HashSet<>(
				Arrays.asList(
						// Dynamic proxies based on venice' DynamicInvocationHandler
						"com.github.jlangch.venice.javainterop.DynamicInvocationHandler*:*",
						
						"java.lang.Object:class",
						
						Byte.class.getName(),
						Short.class.getName(),
						Integer.class.getName(),
						Long.class.getName(),
						Float.class.getName(),
						Double.class.getName(),
						BigDecimal.class.getName(),
						String.class.getName(),
						
						ArrayList.class.getName(),
						HashSet.class.getName(),
						HashMap.class.getName(),
						LinkedHashMap.class.getName()));

	public static final Set<String> DEFAULT_SYSTEM_PROPERTIES = 
			Collections.unmodifiableSet(
				new HashSet<>(
					Arrays.asList(
							"file.separator",
							"java.home",
							"java.vendor",
							"java.vendor.url",
							"java.version",
							"line.separator",
							"os.arch",
							"os.name",
							"os.version",
							"path.separator",
							"user.dir",
							"user.home",
							"user.name")));


	private final Set<String> rules = new HashSet<>(DEFAULT_CLASS_RULES);
}
