/*
 * Decompiled with CFR 0.152.
 */
package com.github.jlangch.venice.impl.repl;

import com.github.jlangch.venice.ParseError;
import com.github.jlangch.venice.impl.Destructuring;
import com.github.jlangch.venice.impl.debug.agent.Break;
import com.github.jlangch.venice.impl.debug.agent.IDebugAgent;
import com.github.jlangch.venice.impl.debug.agent.StepMode;
import com.github.jlangch.venice.impl.debug.breakpoint.BreakpointParser;
import com.github.jlangch.venice.impl.debug.util.StepValidity;
import com.github.jlangch.venice.impl.env.Env;
import com.github.jlangch.venice.impl.env.Var;
import com.github.jlangch.venice.impl.repl.TerminalPrinter;
import com.github.jlangch.venice.impl.types.VncFunction;
import com.github.jlangch.venice.impl.types.VncVal;
import com.github.jlangch.venice.impl.types.collections.VncCollection;
import com.github.jlangch.venice.impl.types.collections.VncList;
import com.github.jlangch.venice.impl.types.collections.VncVector;
import com.github.jlangch.venice.impl.types.util.Types;
import com.github.jlangch.venice.impl.util.CallFrame;
import com.github.jlangch.venice.impl.util.CallStack;
import com.github.jlangch.venice.impl.util.CollectionUtil;
import com.github.jlangch.venice.impl.util.StringUtil;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

public class ReplDebugClient {
    private static final Set<String> DEBUG_COMMANDS = new HashSet<String>(Arrays.asList("attach", "detach", "terminate", "info", "i", "breakpoint", "b", "resume", "r", "step-any", "sa", "step-next", "sn", "step-next-", "sn-", "step-call", "sc", "step-over", "so", "step-entry", "se", "step-exit", "sx", "breaks", "switch-break", "sb", "break?", "b?", "callstack", "cs", "params", "p", "locals", "l", "retval", "ret", "ex"));
    private CallFrame currCallFrame;
    private int currCallFrameLevel = 0;
    private final TerminalPrinter printer;
    private final IDebugAgent agent;
    private final Thread replThread;

    public ReplDebugClient(IDebugAgent agent, TerminalPrinter printer, Thread replThread) {
        this.agent = agent;
        this.printer = printer;
        this.replThread = replThread;
        agent.addBreakListener(this::breakpointListener);
    }

    public static void pringHelp(TerminalPrinter printer) {
        printer.println("stdout", "Venice debugger\n\nThe debugger can break functions at 4 levels:\n  call:        breaks before the passed parameters are evaluated. The\n               unevaluated function parameters are available for inspection.\n  entry:       breaks right after the passed parameters have been evaluated.\n               The evaluated function parameters are available for inspection.\n  exception:   breaks on catching an exception within the function's body. The\n               exception and the evaluated functions parameters are available\n               for inspection.\n  exit:        breaks before returning from the function. The return value and\n               the evaluated function parameters are available for inspection.\n\nCommands:\n  !attach      Attach the debugger to the REPL.\n  !detach      Detach the debugger from the REPL.\n  !terminate   Terminate a running debug session. Sends an interrupt to the\n               script under debugging.\n  !info        Print detail info on the current debug session.\n  !breakpoint  Manage breakpoints\n               o Add a breakpoint\n                  !breakpoint add n\n                  E.g.: !breakpoint add foo/gauss\n                        Ancestor selectors:\n                          direct ancestor: !breakpoint add foo/gauss > filter\n                          any ancestor:    !breakpoint add foo/gauss + filter\n               o Remove a breakpoint\n                  !breakpoint remove n\n                  E.g.: !breakpoint remove foo/gauss\n               o Temporarily skip/unskip all breakpoints\n                  !breakpoint skip\n                  !breakpoint unskip\n                  !breakpoint skip?\n               o List breakpoints\n                  !breakpoint list\n                  E.g.: !breakpoint list\n               Short form: !b ...\n  !resume      Resume from current break.\n               Short form: !r\n  !resume-all  Resume from all breaks.\n               Short form: !ra\n  !step        Step to the next available break at one of the four break\n               levels within the current or the next function whatever is first.\n               Short form: !s\n  !step-next   Step to next function at entry level.\n               Short form: !sn\n  !step-over   Step over the current function to next function entry.\n               Implicitely steps over functions involved with function\n               parameter evaluation.\n               Short form: !so\n  !step-call   Step to next function at call level.\n               Short form: !sc\n  !step-entry  Step to the entry level of the current function.\n               Short form: !se\n  !step-exit   Step to the exit level of the current function.\n               Short form: !sx\n  !breaks      List all breaks. The first break listed is the current break.\n  !break n     Switch to break n as the current break. n is an index from the\n               !breaks command). \n  !break?      Checks if the debugger is in a break.\n  !params      Print the function's parameters.\n               Short form: !p\n  !retval      Print the function's return value.\n               Short form: !ret\n  !ex          Print the function's exception\n  !locals x    Print the local vars from the level x. The level is optional\n               and defaults to the top level.\n               Short form: !l\n  !callstack   Print the current callstack.\n               Short form: !cs\n  !callstack x Callstack command\n               o Select a callframe. The !list and !params commands work on\n                 the current callstack  callframe if one is selected.\n                  !callstack select n\n               o Unselect the callframe\n                  !callstack unselect\n  form         Runs a Venice form in the current break context. Useful to\n               inspect parameters, return values, or global/local vars.\n               Note: Debugging is suspended for evaluating the form if a\n                     break is active!\n               E.g.:  (first param1)");
    }

    public static boolean isDebugCommand(String cmd) {
        return DEBUG_COMMANDS.contains(cmd);
    }

    public Env getEnv() {
        return this.currCallFrame == null ? this.agent.getActiveBreak().getEnv() : this.currCallFrame.getEnv();
    }

    public void handleCommand(String cmdLine) {
        String cmd;
        List<String> params = Arrays.asList(cmdLine.split(" +"));
        StepValidity stepValidity = null;
        switch (cmd = StringUtil.trimToEmpty(CollectionUtil.first(params))) {
            case "info": 
            case "i": {
                this.printer.println("stdout", this.agent.toString());
                this.printer.println("stdout", "Current CallFrame:  " + (this.currCallFrame == null ? "-" : this.currCallFrame));
                break;
            }
            case "breakpoint": 
            case "b": {
                this.handleBreakpointCmd(CollectionUtil.drop(params, 1));
                break;
            }
            case "resume": 
            case "r": {
                this.clearCurrCallFrame();
                this.agent.resume();
                break;
            }
            case "resume-all": 
            case "ra": {
                this.clearCurrCallFrame();
                this.agent.resumeAll();
                break;
            }
            case "step": 
            case "s": 
            case "step-any": 
            case "sa": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToAny);
                if (stepValidity.isValid()) {
                    stepValidity = this.agent.step(StepMode.StepToAny);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-next": 
            case "sn": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToNextFunction);
                if (stepValidity.isValid()) {
                    stepValidity = this.agent.step(StepMode.StepToNextFunction);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-next-": 
            case "sn-": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToNextNonSystemFunction);
                if (stepValidity.isValid()) {
                    stepValidity = this.agent.step(StepMode.StepToNextNonSystemFunction);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-call": 
            case "sc": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToNextFunctionCall);
                if (stepValidity.isValid()) {
                    stepValidity = this.agent.step(StepMode.StepToNextFunctionCall);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-over": 
            case "so": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepOverFunction);
                if (stepValidity.isValid()) {
                    stepValidity = this.agent.step(StepMode.StepOverFunction);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-entry": 
            case "se": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToFunctionEntry);
                if (stepValidity.isValid()) {
                    this.println("Stepping to entry of function %s ...", this.agent.getActiveBreak().getFn().getQualifiedName());
                    stepValidity = this.agent.step(StepMode.StepToFunctionEntry);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "step-exit": 
            case "sx": {
                this.clearCurrCallFrame();
                stepValidity = this.agent.isStepPossible(StepMode.StepToFunctionExit);
                if (stepValidity.isValid()) {
                    this.println("Stepping to exit of function %s ...", this.agent.getActiveBreak().getFn().getQualifiedName());
                    stepValidity = this.agent.step(StepMode.StepToFunctionExit);
                }
                if (stepValidity.isValid()) break;
                this.printer.println("stdout", this.agent.toString());
                this.printlnErr(stepValidity);
                break;
            }
            case "breaks": {
                this.printBreakList();
                break;
            }
            case "break": {
                this.switchBreak(CollectionUtil.first(CollectionUtil.drop(params, 1)));
                break;
            }
            case "break?": {
                this.isBreak();
                break;
            }
            case "callstack": 
            case "cs": {
                this.handleCallstackCmd(CollectionUtil.drop(params, 1));
                break;
            }
            case "params": 
            case "p": {
                this.params(CollectionUtil.drop(params, 1));
                break;
            }
            case "locals": 
            case "l": {
                this.locals(CollectionUtil.second(params));
                break;
            }
            case "retval": 
            case "ret": {
                this.retval();
                break;
            }
            case "ex": {
                this.ex();
                break;
            }
            case "help": 
            case "?": {
                this.println("Venice debugger\n\nThe debugger can break functions at 4 levels:\n  call:        breaks before the passed parameters are evaluated. The\n               unevaluated function parameters are available for inspection.\n  entry:       breaks right after the passed parameters have been evaluated.\n               The evaluated function parameters are available for inspection.\n  exception:   breaks on catching an exception within the function's body. The\n               exception and the evaluated functions parameters are available\n               for inspection.\n  exit:        breaks before returning from the function. The return value and\n               the evaluated function parameters are available for inspection.\n\nCommands:\n  !attach      Attach the debugger to the REPL.\n  !detach      Detach the debugger from the REPL.\n  !terminate   Terminate a running debug session. Sends an interrupt to the\n               script under debugging.\n  !info        Print detail info on the current debug session.\n  !breakpoint  Manage breakpoints\n               o Add a breakpoint\n                  !breakpoint add n\n                  E.g.: !breakpoint add foo/gauss\n                        Ancestor selectors:\n                          direct ancestor: !breakpoint add foo/gauss > filter\n                          any ancestor:    !breakpoint add foo/gauss + filter\n               o Remove a breakpoint\n                  !breakpoint remove n\n                  E.g.: !breakpoint remove foo/gauss\n               o Temporarily skip/unskip all breakpoints\n                  !breakpoint skip\n                  !breakpoint unskip\n                  !breakpoint skip?\n               o List breakpoints\n                  !breakpoint list\n                  E.g.: !breakpoint list\n               Short form: !b ...\n  !resume      Resume from current break.\n               Short form: !r\n  !resume-all  Resume from all breaks.\n               Short form: !ra\n  !step        Step to the next available break at one of the four break\n               levels within the current or the next function whatever is first.\n               Short form: !s\n  !step-next   Step to next function at entry level.\n               Short form: !sn\n  !step-over   Step over the current function to next function entry.\n               Implicitely steps over functions involved with function\n               parameter evaluation.\n               Short form: !so\n  !step-call   Step to next function at call level.\n               Short form: !sc\n  !step-entry  Step to the entry level of the current function.\n               Short form: !se\n  !step-exit   Step to the exit level of the current function.\n               Short form: !sx\n  !breaks      List all breaks. The first break listed is the current break.\n  !break n     Switch to break n as the current break. n is an index from the\n               !breaks command). \n  !break?      Checks if the debugger is in a break.\n  !params      Print the function's parameters.\n               Short form: !p\n  !retval      Print the function's return value.\n               Short form: !ret\n  !ex          Print the function's exception\n  !locals x    Print the local vars from the level x. The level is optional\n               and defaults to the top level.\n               Short form: !l\n  !callstack   Print the current callstack.\n               Short form: !cs\n  !callstack x Callstack command\n               o Select a callframe. The !list and !params commands work on\n                 the current callstack  callframe if one is selected.\n                  !callstack select n\n               o Unselect the callframe\n                  !callstack unselect\n  form         Runs a Venice form in the current break context. Useful to\n               inspect parameters, return values, or global/local vars.\n               Note: Debugging is suspended for evaluating the form if a\n                     break is active!\n               E.g.:  (first param1)", new Object[0]);
                break;
            }
            default: {
                this.printlnErr("Invalid debug command '%s'. Use '!help' for help.\n\nTo run a REPL non debug command, detach the debugger first using '!detach'.", cmd);
            }
        }
    }

    private void handleBreakpointCmd(List<String> params) {
        if (params.isEmpty()) {
            this.printBreakpoints();
        } else {
            try {
                String cmd;
                switch (cmd = StringUtil.trimToEmpty(params.get(0))) {
                    case "add": 
                    case "a": {
                        this.agent.addBreakpoints(BreakpointParser.parseBreakpoints(CollectionUtil.drop(params, 1)));
                        break;
                    }
                    case "remove": 
                    case "rem": 
                    case "r": {
                        this.agent.removeBreakpoints(BreakpointParser.parseBreakpoints(CollectionUtil.drop(params, 1)));
                        break;
                    }
                    case "clear": 
                    case "c": {
                        this.agent.removeAllBreakpoints();
                        break;
                    }
                    case "skip": 
                    case "s": {
                        this.agent.skipBreakpoints(true);
                        break;
                    }
                    case "unskip": 
                    case "u": {
                        this.agent.skipBreakpoints(false);
                        break;
                    }
                    case "skip?": {
                        this.println("Skip breakpoints: %s", this.agent.isSkipBreakpoints());
                        break;
                    }
                    case "list": 
                    case "l": {
                        this.printBreakpoints();
                        break;
                    }
                    default: {
                        this.printlnErr("Invalid breakpoint command '%s'. Use one of 'add', 'remove', 'clear', 'skip', 'unskip', or 'list'.", cmd);
                        break;
                    }
                }
            }
            catch (ParseError ex) {
                this.printer.println("error", ex.getMessage());
            }
        }
    }

    private void handleCallstackCmd(List<String> params) {
        if (!this.agent.hasActiveBreak()) {
            this.println("Not in a debug break!", new Object[0]);
            return;
        }
        if (params.isEmpty()) {
            this.printCallstack();
        } else {
            try {
                String cmd;
                switch (cmd = StringUtil.trimToEmpty(params.get(0))) {
                    case "list": {
                        this.printCallstack();
                        break;
                    }
                    case "select": 
                    case "s": {
                        int level;
                        List<CallFrame> frames = this.getCallFrames(this.agent.getActiveBreak());
                        this.currCallFrameLevel = level = this.parseCallStackLevel(params.get(1), frames.size());
                        this.currCallFrame = frames.get(level - 1);
                        this.println("Selected call frame -> [%d/%d]: %s", level, frames.size(), this.currCallFrame);
                        break;
                    }
                    case "up": {
                        List<CallFrame> frames = this.getCallFrames(this.agent.getActiveBreak());
                        this.currCallFrameLevel = this.limit(this.currCallFrameLevel + 1, 1, frames.size());
                        this.currCallFrame = frames.get(this.currCallFrameLevel - 1);
                        this.println("Selected call frame -> [%d/%d]: %s", this.currCallFrameLevel, frames.size(), this.currCallFrame);
                        break;
                    }
                    case "down": {
                        List<CallFrame> frames = this.getCallFrames(this.agent.getActiveBreak());
                        this.currCallFrameLevel = this.limit(this.currCallFrameLevel - 1, 1, frames.size());
                        this.currCallFrame = frames.get(this.currCallFrameLevel - 1);
                        this.println("Selected call frame -> [%d/%d]: %s", this.currCallFrameLevel, frames.size(), this.currCallFrame);
                        break;
                    }
                    case "dselect": 
                    case "d": {
                        this.println("Cleared call frame operations", new Object[0]);
                        this.currCallFrame = null;
                        break;
                    }
                    default: {
                        this.printlnErr("Invalid callstack command '%s'. Use one of 'list', 'select', 'up', 'down', or 'deselect'.", cmd);
                        break;
                    }
                }
            }
            catch (ParseError ex) {
                this.printer.println("error", ex.getMessage());
            }
            catch (RuntimeException ex) {
                this.printer.println("error", ex.getMessage());
            }
        }
    }

    private void switchBreak(String sIndex) {
        int breakCount = this.agent.getAllBreaks().size();
        if (breakCount == 0) {
            this.printlnErr("No breaks available!", new Object[0]);
        } else {
            int index = this.parseBreakIndex(sIndex);
            if (index < 1 || index > breakCount) {
                this.printlnErr("Invalid break index %d. Must be in the range [1..%d].", index, breakCount);
            } else {
                Break br = this.agent.switchActiveBreak(index);
                if (br != null) {
                    this.println("Active break -> %s", br.getBreakFnInfo(false));
                } else {
                    this.printlnErr("Failed switching active break!", new Object[0]);
                }
            }
        }
    }

    private void isBreak() {
        this.println(this.agent.hasActiveBreak() ? this.formatStop(this.agent.getActiveBreak()) : "Not in a debug break!", new Object[0]);
    }

    private void printCallstack() {
        Break br = this.agent.getActiveBreak();
        this.println(this.formatBreakOverview(br), new Object[0]);
        this.println();
        this.println("Callstack:", new Object[0]);
        this.println(this.formatCallstack(br.getCallStack(), true), new Object[0]);
    }

    private void printBreakList() {
        List<Break> breaks = this.agent.getAllBreaks();
        AtomicLong idx = new AtomicLong(1L);
        if (breaks.isEmpty()) {
            this.println("No breaks available!", new Object[0]);
        } else {
            this.println("Breaks", new Object[0]);
            breaks.forEach(b -> this.println("  [%d]: %s", idx.getAndIncrement(), b.getBreakFnInfo(false)));
        }
    }

    private void printBreakpoints() {
        if (this.agent.getBreakpoints().isEmpty()) {
            this.printer.println("stdout", "No breakpoints defined!");
        } else {
            boolean skip = this.agent.isSkipBreakpoints();
            this.agent.getBreakpoints().stream().forEach(b -> b.getSelectors().forEach(s -> this.printer.println("stdout", String.format("  %s%s", skip ? "[-] " : "", s.formatForBaseFn(b.getQualifiedFnName(), true)))));
        }
    }

    private void params(List<String> params) {
        if (!this.agent.hasActiveBreak()) {
            this.println("Not in a debug break!", new Object[0]);
            return;
        }
        Break br = this.agent.getActiveBreak();
        if (this.currCallFrame == null) {
            this.println(this.formatBreakOverview(br), new Object[0]);
            if (br.getBreakpoint().getQualifiedName().equals("if")) {
                this.println(this.renderIfSpecialFormParams(br), new Object[0]);
            } else if (br.isBreakInNativeFn()) {
                this.println(this.renderNativeFnParams(br), new Object[0]);
            } else {
                VncVector spec = br.getFn().getParams();
                boolean plainSymbolParams = Destructuring.isFnParamsWithoutDestructuring(spec);
                if (plainSymbolParams) {
                    this.println(this.renderFnNoDestructuring(br), new Object[0]);
                } else {
                    this.println(this.renderFnDestructuring(br), new Object[0]);
                }
            }
        } else {
            this.println(this.renderCallFrameParams(this.currCallFrame), new Object[0]);
        }
    }

    private void locals(String sLevel) {
        if (!this.agent.hasActiveBreak()) {
            this.println("Not in a debug break!", new Object[0]);
            return;
        }
        Break br = this.agent.getActiveBreak();
        this.println(this.formatBreakOverview(br), new Object[0]);
        if (this.currCallFrame == null) {
            Env env = this.agent.getActiveBreak().getEnv();
            int maxLevel = env.level() + 1;
            int level = this.parseEnvLevel(sLevel, 1, maxLevel);
            this.println("Local vars at breakpoint env level %d/%d:\n%s", level, maxLevel, this.renderLocalCars(env, level));
        } else {
            Env env = this.currCallFrame.getEnv();
            int maxLevel = env.level() + 1;
            int level = this.parseEnvLevel(sLevel, 1, maxLevel);
            this.println("Local vars at env level %d/%d of call frame (%d) of %s:\n%s", level, maxLevel, this.currCallFrameLevel, this.currCallFrame, this.renderLocalCars(env, level));
        }
    }

    private void retval() {
        if (!this.agent.hasActiveBreak()) {
            this.println("Not in a debug break!", new Object[0]);
            return;
        }
        Break br = this.agent.getActiveBreak();
        this.println(this.formatBreakOverview(br), new Object[0]);
        VncVal v = br.getRetVal();
        if (v == null) {
            this.println("Return value: <not available>", new Object[0]);
        } else {
            this.println("Return value: %s", this.renderValue(v, 100));
        }
    }

    private void ex() {
        if (!this.agent.hasActiveBreak()) {
            this.println("Not in a debug break!", new Object[0]);
            return;
        }
        Break br = this.agent.getActiveBreak();
        this.println(this.formatBreakOverview(br), new Object[0]);
        Exception e = br.getException();
        if (e == null) {
            this.println("exception: <not available>", new Object[0]);
        } else {
            this.printer.printex("debug", e);
        }
    }

    private void breakpointListener(Break b) {
        this.clearCurrCallFrame();
        this.printer.println("debug", this.formatStop(b));
        this.replThread.interrupt();
    }

    private String renderIfSpecialFormParams(Break br) {
        StringBuilder sb = new StringBuilder();
        sb.append("Arguments passed to if:");
        sb.append("\n");
        sb.append(this.formatVar(0, br.getArgs().first()));
        if (br.getRetVal() != null) {
            sb.append('\n');
            sb.append(this.formatReturnVal(br.getRetVal()));
        }
        return sb.toString();
    }

    private String renderCallFrameParams(CallFrame frame) {
        VncList args = frame.getArgs();
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("Arguments passed to the call frame %s:", frame));
        sb.append(this.renderIndexedParams(args));
        return sb.toString();
    }

    private String renderNativeFnParams(Break br) {
        VncFunction fn = br.getFn();
        VncList args = br.getArgs();
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("Arguments passed to native function %s:", fn.getQualifiedName()));
        sb.append(this.renderIndexedParams(args));
        if (br.getRetVal() != null) {
            sb.append('\n');
            sb.append(this.formatReturnVal(br.getRetVal()));
        }
        return sb.toString();
    }

    private String renderFnNoDestructuring(Break br) {
        VncFunction fn = br.getFn();
        VncVector spec = fn.getParams();
        VncList args = br.getArgs();
        StringBuilder sb = new StringBuilder();
        VncVector spec_ = spec;
        VncList args_ = args;
        sb.append(String.format("Arguments passed to %s %s:", br.isBreakInSpecialForm() ? "special form" : (fn.isMacro() ? "macro" : "function"), fn.getQualifiedName()));
        do {
            sb.append("\n");
            sb.append(this.formatVar(spec_.first(), args_.first()));
            spec_ = spec_.rest();
            args_ = args_.rest();
        } while (!spec_.isEmpty());
        if (!args_.isEmpty()) {
            sb.append(String.format("\n... %d more arguments not matching a parameter", args_.size()));
        }
        if (br.getRetVal() != null) {
            sb.append('\n');
            sb.append(this.formatReturnVal(br.getRetVal()));
        }
        return sb.toString();
    }

    private String renderFnDestructuring(Break br) {
        VncFunction fn = br.getFn();
        VncVector spec = fn.getParams();
        VncList args = br.getArgs();
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("Arguments passed to %s %s (destructured):", br.isBreakInSpecialForm() ? "special form" : (fn.isMacro() ? "macro" : "function"), fn.getQualifiedName()));
        List<Var> vars = Destructuring.destructure(spec, args);
        vars.forEach(v -> sb.append("\n" + this.formatVar((Var)v)));
        if (br.getRetVal() != null) {
            sb.append("\n" + this.formatReturnVal(br.getRetVal()));
        }
        return sb.toString();
    }

    private String renderIndexedParams(VncList args) {
        StringBuilder sb = new StringBuilder();
        VncList args_ = args.take(10);
        int ii = 0;
        while (!args_.isEmpty()) {
            sb.append("\n" + this.formatVar(ii++, args_.first()));
            args_ = args_.rest();
        }
        args_ = args.drop(10);
        if (!args_.isEmpty()) {
            sb.append(String.format("\n... %d more arguments not displayed", args_.size()));
        }
        return sb.toString();
    }

    private String renderLocalCars(Env env, int level) {
        List<Var> vars = env.getLocalVars(level - 1);
        return vars == null || vars.isEmpty() ? String.format("   <no local vars at env level %d>", level) : vars.stream().map(v -> this.formatVar((Var)v)).collect(Collectors.joining("\n"));
    }

    private String formatVar(Var v) {
        return this.formatVar(v.getName(), v.getVal());
    }

    private String formatVar(VncVal name, VncVal value) {
        String sval = StringUtil.truncate(value.toString(true), 100, "...");
        String sname = name.toString(true);
        return String.format("%s -> %s", sname, sval);
    }

    private String formatVar(int index, VncVal value) {
        String sval = StringUtil.truncate(value.toString(true), 100, "...");
        return String.format("[%d] -> %s", index, sval);
    }

    private String formatReturnVal(VncVal retval) {
        String sval = StringUtil.truncate(retval.toString(true), 100, "...");
        return String.format("[return] -> %s", sval);
    }

    private String formatBreakOverview(Break br) {
        VncFunction fn = br.getFn();
        return String.format("Break in %s %s at %s level.", br.isBreakInSpecialForm() ? "special form" : (fn.isMacro() ? "macro" : "function"), fn.getQualifiedName(), br.getBreakpointScope().description());
    }

    private String formatCallstack(CallStack cs, boolean showLevel) {
        if (showLevel) {
            int digits = cs.isEmpty() ? 1 : (int)Math.floor(Math.log10(cs.size())) + 1;
            String format = "%s%" + digits + "d: %s";
            boolean printMarker = this.currCallFrame != null;
            AtomicLong idx = new AtomicLong(1L);
            return cs.toList().stream().map(f -> {
                Object[] objectArray = new Object[3];
                objectArray[0] = printMarker ? (idx.get() == (long)this.currCallFrameLevel ? "* " : "  ") : "";
                objectArray[1] = idx.getAndIncrement();
                objectArray[2] = f.toString();
                return String.format(format, objectArray);
            }).collect(Collectors.joining("\n"));
        }
        return cs.toList().stream().collect(Collectors.joining("\n"));
    }

    private String formatStop(Break br) {
        VncFunction fn = br.getFn();
        return String.format("Stopped in %s %s%s at %s level.", br.isBreakInSpecialForm() ? "special form" : (fn.isMacro() ? "macro" : "function"), fn.getQualifiedName(), fn.isNative() ? "" : " (" + new CallFrame(fn).getSourcePosInfo() + ")", br.getBreakpointScope().description());
    }

    private String renderValue(VncVal val, int maxLen) {
        String sVal = StringUtil.truncate(val.toString(true), maxLen, "...");
        String type = Types.getType(val).toString();
        if (val instanceof VncCollection) {
            int size = ((VncCollection)val).size();
            return String.format("%s [%d]: %s", type, size, sVal);
        }
        return sVal;
    }

    private int parseCallStackLevel(String level, int max) {
        int lvl = this.parseCallStackLevel(level);
        if (lvl < 1 || lvl > max) {
            throw new RuntimeException(String.format("Invalid callstack level '%d'. Must be a in the range [1..%d].", lvl, max));
        }
        return lvl;
    }

    private int parseCallStackLevel(String level) {
        try {
            return Integer.parseInt(level);
        }
        catch (Exception ex) {
            throw new RuntimeException(String.format("Invalid callstack level '%s'. Must be a number.", level));
        }
    }

    private int parseEnvLevel(String sLevel, int min, int max) {
        try {
            int level = sLevel == null ? 1 : Integer.parseInt(sLevel);
            return this.limit(level, min, max);
        }
        catch (Exception ex) {
            throw new RuntimeException(String.format("Invalid env level '%s'. Must be a number.", sLevel));
        }
    }

    private int parseBreakIndex(String sIndex) {
        try {
            return Integer.parseInt(sIndex);
        }
        catch (Exception ex) {
            throw new RuntimeException(String.format("Invalid env level '%s'. Must be a number.", sIndex));
        }
    }

    private void println() {
        this.printer.println("debug", "");
    }

    private void println(String format, Object ... args) {
        this.printer.println("debug", String.format(format, args));
    }

    private void printlnErr(String format, Object ... args) {
        this.printer.println("error", String.format(format, args));
    }

    private void printlnErr(StepValidity validity) {
        if (!validity.isValid()) {
            this.printer.println("error", validity.getErrMsg());
        }
    }

    private void clearCurrCallFrame() {
        this.currCallFrame = null;
        this.currCallFrameLevel = 0;
    }

    private List<CallFrame> getCallFrames(Break br) {
        return br.getCallStack().callstack();
    }

    private int limit(int val, int min, int max) {
        return Math.max(Math.min(val, max), min);
    }
}

