/*   __    __         _
 *   \ \  / /__ _ __ (_) ___ ___
 *    \ \/ / _ \ '_ \| |/ __/ _ \
 *     \  /  __/ | | | | (_|  __/
 *      \/ \___|_| |_|_|\___\___|
 *
 *
 * Copyright 2017-2024 Venice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.jlangch.venice.impl.util.callstack;

import java.util.function.Supplier;

import com.github.jlangch.venice.impl.thread.ThreadContext;


/**
 * Run code with an explicit Venice call stack
 *
 * <pre>
 *    try (WithCallStack cs = new WithCallStack(CallFrame.fromVal("rest", ast))) {
 *       return ast.rest();
 *    }
 * </pre>
 *
 * <pre>
 *    WithCallStack.run(
 *       CallFrame.fromVal("rest", ast)));
 *       () -&gt; return ast.rest();
 *    }
 * </pre>
 */
public class WithCallStack implements AutoCloseable {

    public WithCallStack(final CallFrame callFrame) {
        if (callFrame == null) {
            throw new IllegalArgumentException("A 'callFrame' must not be null");
        }

        ThreadContext.getCallStack().push(callFrame);
    }

    @Override
    public void close() {
        ThreadContext.getCallStack().pop();
    }


    public static <T> T call(final CallFrame callFrame, final Supplier<T> func) {
        try (WithCallStack cs = new WithCallStack(callFrame)) {
            return func.get();
        }
    }

    public static void run(final CallFrame callFrame, final Runnable func) {
        try (WithCallStack cs = new WithCallStack(callFrame)) {
            func.run();
        }
    }

}
