/*   __    __         _
 *   \ \  / /__ _ __ (_) ___ ___ 
 *    \ \/ / _ \ '_ \| |/ __/ _ \
 *     \  /  __/ | | | | (_|  __/
 *      \/ \___|_| |_|_|\___\___|
 *
 *
 * Copyright 2017-2020 Venice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.jlangch.venice.impl.types;

import java.util.Arrays;
import java.util.HashMap;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.github.jlangch.venice.impl.MetaUtil;
import com.github.jlangch.venice.impl.types.collections.VncHashMap;
import com.github.jlangch.venice.impl.types.collections.VncList;
import com.github.jlangch.venice.impl.types.collections.VncTinyList;
import com.github.jlangch.venice.impl.types.collections.VncVector;
import com.github.jlangch.venice.impl.types.util.Types;
import com.github.jlangch.venice.impl.util.StringUtil;


public abstract class VncFunction extends VncVal implements IVncFunction {

	public VncFunction(final String name) {
		this(name, null, null);
	}
	
	public VncFunction(final String name, final VncVal meta) {
		this(name, null, meta);
	}

	public VncFunction(final String name, final VncVector params) {
		this(name, params, Constants.Nil);
	}

	public VncFunction(final String name, final VncVector params, final VncVal meta) {
		super(Constants.Nil);
		
		final String ns = getNamespace(name);

		this.simpleName = getSimpleName(name);
		this.params = params;
		this.qualifiedName = "core".equals(ns) ? simpleName : ns + "/" + simpleName;
		
		this.fnMeta.set(MetaUtil.setNamespace(meta, ns));
		this._private = MetaUtil.isPrivate(meta);
		this.ns = ns;
		
		this.fixedArgsCount = params == null ? 0 : countFixedArgs(params);
		this.variadicArgs = params == null ? false : hasRemaingsArgs(params);
	}
	
	@Override
	public VncFunction withMeta(final VncVal meta) {
		this.fnMeta.set(meta);
		this._private = MetaUtil.isPrivate(meta);
		return this;
	}

	@Override
	public abstract VncVal apply(VncList args);

	public void setNamespace(final String ns) { 
		this.fnMeta.set(MetaUtil.setNamespace(fnMeta.get(), ns));
		this.ns = ns;
	}

	public boolean isRedefinable() { 
		return true; 
	}
	
	public VncVector getParams() { 
		return params; 
	}
	
	public boolean isMacro() { 
		return macro; 
	}
	
	public void setMacro() { 
		macro = true; 
	}
	
	public String getSimpleName() { 
		return simpleName; 
	}
	
	public String getQualifiedName() { 
		return qualifiedName; 
	}

	public VncList getArgLists() { 
		return (VncList)getMetaVal(MetaUtil.ARGLIST, VncTinyList.empty());
	}
	
	public VncVal getDoc() { 
		return getMetaVal(MetaUtil.DOC); 
	}
	
	public VncList getExamples() { 
		return (VncList)getMetaVal(MetaUtil.EXAMPLES, VncTinyList.empty());
	}
	
	public int getFixedArgsCount() {
		return fixedArgsCount;
	}
	
	public boolean hasVariadicArgs() {
		return variadicArgs;
	}

	public VncVal getBody() { 
		return Constants.Nil;
	}

	@Override
	public VncVal getMeta() { 
		return fnMeta.get(); 
	}
	
	@Override
	public boolean isPrivate() {
		return _private;
	}
	
	public String getNamespace() {
		return ns;
	}
	
	@Override 
	public int typeRank() {
		return 100;
	}

	@Override
	public Object convertToJavaObject() {
		return null;
	}

	@Override 
	public String toString() {
		return String.format(
				"%s %s %s", 
				isMacro() ? "macro" : "function", 
				getQualifiedName(),
				new StringBuilder()
					.append("{")
					.append("visibility ")
					.append(isPrivate() ? ":private" : ":public")
					.append(", ns ")
					.append(StringUtil.quote(ns == null ? "" : ns, '\"'))
					.append("}"));
	}

	
	public static String createAnonymousFuncName() {
		return createAnonymousFuncName(null);
	}

	public static String createAnonymousFuncName(final String name) {
		return StringUtil.isEmpty(name)
				? "anonymous-" + UUID.randomUUID().toString()
				: "anonymous-" + name + "-" + UUID.randomUUID().toString();
	}

	private static String getNamespace(final String qualifiedName) {
		final int pos = qualifiedName.indexOf("/");
		return pos < 1 ? "core" : qualifiedName.substring(0, pos);
	}

	private static String getSimpleName(final String qualifiedName) {
		final int pos = qualifiedName.indexOf("/");
		return pos < 1 ? qualifiedName : qualifiedName.substring(pos+1);
	}

	private static int countFixedArgs(final VncVector params) {
		int fixedArgs = 0;
		
		for(VncVal p : params.getList()) {
			if (isElisionSymbol(p)) break;
			fixedArgs++;
		}
		
		return fixedArgs;
	}

	private static boolean hasRemaingsArgs(final VncVector params) {
		for(VncVal p : params.getList()) {
			if (isElisionSymbol(p)) return true;
		}
		return false;
	}

	private static boolean isElisionSymbol(final VncVal val) {
		return Types.isVncSymbol(val) && ((VncSymbol)val).getName().equals("&");
	}
	
	
	
	public static MetaBuilder meta() {
		return new MetaBuilder();
	}
	
	
	public static class MetaBuilder  {

		public MetaBuilder() {
		}
		
		public MetaBuilder arglists(final String... arglists) {
			meta.put(
				MetaUtil.ARGLIST, 
				new VncList(Arrays.stream(arglists).map(s -> new VncString(s)).collect(Collectors.toList())));
			return this;
		}
		
		public MetaBuilder doc(final String doc) { 
			meta.put(MetaUtil.DOC, new VncString(doc));
			return this;
		}
		
		public MetaBuilder examples(final String... examples) { 
			meta.put(
				MetaUtil.EXAMPLES, 
				new VncList(Arrays.stream(examples).map(s -> new VncString(s)).collect(Collectors.toList())));
			return this;
		}
			
		public VncHashMap build() {
			return new VncHashMap(meta);
		}

		private final HashMap<VncVal,VncVal> meta = new HashMap<>();
	}
	

    private static final long serialVersionUID = -1848883965231344442L;

	private final VncVector params;
	private volatile boolean macro = false;
	private final String simpleName;
	private final String qualifiedName;
	private final int fixedArgsCount;
	private final boolean variadicArgs;
	
	// Functions handle its meta data locally (functions cannot be copied)
	private final AtomicReference<VncVal> fnMeta = new AtomicReference<>(Constants.Nil);
	private volatile boolean _private;
	private volatile String ns;
}