/*   __    __         _
 *   \ \  / /__ _ __ (_) ___ ___ 
 *    \ \/ / _ \ '_ \| |/ __/ _ \
 *     \  /  __/ | | | | (_|  __/
 *      \/ \___|_| |_|_|\___\___|
 *
 *
 * Copyright 2017-2020 Venice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.jlangch.venice.impl.types.collections;

import static com.github.jlangch.venice.impl.types.Constants.False;
import static com.github.jlangch.venice.impl.types.Constants.True;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

import com.github.jlangch.venice.VncException;
import com.github.jlangch.venice.impl.Printer;
import com.github.jlangch.venice.impl.types.Constants;
import com.github.jlangch.venice.impl.types.VncSymbol;
import com.github.jlangch.venice.impl.types.VncVal;
import com.github.jlangch.venice.impl.types.util.Types;
import com.github.jlangch.venice.impl.util.ErrorMessage;


public class VncSortedMap extends VncMap {

	public VncSortedMap() {
		this((io.vavr.collection.TreeMap<VncVal,VncVal>)null, null);
	}

	public VncSortedMap(final VncVal meta) {
		this((io.vavr.collection.TreeMap<VncVal,VncVal>)null, meta);
	}

	public VncSortedMap(final io.vavr.collection.Map<VncVal,VncVal> val) {
		this(val, null);
	}

	public VncSortedMap(final Map<VncVal,VncVal> vals) {
		this(vals, null);
	}

	public VncSortedMap(final Map<VncVal,VncVal> vals, final VncVal meta) {
		this(vals == null ? null : io.vavr.collection.TreeMap.ofAll(vals), meta);
	}

	public VncSortedMap(final io.vavr.collection.Map<VncVal,VncVal> val, final VncVal meta) {
		super(meta == null ? Constants.Nil : meta);
		if (val == null) {
			value = io.vavr.collection.TreeMap.empty();
		}
		else if (val instanceof io.vavr.collection.TreeMap) {
			value = (io.vavr.collection.TreeMap<VncVal,VncVal>)val;
		}
		else {
			value = io.vavr.collection.TreeMap.ofEntries(val);
		}
	}
	
	
	public static VncSortedMap ofAll(final VncSequence lst) {
		if (lst != null && (lst.size() %2 != 0)) {
			throw new VncException(String.format(
					"sorted-map: create requires an even number of list items. %s", 
					ErrorMessage.buildErrLocation(lst)));
		}

		return new VncSortedMap().assoc(lst);
	}
	
	public static VncSortedMap ofAll(final VncVector vec) {
		if (vec != null && (vec.size() %2 != 0)) {
			throw new VncException(String.format(
					"sorted-map: create requires an even number of vector items. %s", 
					ErrorMessage.buildErrLocation(vec)));
		}

		return new VncSortedMap().assoc(vec);
	}
	
	public static VncSortedMap of(final VncVal... mvs) {
		if (mvs != null && (mvs.length %2 != 0)) {
			throw new VncException(String.format(
					"sorted-map: create requires an even number of items. %s", 
					ErrorMessage.buildErrLocation(mvs[0])));
		}
		
		return new VncSortedMap().assoc(mvs);
	}
	

	@Override
	public VncSortedMap emptyWithMeta() {
		return new VncSortedMap(getMeta());
	}

	@Override
	public VncSortedMap withValues(final Map<VncVal,VncVal> replaceVals) {
		return new VncSortedMap(replaceVals, getMeta());
	}
	
	@Override
	public VncSortedMap withValues(
			final Map<VncVal,VncVal> replaceVals, 
			final VncVal meta
	) {
		return new VncSortedMap(replaceVals, meta);
	}

	@Override
	public VncSortedMap withMeta(final VncVal meta) {
		return new VncSortedMap(value, meta);
	}
	
	@Override
	public Map<VncVal,VncVal> getMap() {
		return Collections.unmodifiableMap(value.toJavaMap());
	}
	
	@Override
	public VncVal get(final VncVal key) {
		return value.get(key).getOrElse(Constants.Nil);
	}

	@Override
	public VncVal containsKey(final VncVal key) {
		return value.containsKey(key) ? True : False;
	}

	@Override
	public VncList keys() {
		return new VncList(new ArrayList<>(value.keySet().toJavaList()));
	}

	@Override
	public List<VncMapEntry> entries() {
		return Collections.unmodifiableList(
					value
						.map(e -> new VncMapEntry(e._1, e._2))
						.collect(Collectors.toList()));
	}

	@Override
	public VncSortedMap putAll(final VncMap map) {
		return new VncSortedMap(
						value.merge(io.vavr.collection.TreeMap.ofAll(map.getMap())),
						getMeta());
	}
	
	@Override
	public VncSortedMap assoc(final VncVal... mvs) {
		if (mvs.length %2 != 0) {
			throw new VncException(String.format(
					"sorted-map: assoc requires an even number of items. %s", 
					ErrorMessage.buildErrLocation(mvs[0])));
		}
		
		io.vavr.collection.TreeMap<VncVal,VncVal> tmp = value;
		for (int i=0; i<mvs.length; i+=2) {
			tmp = tmp.put(mvs[i], mvs[i+1]);
		}
		return new VncSortedMap(tmp, getMeta());
	}

	@Override
	public VncSortedMap assoc(final VncSequence mvs) {
		if (mvs.size() %2 != 0) {
			throw new VncException(String.format(
					"sorted-map: assoc requires an even number of items. %s", 
					ErrorMessage.buildErrLocation(mvs)));
		}	

		io.vavr.collection.TreeMap<VncVal,VncVal> tmp = value;
		for (int i=0; i<mvs.getList().size(); i+=2) {
			tmp = tmp.put(mvs.nth(i), mvs.nth(i+1));
		}
		return new VncSortedMap(tmp, getMeta());
	}

	@Override
	public VncSortedMap dissoc(final VncVal... keys) {
		return new VncSortedMap(
					value.removeAll(Arrays.asList(keys)),
					getMeta());
	}

	@Override
	public VncSortedMap dissoc(final VncSequence keys) {
		return new VncSortedMap(
					value.removeAll(keys.getList()),
					getMeta());
	}
	
	@Override
	public VncList toVncList() {
		return new VncList(
						value.map(e -> VncVector.of(e._1, e._2))
							 .collect(Collectors.toList()),
						getMeta());
	}
	
	@Override
	public VncVector toVncVector() {
		return new VncVector(
						value.map(e -> VncVector.of(e._1, e._2))
							 .collect(Collectors.toList()),
						getMeta());
	}
	
	@Override
	public int size() {
		return value.size();
	}
	
	@Override
	public boolean isEmpty() {
		return value.isEmpty();
	}
	
	@Override public int typeRank() {
		return 210;
	}
	
	@Override
	public int compareTo(final VncVal o) {
		if (o == Constants.Nil) {
			return 1;
		}
		else if (Types.isVncSortedMap(o)) {
			final Integer sizeThis = size();
			final Integer sizeOther = ((VncSortedMap)o).size();
			int c = sizeThis.compareTo(sizeOther);
			if (c != 0) {
				return c;
			}
			else {
				return equals(o) ? 0 : -1;
			}
		}

		return super.compareTo(o);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		VncSortedMap other = (VncSortedMap) obj;
		if (value == null) {
			if (other.value != null)
				return false;
		} else if (!value.equals(other.value))
			return false;
		return true;
	}

	@Override 
	public String toString() {
		return toString(true);
	}
	
	@Override
	public String toString(final boolean print_readably) {
		final List<VncVal> list = value
									.map(e -> VncList.of(e._1, e._2).getList())
									.collect(Collectors.toList())
									.stream()
									.flatMap(l -> l.stream())
									.collect(Collectors.toList());

		return "{" + Printer.join(list, " ", print_readably) + "}";
	}
	
	public static class Builder {
		public Builder() {
		}
		
		public Builder put(final String key, final VncVal val) {
			map.put(new VncSymbol(key), val);
			return this;
		}

		public Builder put(final VncVal key, final VncVal val) {
			map.put(key, val);
			return this;
		}

		public VncSortedMap build() {
			return new VncSortedMap(map);
		}
		
		public Map<VncVal,VncVal> toMap() {
			return map;
		}
		
		private TreeMap<VncVal,VncVal> map = new TreeMap<>();
	}
	

    private static final long serialVersionUID = -1848883965231344442L;

	private final io.vavr.collection.TreeMap<VncVal,VncVal> value;	
}