/*   __    __         _
 *   \ \  / /__ _ __ (_) ___ ___ 
 *    \ \/ / _ \ '_ \| |/ __/ _ \
 *     \  /  __/ | | | | (_|  __/
 *      \/ \___|_| |_|_|\___\___|
 *
 *
 * Copyright 2017-2020 Venice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.jlangch.venice.impl.specialforms;

import java.util.List;
import java.util.stream.Collectors;

import com.github.jlangch.venice.VncException;
import com.github.jlangch.venice.impl.ModuleLoader;
import com.github.jlangch.venice.impl.SpecialForms;
import com.github.jlangch.venice.impl.ansi.AnsiColorTheme;
import com.github.jlangch.venice.impl.ansi.AnsiColorThemes;
import com.github.jlangch.venice.impl.env.Env;
import com.github.jlangch.venice.impl.functions.CoreFunctions;
import com.github.jlangch.venice.impl.reader.HighlightItem;
import com.github.jlangch.venice.impl.reader.HighlightParser;
import com.github.jlangch.venice.impl.types.VncKeyword;
import com.github.jlangch.venice.impl.types.VncString;
import com.github.jlangch.venice.impl.types.VncSymbol;
import com.github.jlangch.venice.impl.types.VncVal;
import com.github.jlangch.venice.impl.types.collections.VncList;
import com.github.jlangch.venice.impl.types.collections.VncSet;
import com.github.jlangch.venice.impl.types.custom.VncChoiceTypeDef;
import com.github.jlangch.venice.impl.types.custom.VncCustomTypeDef;
import com.github.jlangch.venice.impl.types.custom.VncWrappingTypeDef;
import com.github.jlangch.venice.impl.types.util.Types;
import com.github.jlangch.venice.impl.util.Doc;


public class DocForm {

	public static VncString doc(final VncVal ref, final Env env) {			
		if (Types.isVncSymbol(ref)) {
			return docForSymbol((VncSymbol)ref, env);
		}
		else if (Types.isVncKeyword(ref)) {
			return docForKeyword((VncKeyword)ref, env);
		}
		else if (Types.isVncString(ref)) {
			return docForSymbol(((VncString)ref).toSymbol(), env);
		}
		else {
			// last resort
			final VncString name = (VncString)CoreFunctions.name.apply(VncList.of(ref));
			return docForSymbol(name.toSymbol(), env);
		}
	}
	
	private static VncString docForSymbol(final VncSymbol sym, final Env env) {
		VncVal docVal = SpecialForms.ns.get(sym); // special form?
		if (docVal == null) {
			docVal = env.get(sym); // var?
		}
		
		return Doc.getDoc(docVal);
	}

	private static VncString docForKeyword(final VncKeyword keyword, final Env env) {
		if (ModuleLoader.isValidModule(keyword)) {
			return docForModule(keyword, env);
		}
		else {
			return docForCustomType(keyword, env);
		}
	}

	private static VncString docForModule(
			final VncKeyword module, 
			final Env env
	) {
		final AnsiColorTheme theme = AnsiColorThemes.getTheme(getColorTheme(env));

		final String script = ModuleLoader.loadModule(module.getValue());
		
		if (theme == null) {
			return new VncString(script);
		}
		else {			
			final List<HighlightItem> items = HighlightParser
												.parse("(do " + script + ")");
			
			return new VncString(
					AnsiColorTheme.ANSI_RESET +
					items.subList(3, items.size()-2)
						 .stream()
						 .map(it -> theme.style(it.getForm(), it.getClazz()))
						 .collect(Collectors.joining()));
		}
	}
	
	private static VncString docForCustomType(
			final VncKeyword type, 
			final Env env
	) {
		final VncVal tdef = env.getGlobalOrNull(type.toSymbol());

		if (tdef == null) {
			if (type.hasNamespace()) {
				throw new VncException(String.format(
						":%s is not a custom type. No documentation available!",
						type.getValue()));
			}
			else {
				throw new VncException(String.format(
						":%s is not a custom type. Please qualify the type with its namespace!",
						type.getValue()));
			}
		}
		else if (tdef instanceof VncCustomTypeDef) {
			final VncCustomTypeDef typeDef = (VncCustomTypeDef)tdef;
			final StringBuilder sb = new StringBuilder();
			
			sb.append(String.format("Custom type :%s\n", type.getValue()));
			sb.append("Fields: \n");
			typeDef.getFieldDefs().forEach(f -> sb.append(String.format(
																"   %s :%s\n", 
																f.getName().getValue(),
																f.getType().getValue())));
			if (typeDef.getValidationFn() != null) {
				sb.append(String.format("Validation function: :%s\n", typeDef.getValidationFn().getQualifiedName()));
			}
			
			return new VncString(sb.toString());
		}
		else if (tdef instanceof VncWrappingTypeDef) {
			final VncWrappingTypeDef typeDef = (VncWrappingTypeDef)tdef;
			final StringBuilder sb = new StringBuilder();
			
			sb.append(String.format("Custom wrapped type :%s\n", type.getValue()));
			sb.append(String.format("Base type :%s\n", typeDef.getBaseType().getValue()));
			if (typeDef.getValidationFn() != null) {
				sb.append(String.format("Validation function: :%s\n", typeDef.getValidationFn().getQualifiedName()));
			}
			
			return new VncString(sb.toString());
		}
		else if (tdef instanceof VncChoiceTypeDef) {
			final VncChoiceTypeDef typeDef = (VncChoiceTypeDef)tdef;
			
			final VncSet types = typeDef.typesOnly();
			final VncSet values = typeDef.valuesOnly();
			
			final StringBuilder sb = new StringBuilder();
			sb.append(String.format("Custom choice type :%s\n", type.getValue()));
			if (!types.isEmpty()) {
				sb.append("Types: \n");
				typeDef.typesOnly().getList().forEach(v -> sb.append(String.format("   %s\n", v.toString())));
			}
			if (!values.isEmpty()) {
				sb.append("Values: \n");
				typeDef.valuesOnly().getList().forEach(v -> sb.append(String.format("   %s\n", v.toString())));
			}
			
			return new VncString(sb.toString());
		}
		else {
			throw new VncException(String.format(
					":%s is not a custom type. Please qualify the type with its namespace!",
					type.getValue()));
		}
	}
	
	private static String getColorTheme(final Env env) {
		// Note: there is a color theme only if we're running in a REPL!
		
		final VncVal runMode = env.get(new VncSymbol("*run-mode*"));
		if (Types.isVncKeyword(runMode)) {
			final String sRunMode = ((VncKeyword)runMode).getValue();
			if ("repl".equals(sRunMode)) {
				final VncVal theme = env.get(new VncSymbol("*repl-color-theme*"));
				if (Types.isVncKeyword(theme)) {
					final String sTheme = ((VncKeyword)theme).getValue();
					return "none".equals(sTheme) ? null : sTheme;
				}
			}
		}
				
		return null;
	}
}
