/*
 * Copyright 2019 Kroto+ Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.github.marcoferrer.krotoplus.generators

import com.github.marcoferrer.krotoplus.config.CompilerConfig
import com.github.marcoferrer.krotoplus.script.ScriptManager
import com.google.protobuf.compiler.PluginProtos
import com.squareup.kotlinpoet.FileSpec
import java.io.File
import kotlin.script.templates.standard.ScriptTemplateWithArgs

interface Generator : () -> PluginProtos.CodeGeneratorResponse {

    val context: GeneratorContext
        get() = contextInstance

    val isEnabled: Boolean
        get() = false

    fun FileSpec.toResponseFileProto(): PluginProtos.CodeGeneratorResponse.File =
        PluginProtos.CodeGeneratorResponse.File.newBuilder()
            .apply {
                val basePath = packageName.replace(".", "/")
                val fileName = "${this@toResponseFileProto.name}.kt"

                name = if (basePath.isNotEmpty())
                    "$basePath/$fileName" else fileName

                content = this@toResponseFileProto.toString()
            }
            .build()

    companion object {
        const val AutoGenerationDisclaimer =
            "THIS IS AN AUTOGENERATED FILE. DO NOT EDIT THIS FILE DIRECTLY."
    }
}

val ScriptTemplateWithArgs.context: GeneratorContext
    get() = contextInstance

val ScriptManager.context: GeneratorContext
    get() = contextInstance

fun initializeContext(compilerConfig: CompilerConfig? = null) {
    compilerConfigOverride = compilerConfig
    contextInstance
}

private var compilerConfigOverride: CompilerConfig? = null

private val contextInstance by lazy {
    val inputStream = System.getProperty("krotoplus.debug.request.src")
        ?.let { File(it).inputStream() }
        ?: System.`in`

    val protoRequest = PluginProtos.CodeGeneratorRequest.parseFrom(inputStream)

    compilerConfigOverride
        ?.let { GeneratorContext(protoRequest, config = it) }
        ?: GeneratorContext(protoRequest)
}


