/*
 * #%L
 * Native ARchive plugin for Maven
 * %%
 * Copyright (C) 2002 - 2014 NAR Maven Plugin developers.
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package com.github.maven_nar.cpptasks;

import java.io.File;
import java.util.Enumeration;
import java.util.Vector;

import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.DataType;

import com.github.maven_nar.cpptasks.types.ConditionalFileSet;

/**
 * An element that specifies a prototype file and rules for source files that
 * should not use precompiled headers
 *
 * @author Curt Arnold
 */
public final class PrecompileDef extends DataType {
  private final Vector exceptSets = new Vector();
  private String ifCond;
  /**
   * Directory of prototype file
   */
  private File prototype = new File("stdafx.cpp");
  private String unlessCond;

  /**
   * Constructor
   * 
   */
  public PrecompileDef() {
  }

  /**
   * Method used by PrecompileExceptDef to add exception set to
   * PrecompileDef.
   */
  public void appendExceptFileSet(final ConditionalFileSet exceptSet) {
    exceptSet.setProject(getProject());
    this.exceptSets.addElement(exceptSet);
  }

  /**
   * Adds filesets that specify files that should not be processed with
   * precompiled headers enabled.
   * 
   */
  public PrecompileExceptDef createExcept() {
    return new PrecompileExceptDef(this);
  }

  public void execute() throws org.apache.tools.ant.BuildException {
    throw new org.apache.tools.ant.BuildException("Not an actual task, but looks like one for documentation purposes");
  }

  public String[] getExceptFiles() {
    final PrecompileDef ref = getRef();
    if (ref != null) {
      return ref.getExceptFiles();
    }
    if (this.exceptSets.size() == 0) {
      return new String[0];
    }
    final Project p = getProject();
    String[] exceptFiles = null;
    final Enumeration setEnum = this.exceptSets.elements();
    while (setEnum.hasMoreElements()) {
      final ConditionalFileSet exceptSet = (ConditionalFileSet) setEnum.nextElement();
      if (exceptSet.isActive()) {
        final DirectoryScanner scanner = exceptSet.getDirectoryScanner(p);
        final String[] scannerFiles = scanner.getIncludedFiles();
        if (exceptFiles == null) {
          exceptFiles = scannerFiles;
        } else {
          if (scannerFiles.length > 0) {
            final String[] newFiles = new String[exceptFiles.length + scannerFiles.length];
            System.arraycopy(exceptFiles, 0, newFiles, 0, exceptFiles.length);
            int index = exceptFiles.length;
            for (final String scannerFile : scannerFiles) {
              newFiles[index++] = scannerFile;
            }
            exceptFiles = newFiles;
          }
        }
      }
    }
    if (exceptFiles == null) {
      exceptFiles = new String[0];
    }
    return exceptFiles;
  }

  /**
   * Gets prototype source file
   * 
   */
  public File getPrototype() {
    final PrecompileDef ref = getRef();
    if (ref != null) {
      return ref.getPrototype();
    }
    return this.prototype;
  }

  private PrecompileDef getRef() {
    if (isReference()) {
      return (PrecompileDef) getCheckedRef(PrecompileDef.class, "PrecompileDef");
    }
    return null;
  }

  public boolean isActive() {
    final boolean isActive = CUtil.isActive(getProject(), this.ifCond, this.unlessCond);
    if (!isActive) {
      final PrecompileDef ref = getRef();
      if (ref != null) {
        return ref.isActive();
      }
    }
    return isActive;
  }

  /**
   * Sets a description of the current data type.
   */
  @Override
  public void setDescription(final String desc) {
    super.setDescription(desc);
  }

  /**
   * Sets an id that can be used to reference this element.
   * 
   * @param id
   *          id
   */
  public void setId(final String id) {
    //
    // this is actually accomplished by a different
    // mechanism, but we can document it
    //
  }

  /**
   * Set the 'if' condition.
   * 
   * The processor will be ignored unless the property is defined.
   * 
   * The value of property is insignificant, but values that would imply
   * misinterpretation ("false", "no") will throw an exception when
   * isActive() is evaluated.
   * 
   * @param propName
   *          name of property
   */
  public void setIf(final String propName) {
    this.ifCond = propName;
  }

  /**
   * Sets file to precompile.
   * 
   * Should be a source file that includes only one unguarded header file.
   * Default value is "stdafx.cpp".
   * 
   * @param prototype
   *          file path for prototype source file
   */
  public void setPrototype(final File prototype) {
    if (isReference()) {
      throw tooManyAttributes();
    }
    if (prototype == null) {
      throw new NullPointerException("prototype");
    }
    this.prototype = prototype;
  }

  /**
   * Specifies that this element should behave as if the content of the
   * element with the matching id attribute was inserted at this location.
   * 
   * @param ref
   *          Reference to other element
   * 
   */
  @Override
  public void setRefid(final org.apache.tools.ant.types.Reference ref) {
    super.setRefid(ref);
  }

  /**
   * Set the 'unless' condition. If named property exists at execution time,
   * the processor will be ignored.
   * 
   * Value of property is insignificant, but values that would imply
   * misinterpretation ("false", "no") of the behavior will throw an
   * exception when isActive is called.
   * 
   * @param propName
   *          name of property
   */
  public void setUnless(final String propName) {
    this.unlessCond = propName;
  }
}
