/*
 * Decompiled with CFR 0.152.
 */
package deviceinfo.mayur.medialibrary.util;

import android.util.Log;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.zip.Adler32;

public class BlobCache
implements Closeable {
    private static final String TAG = "BlobCache";
    private static final int MAGIC_INDEX_FILE = -1289277392;
    private static final int MAGIC_DATA_FILE = -1121680112;
    private static final int IH_MAGIC = 0;
    private static final int IH_MAX_ENTRIES = 4;
    private static final int IH_MAX_BYTES = 8;
    private static final int IH_ACTIVE_REGION = 12;
    private static final int IH_ACTIVE_ENTRIES = 16;
    private static final int IH_ACTIVE_BYTES = 20;
    private static final int IH_VERSION = 24;
    private static final int IH_CHECKSUM = 28;
    private static final int INDEX_HEADER_SIZE = 32;
    private static final int DATA_HEADER_SIZE = 4;
    private static final int BH_KEY = 0;
    private static final int BH_CHECKSUM = 8;
    private static final int BH_OFFSET = 12;
    private static final int BH_LENGTH = 16;
    private static final int BLOB_HEADER_SIZE = 20;
    private RandomAccessFile mIndexFile;
    private RandomAccessFile mDataFile0;
    private RandomAccessFile mDataFile1;
    private FileChannel mIndexChannel;
    private MappedByteBuffer mIndexBuffer;
    private int mMaxEntries;
    private int mMaxBytes;
    private int mActiveRegion;
    private int mActiveEntries;
    private int mActiveBytes;
    private int mVersion;
    private RandomAccessFile mActiveDataFile;
    private RandomAccessFile mInactiveDataFile;
    private int mActiveHashStart;
    private int mInactiveHashStart;
    private byte[] mIndexHeader = new byte[32];
    private byte[] mBlobHeader = new byte[20];
    private Adler32 mAdler32 = new Adler32();
    private LookupRequest mLookupRequest = new LookupRequest();
    private int mSlotOffset;
    private int mFileOffset;

    public BlobCache(String path, int maxEntries, int maxBytes, boolean reset) throws IOException {
        this(path, maxEntries, maxBytes, reset, 0);
    }

    public BlobCache(String path, int maxEntries, int maxBytes, boolean reset, int version) throws IOException {
        this.mIndexFile = new RandomAccessFile(path + ".idx", "rw");
        this.mDataFile0 = new RandomAccessFile(path + ".0", "rw");
        this.mDataFile1 = new RandomAccessFile(path + ".1", "rw");
        this.mVersion = version;
        if (!reset && this.loadIndex()) {
            return;
        }
        this.resetCache(maxEntries, maxBytes);
        if (!this.loadIndex()) {
            this.closeAll();
            throw new IOException("unable to load index");
        }
    }

    public static void deleteFiles(String path) {
        BlobCache.deleteFileSilently(path + ".idx");
        BlobCache.deleteFileSilently(path + ".0");
        BlobCache.deleteFileSilently(path + ".1");
    }

    private static void deleteFileSilently(String path) {
        try {
            new File(path).delete();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    @Override
    public void close() {
        this.syncAll();
        this.closeAll();
    }

    private void closeAll() {
        BlobCache.closeSilently(this.mIndexChannel);
        BlobCache.closeSilently(this.mIndexFile);
        BlobCache.closeSilently(this.mDataFile0);
        BlobCache.closeSilently(this.mDataFile1);
    }

    private boolean loadIndex() {
        try {
            this.mIndexFile.seek(0L);
            this.mDataFile0.seek(0L);
            this.mDataFile1.seek(0L);
            byte[] buf = this.mIndexHeader;
            if (this.mIndexFile.read(buf) != 32) {
                Log.w((String)TAG, (String)"cannot read header");
                return false;
            }
            if (BlobCache.readInt(buf, 0) != -1289277392) {
                Log.w((String)TAG, (String)"cannot read header magic");
                return false;
            }
            if (BlobCache.readInt(buf, 24) != this.mVersion) {
                Log.w((String)TAG, (String)"version mismatch");
                return false;
            }
            this.mMaxEntries = BlobCache.readInt(buf, 4);
            this.mMaxBytes = BlobCache.readInt(buf, 8);
            this.mActiveRegion = BlobCache.readInt(buf, 12);
            this.mActiveEntries = BlobCache.readInt(buf, 16);
            this.mActiveBytes = BlobCache.readInt(buf, 20);
            int sum = BlobCache.readInt(buf, 28);
            if (this.checkSum(buf, 0, 28) != sum) {
                Log.w((String)TAG, (String)"header checksum does not match");
                return false;
            }
            if (this.mMaxEntries <= 0) {
                Log.w((String)TAG, (String)"invalid max entries");
                return false;
            }
            if (this.mMaxBytes <= 0) {
                Log.w((String)TAG, (String)"invalid max bytes");
                return false;
            }
            if (this.mActiveRegion != 0 && this.mActiveRegion != 1) {
                Log.w((String)TAG, (String)"invalid active region");
                return false;
            }
            if (this.mActiveEntries < 0 || this.mActiveEntries > this.mMaxEntries) {
                Log.w((String)TAG, (String)"invalid active entries");
                return false;
            }
            if (this.mActiveBytes < 4 || this.mActiveBytes > this.mMaxBytes) {
                Log.w((String)TAG, (String)"invalid active bytes");
                return false;
            }
            if (this.mIndexFile.length() != (long)(32 + this.mMaxEntries * 12 * 2)) {
                Log.w((String)TAG, (String)"invalid index file length");
                return false;
            }
            byte[] magic = new byte[4];
            if (this.mDataFile0.read(magic) != 4) {
                Log.w((String)TAG, (String)"cannot read data file magic");
                return false;
            }
            if (BlobCache.readInt(magic, 0) != -1121680112) {
                Log.w((String)TAG, (String)"invalid data file magic");
                return false;
            }
            if (this.mDataFile1.read(magic) != 4) {
                Log.w((String)TAG, (String)"cannot read data file magic");
                return false;
            }
            if (BlobCache.readInt(magic, 0) != -1121680112) {
                Log.w((String)TAG, (String)"invalid data file magic");
                return false;
            }
            this.mIndexChannel = this.mIndexFile.getChannel();
            this.mIndexBuffer = this.mIndexChannel.map(FileChannel.MapMode.READ_WRITE, 0L, this.mIndexFile.length());
            this.mIndexBuffer.order(ByteOrder.LITTLE_ENDIAN);
            this.setActiveVariables();
            return true;
        }
        catch (IOException ex) {
            Log.e((String)TAG, (String)"loadIndex failed.", (Throwable)ex);
            return false;
        }
    }

    private void setActiveVariables() throws IOException {
        this.mActiveDataFile = this.mActiveRegion == 0 ? this.mDataFile0 : this.mDataFile1;
        this.mInactiveDataFile = this.mActiveRegion == 1 ? this.mDataFile0 : this.mDataFile1;
        this.mActiveDataFile.setLength(this.mActiveBytes);
        this.mActiveDataFile.seek(this.mActiveBytes);
        this.mActiveHashStart = 32;
        this.mInactiveHashStart = 32;
        if (this.mActiveRegion == 0) {
            this.mInactiveHashStart += this.mMaxEntries * 12;
        } else {
            this.mActiveHashStart += this.mMaxEntries * 12;
        }
    }

    private void resetCache(int maxEntries, int maxBytes) throws IOException {
        this.mIndexFile.setLength(0L);
        this.mIndexFile.setLength(32 + maxEntries * 12 * 2);
        this.mIndexFile.seek(0L);
        byte[] buf = this.mIndexHeader;
        BlobCache.writeInt(buf, 0, -1289277392);
        BlobCache.writeInt(buf, 4, maxEntries);
        BlobCache.writeInt(buf, 8, maxBytes);
        BlobCache.writeInt(buf, 12, 0);
        BlobCache.writeInt(buf, 16, 0);
        BlobCache.writeInt(buf, 20, 4);
        BlobCache.writeInt(buf, 24, this.mVersion);
        BlobCache.writeInt(buf, 28, this.checkSum(buf, 0, 28));
        this.mIndexFile.write(buf);
        this.mDataFile0.setLength(0L);
        this.mDataFile1.setLength(0L);
        this.mDataFile0.seek(0L);
        this.mDataFile1.seek(0L);
        BlobCache.writeInt(buf, 0, -1121680112);
        this.mDataFile0.write(buf, 0, 4);
        this.mDataFile1.write(buf, 0, 4);
    }

    private void flipRegion() throws IOException {
        this.mActiveRegion = 1 - this.mActiveRegion;
        this.mActiveEntries = 0;
        this.mActiveBytes = 4;
        BlobCache.writeInt(this.mIndexHeader, 12, this.mActiveRegion);
        BlobCache.writeInt(this.mIndexHeader, 16, this.mActiveEntries);
        BlobCache.writeInt(this.mIndexHeader, 20, this.mActiveBytes);
        this.updateIndexHeader();
        this.setActiveVariables();
        this.clearHash(this.mActiveHashStart);
        this.syncIndex();
    }

    private void updateIndexHeader() {
        BlobCache.writeInt(this.mIndexHeader, 28, this.checkSum(this.mIndexHeader, 0, 28));
        this.mIndexBuffer.position(0);
        this.mIndexBuffer.put(this.mIndexHeader);
    }

    private void clearHash(int hashStart) {
        int todo;
        byte[] zero = new byte[1024];
        this.mIndexBuffer.position(hashStart);
        for (int count = this.mMaxEntries * 12; count > 0; count -= todo) {
            todo = Math.min(count, 1024);
            this.mIndexBuffer.put(zero, 0, todo);
        }
    }

    public void insert(long key, byte[] data) throws IOException {
        if (24 + data.length > this.mMaxBytes) {
            throw new RuntimeException("blob is too large!");
        }
        if (this.mActiveBytes + 20 + data.length > this.mMaxBytes || this.mActiveEntries * 2 >= this.mMaxEntries) {
            this.flipRegion();
        }
        if (!this.lookupInternal(key, this.mActiveHashStart)) {
            ++this.mActiveEntries;
            BlobCache.writeInt(this.mIndexHeader, 16, this.mActiveEntries);
        }
        this.insertInternal(key, data, data.length);
        this.updateIndexHeader();
    }

    public void clearEntry(long key) throws IOException {
        if (!this.lookupInternal(key, this.mActiveHashStart)) {
            return;
        }
        byte[] header = this.mBlobHeader;
        Arrays.fill(header, (byte)0);
        this.mActiveDataFile.seek(this.mFileOffset);
        this.mActiveDataFile.write(header);
    }

    private void insertInternal(long key, byte[] data, int length) throws IOException {
        byte[] header = this.mBlobHeader;
        int sum = this.checkSum(data);
        BlobCache.writeLong(header, 0, key);
        BlobCache.writeInt(header, 8, sum);
        BlobCache.writeInt(header, 12, this.mActiveBytes);
        BlobCache.writeInt(header, 16, length);
        this.mActiveDataFile.write(header);
        this.mActiveDataFile.write(data, 0, length);
        this.mIndexBuffer.putLong(this.mSlotOffset, key);
        this.mIndexBuffer.putInt(this.mSlotOffset + 8, this.mActiveBytes);
        this.mActiveBytes += 20 + length;
        BlobCache.writeInt(this.mIndexHeader, 20, this.mActiveBytes);
    }

    public byte[] lookup(long key) throws IOException {
        this.mLookupRequest.key = key;
        this.mLookupRequest.buffer = null;
        if (this.lookup(this.mLookupRequest)) {
            return this.mLookupRequest.buffer;
        }
        return null;
    }

    public boolean lookup(LookupRequest req) throws IOException {
        if (this.lookupInternal(req.key, this.mActiveHashStart) && this.getBlob(this.mActiveDataFile, this.mFileOffset, req)) {
            return true;
        }
        int insertOffset = this.mSlotOffset;
        if (this.lookupInternal(req.key, this.mInactiveHashStart) && this.getBlob(this.mInactiveDataFile, this.mFileOffset, req)) {
            if (this.mActiveBytes + 20 + req.length > this.mMaxBytes || this.mActiveEntries * 2 >= this.mMaxEntries) {
                return true;
            }
            this.mSlotOffset = insertOffset;
            try {
                this.insertInternal(req.key, req.buffer, req.length);
                ++this.mActiveEntries;
                BlobCache.writeInt(this.mIndexHeader, 16, this.mActiveEntries);
                this.updateIndexHeader();
            }
            catch (Throwable t) {
                Log.e((String)TAG, (String)"cannot copy over");
            }
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean getBlob(RandomAccessFile file, int offset, LookupRequest req) throws IOException {
        byte[] header = this.mBlobHeader;
        long oldPosition = file.getFilePointer();
        try {
            file.seek(offset);
            if (file.read(header) != 20) {
                Log.w((String)TAG, (String)"cannot read blob header");
                boolean bl = false;
                return bl;
            }
            long blobKey = BlobCache.readLong(header, 0);
            if (blobKey == 0L) {
                boolean bl = false;
                return bl;
            }
            if (blobKey != req.key) {
                Log.w((String)TAG, (String)("blob key does not match: " + blobKey));
                boolean bl = false;
                return bl;
            }
            int sum = BlobCache.readInt(header, 8);
            int blobOffset = BlobCache.readInt(header, 12);
            if (blobOffset != offset) {
                Log.w((String)TAG, (String)("blob offset does not match: " + blobOffset));
                boolean bl = false;
                return bl;
            }
            int length = BlobCache.readInt(header, 16);
            if (length < 0 || length > this.mMaxBytes - offset - 20) {
                Log.w((String)TAG, (String)("invalid blob length: " + length));
                boolean bl = false;
                return bl;
            }
            if (req.buffer == null || req.buffer.length < length) {
                req.buffer = new byte[length];
            }
            byte[] blob = req.buffer;
            req.length = length;
            if (file.read(blob, 0, length) != length) {
                Log.w((String)TAG, (String)"cannot read blob data");
                boolean bl = false;
                return bl;
            }
            if (this.checkSum(blob, 0, length) != sum) {
                Log.w((String)TAG, (String)("blob checksum does not match: " + sum));
                boolean bl = false;
                return bl;
            }
            boolean bl = true;
            return bl;
        }
        catch (Throwable t) {
            Log.e((String)TAG, (String)"getBlob failed.", (Throwable)t);
            boolean bl = false;
            return bl;
        }
        finally {
            file.seek(oldPosition);
        }
    }

    private boolean lookupInternal(long key, int hashStart) {
        int slot = (int)(key % (long)this.mMaxEntries);
        if (slot < 0) {
            slot += this.mMaxEntries;
        }
        int slotBegin = slot;
        while (true) {
            int offset = hashStart + slot * 12;
            long candidateKey = this.mIndexBuffer.getLong(offset);
            int candidateOffset = this.mIndexBuffer.getInt(offset + 8);
            if (candidateOffset == 0) {
                this.mSlotOffset = offset;
                return false;
            }
            if (candidateKey == key) {
                this.mSlotOffset = offset;
                this.mFileOffset = candidateOffset;
                return true;
            }
            if (++slot >= this.mMaxEntries) {
                slot = 0;
            }
            if (slot != slotBegin) continue;
            Log.w((String)TAG, (String)"corrupted index: clear the slot.");
            this.mIndexBuffer.putInt(hashStart + slot * 12 + 8, 0);
        }
    }

    public void syncIndex() {
        try {
            this.mIndexBuffer.force();
        }
        catch (Throwable t) {
            Log.w((String)TAG, (String)"sync index failed", (Throwable)t);
        }
    }

    public void syncAll() {
        this.syncIndex();
        try {
            this.mDataFile0.getFD().sync();
        }
        catch (Throwable t) {
            Log.w((String)TAG, (String)"sync data file 0 failed", (Throwable)t);
        }
        try {
            this.mDataFile1.getFD().sync();
        }
        catch (Throwable t) {
            Log.w((String)TAG, (String)"sync data file 1 failed", (Throwable)t);
        }
    }

    int getActiveCount() {
        int count = 0;
        for (int i = 0; i < this.mMaxEntries; ++i) {
            int offset = this.mActiveHashStart + i * 12;
            long candidateKey = this.mIndexBuffer.getLong(offset);
            int candidateOffset = this.mIndexBuffer.getInt(offset + 8);
            if (candidateOffset == 0) continue;
            ++count;
        }
        if (count == this.mActiveEntries) {
            return count;
        }
        Log.e((String)TAG, (String)("wrong active count: " + this.mActiveEntries + " vs " + count));
        return -1;
    }

    int checkSum(byte[] data) {
        this.mAdler32.reset();
        this.mAdler32.update(data);
        return (int)this.mAdler32.getValue();
    }

    int checkSum(byte[] data, int offset, int nbytes) {
        this.mAdler32.reset();
        this.mAdler32.update(data, offset, nbytes);
        return (int)this.mAdler32.getValue();
    }

    static void closeSilently(Closeable c) {
        if (c == null) {
            return;
        }
        try {
            c.close();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    static int readInt(byte[] buf, int offset) {
        return buf[offset] & 0xFF | (buf[offset + 1] & 0xFF) << 8 | (buf[offset + 2] & 0xFF) << 16 | (buf[offset + 3] & 0xFF) << 24;
    }

    static long readLong(byte[] buf, int offset) {
        long result = buf[offset + 7] & 0xFF;
        for (int i = 6; i >= 0; --i) {
            result = result << 8 | (long)(buf[offset + i] & 0xFF);
        }
        return result;
    }

    static void writeInt(byte[] buf, int offset, int value) {
        for (int i = 0; i < 4; ++i) {
            buf[offset + i] = (byte)(value & 0xFF);
            value >>= 8;
        }
    }

    static void writeLong(byte[] buf, int offset, long value) {
        for (int i = 0; i < 8; ++i) {
            buf[offset + i] = (byte)(value & 0xFFL);
            value >>= 8;
        }
    }

    public static class LookupRequest {
        public long key;
        public byte[] buffer;
        public int length;
    }
}

