/*
 * Copyright 2019 Ivan Pekov
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.github.mrivanplays.bungee.vault.api;

import net.md_5.bungee.api.connection.ProxiedPlayer;

import java.util.Collection;

/** Represents permission things for the players */
public abstract class Permission {
  /**
   * Because BungeeCord doesn't have a service provider as well as bukkit, we should access this
   * class via a instance.
   */
  private static Permission instance;

  /**
   * Gets an instance of this class
   *
   * @return instance of this class
   */
  public static Permission getPermission() {
    return instance;
  }

  /**
   * Sets an instance of this class <b>If the instance is already set, this method will throw an
   * {@link IllegalArgumentException}. If the new instance is null, the method again will throw an
   * {@link IllegalArgumentException}</b>
   *
   * @param instance new instance
   */
  public static void setInstance(Permission instance) {
    if (instance == null) {
      throw new IllegalArgumentException("New instance cannot be null");
    }
    if (Permission.instance != null) {
      throw new IllegalArgumentException("Instance already set");
    }
    Permission.instance = instance;
  }

  /**
   * Gets the 1st group of the player <b>If no permission plugin is installed, it will get the first
   * BungeeCord one.</b>
   *
   * @param player checked player
   * @return player group
   */
  public abstract String getPrimaryGroup(ProxiedPlayer player);

  /**
   * Sets the 1st group of the player
   *
   * @param player group set for the player
   * @param group set group
   * @throws NullPointerException if the group isn't found <b>(BungeeCord Specific)</b>
   */
  public abstract void setPrimaryGroup(ProxiedPlayer player, String group);

  /**
   * Gets the player permissions
   *
   * @param player checked player
   * @return player permissions
   */
  public abstract Collection<String> getPermissions(ProxiedPlayer player);

  /**
   * Gets the provider
   *
   * @return provider
   */
  public abstract Provider getProvider();

  /**
   * Gets the player groups
   *
   * @param player groups of the player
   * @return player groups
   */
  public abstract Collection<String> getGroups(ProxiedPlayer player);
}
