/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2003 The Nimbus Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus Project.
 */
package jp.ossc.nimbus.service.aop.interceptor.servlet;

import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;


import javax.servlet.http.HttpServletRequest;

/* **Version Difference Comment Start**
import jakarta.servlet.http.HttpServletRequest;
**Version Difference Comment End** */

import jp.ossc.nimbus.beans.Property;
import jp.ossc.nimbus.beans.PropertyAccess;
import jp.ossc.nimbus.beans.dataset.RecordList;
import jp.ossc.nimbus.beans.dataset.Record;
import jp.ossc.nimbus.beans.dataset.PropertySchema;
import jp.ossc.nimbus.recset.RecordSet;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.service.aop.InterceptorChain;
import jp.ossc.nimbus.service.aop.ServletFilterInvocationContext;
import jp.ossc.nimbus.service.aop.interceptor.ThreadContextKey;
import jp.ossc.nimbus.service.context.Context;

/**
 * HttpServletRequestプロパティ設定インターセプタ。
 * <p>
 *
 * @author M.Ishida
 */
public class HttpServletRequestTransferInterceptorService extends ServletFilterInterceptorService implements
        HttpServletRequestTransferInterceptorServiceMBean {

    private static final long serialVersionUID = 8599129621419714729L;

    protected ServiceName threadContextServiceName;
    protected Context threadContext;
    protected PropertyAccess propertyAccess;
    protected String codeMasterContextKey = ThreadContextKey.CODEMASTER;
    protected List transferSettings;

    public void setThreadContextServiceName(ServiceName name) {
        threadContextServiceName = name;
    }

    public ServiceName getThreadContextServiceName() {
        return threadContextServiceName;
    }

    public void setRequestPropertyAndContextKeyMapping(Map mapping) {
        Iterator entries = mapping.entrySet().iterator();
        while(entries.hasNext()){
            Map.Entry entry = (Map.Entry)entries.next();
            TransferSetting setting = new TransferSetting();
            setting.setRequestProperty((String)entry.getKey());
            setting.setContextKey((String)entry.getValue());
            setTransferSetting(setting);
        }
    }
    
    public void setCodeMasterContextKey(String key){
        codeMasterContextKey = key;
    }
    public String getCodeMasterContextKey(){
        return codeMasterContextKey;
    }
    
    public void setTransferSetting(TransferSetting setting){
        transferSettings.add(setting);
    }
    
    public void clearTransferSettings(){
        if(transferSettings != null){
            transferSettings.clear();
        }
    }
    
    public List getTransferSettings(){
        return transferSettings == null ? null : new ArrayList(transferSettings);
    }
    
    /**
     * サービスの生成処理を行う。
     * <p>
     *
     * @exception Exception サービスの生成に失敗した場合
     */
    public void createService() throws Exception {
        transferSettings = new ArrayList();
        propertyAccess = new PropertyAccess();
        propertyAccess.setIgnoreNullProperty(true);
    }

    /**
     * サービスの開始処理を行う。
     * <p>
     *
     * @exception Exception サービスの開始に失敗した場合
     */
    public void startService() throws Exception {
        if (threadContextServiceName != null) {
            threadContext = (Context) ServiceManagerFactory.getServiceObject(threadContextServiceName);
        }
    }

    /**
     * RequestPropertyAndContextKeyMappingに設定されたマッピングに従い、
     * HttpServletRequestのプロパティ値をThreadContextに設定して、次のインターセプタを呼び出す。
     * <p>
     * サービスが開始されていない場合は、何もせずに次のインターセプタを呼び出す。<br>
     *
     * @param context 呼び出しのコンテキスト情報
     * @param chain 次のインターセプタを呼び出すためのチェーン
     * @return 呼び出し結果の戻り値
     * @exception Throwable 呼び出し先で例外が発生した場合、またはこのインターセプタで任意の例外が発生した場合。但し、
     *                本来呼び出される処理がthrowしないRuntimeException以外の例外をthrowしても
     *                、呼び出し元には伝播されない。
     */
    public Object invokeFilter(ServletFilterInvocationContext context, InterceptorChain chain) throws Throwable {
        if (getState() == STARTED && transferSettings != null && transferSettings.size() != 0) {
            final HttpServletRequest request = (HttpServletRequest) context.getServletRequest();
            Iterator itr = transferSettings.iterator();
            while(itr.hasNext()){
                TransferSetting setting = (TransferSetting)itr.next();
                Map codeMasters = (Map)threadContext.get(codeMasterContextKey);
                setting.transfer(propertyAccess, request, threadContext, codeMasters);
            }
        }
        return chain.invokeNext(context);
    }
    
    public static class TransferSetting implements java.io.Serializable{
        private static final long serialVersionUID = 7998921206986731545L;
        
        protected String requestProperty;
        protected String contextKey;
        protected String codeMasterName;
        protected String codeMasterProperty;
        
        /**
         * リクエストから取得するプロパティ名を設定する。<p>
         *
         * @param property プロパティ名
         */
        public void setRequestProperty(String property){
            requestProperty = property;
        }
        
        /**
         * コンテキストに設定するキー名を設定する。<p>
         *
         * @param key キー名
         */
        public void setContextKey(String key){
            contextKey = key;
        }
        
        /**
         * リクエストから取得した値をキーにして、コンテキストに設定する値を引きにいくコードマスタ名を設定する。<p>
         *
         * @param name コードマスタ名
         */
        public void setCodeMasterName(String name){
            codeMasterName = name;
        }
        
        /**
         * コードマスタのレコードから取得する値のプロパティ名を設定する。<p>
         *
         * @param property プロパティ名
         */
        public void setCodeMasterProperty(String property){
            codeMasterProperty = property;
        }
        
        public String toString(){
            StringBuilder buf = new StringBuilder();
            buf.append("{requestProperty=").append(requestProperty);
            buf.append(",contextKey=").append(contextKey);
            if(codeMasterName != null){
                buf.append(",codeMasterName=").append(codeMasterName);
            }
            if(codeMasterProperty != null){
                buf.append(",codeMasterProperty=").append(codeMasterProperty);
            }
            buf.append('}');
            return buf.toString();
        }
        
        protected void transfer(PropertyAccess propertyAccess, HttpServletRequest request, Context context, Map codeMasters) throws Throwable{
            Object value = null;
            try{
                value = propertyAccess.get(request, requestProperty);
            }catch (InvocationTargetException e) {
                throw e.getTargetException();
            }
            if(codeMasterName != null){
                if(codeMasters == null){
                    throw new IllegalArgumentException("CodeMaster did not bind on context.");
                }
                Object master = codeMasters.get(codeMasterName);
                if(master == null){
                    throw new IllegalArgumentException("Master did not find on CodeMaster. codeMasterName=" + codeMasterName);
                }
                if(master instanceof RecordSet){
                    RecordSet recset = (RecordSet)master;
                    value = recset.get(value == null ? null : value.toString());
                    if(value != null && codeMasterProperty != null){
                        try{
                            value = propertyAccess.get(value, codeMasterProperty);
                        }catch (InvocationTargetException e) {
                            throw e.getTargetException();
                        }
                    }
                }else if(master instanceof RecordList){
                    RecordList recordList = (RecordList)master;
                    PropertySchema[] schemata = recordList.getRecordSchema().getPrimaryKeyPropertySchemata();
                    if(schemata == null || schemata.length != 1){
                        throw new IllegalArgumentException("Size of primary key property not equal 1. = schema=" + recordList.getRecordSchema());
                    }
                    Record key = recordList.createRecord();
                    key.setProperty(schemata[0].getName(), value);
                    value = recordList.searchByPrimaryKey(key);
                    if(value != null && codeMasterProperty != null){
                        try{
                            value = propertyAccess.get(value, codeMasterProperty);
                        }catch (InvocationTargetException e) {
                            throw e.getTargetException();
                        }
                    }
                }else{
                    throw new IllegalArgumentException("Unsupported master type. type=" + master.getClass());
                }
            }
            context.put(contextKey, value);
        }
    }
}
