/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2003 The Nimbus Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus Project.
 */
package jp.ossc.nimbus.service.aop.interceptor.servlet;

import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.service.context.SharedContext;
import jp.ossc.nimbus.beans.PropertyAccess;


import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

/* **Version Difference Comment Start**
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;
**Version Difference Comment End** */

/**
 * 共有コンテキスト認証ストア。<p>
 *
 * @author M.Takata
 */
public class SharedContextAuthenticateStoreService extends ServiceBase implements AuthenticateStore, SharedContextAuthenticateStoreServiceMBean{
    
    private ServiceName sharedContextServiceName;
    private String keyPropertyOnCreate;
    private String keyPropertyOnActivate;
    private String keyPropertyOnDestroy;
    private long timeout;
    
    private SharedContext sharedContext;
    private PropertyAccess propertyAccess;
    
    public void setSharedContextServiceName(ServiceName name){
        sharedContextServiceName = name;
    }
    public ServiceName getSharedContextServiceName(){
        return sharedContextServiceName;
    }
    
    public void setKeyPropertyOnCreate(String keyProperty){
        keyPropertyOnCreate = keyProperty;
    }
    public String getKeyPropertyOnCreate(){
        return keyPropertyOnCreate;
    }
    
    public void setKeyPropertyOnActivate(String keyProperty){
        keyPropertyOnActivate = keyProperty;
    }
    public String getKeyPropertyOnActivate(){
        return keyPropertyOnActivate;
    }
    
    public void setKeyPropertyOnDestroy(String keyProperty){
        keyPropertyOnDestroy = keyProperty;
    }
    public String getKeyPropertyOnDestroy(){
        return keyPropertyOnDestroy;
    }
    
    public void setTimeout(long timeout){
        this.timeout = timeout;
    }
    public long getTimeout(){
        return timeout;
    }
    
    public void setSharedContext(SharedContext context){
        sharedContext = context;
    }
    
    public void startService() throws Exception{
        if(sharedContextServiceName != null){
            sharedContext = (SharedContext)ServiceManagerFactory.getServiceObject(sharedContextServiceName);
        }
        if(sharedContext == null){
            throw new IllegalArgumentException("SharedContext is null.");
        }
        
        if(keyPropertyOnCreate == null){
            throw new IllegalArgumentException("KeyPropertyOnCreate is null.");
        }
        
        propertyAccess = new PropertyAccess();
        propertyAccess.setIgnoreNullProperty(true);
    }
    
    public void create(HttpServletRequest request, Object authenticatedInfo) throws AuthenticateStoreException{
        try{
            Object key = propertyAccess.get(authenticatedInfo, keyPropertyOnCreate);
            sharedContext.put(key, authenticatedInfo, timeout);
        }catch(Exception e){
            throw new AuthenticateStoreException(e);
        }
    }
    
    public Object activate(HttpServletRequest request, Object authenticatedKey) throws AuthenticateStoreException{
        try{
            Object key = keyPropertyOnActivate == null ? authenticatedKey : propertyAccess.get(authenticatedKey, keyPropertyOnActivate);
            return key == null ? null : sharedContext.get(key, timeout);
        }catch(Exception e){
            throw new AuthenticateStoreException(e);
        }
    }
    
    public void deactivate(HttpSession session, Object authenticatedInfo) throws AuthenticateStoreException{
    }
    
    public void destroy(HttpServletRequest request, Object authenticatedKey) throws AuthenticateStoreException{
        try{
            String keyProperty = keyPropertyOnDestroy;
            if(keyProperty == null){
                keyProperty = keyPropertyOnCreate;
            }
            Object key = keyProperty == null ? authenticatedKey : propertyAccess.get(authenticatedKey, keyProperty);
            sharedContext.remove(key, timeout);
        }catch(Exception e){
            throw new AuthenticateStoreException(e);
        }
    }
}
